<?php
declare(strict_types=1);

final class GoogleOAuth
{
    public static function start(): string
    {
        $cfg = $GLOBALS['APP_CONFIG']['google_oauth'] ?? [];
        if (empty($cfg['client_id']) || empty($cfg['redirect_uri'])) {
            throw new RuntimeException('Google OAuth no configurado.');
        }

        $state = bin2hex(random_bytes(16));
        $verifier = self::base64url(random_bytes(32));
        $challenge = self::base64url(hash('sha256', $verifier, true));

        $_SESSION['__g_state'] = $state;
        $_SESSION['__g_verifier'] = $verifier;

        $params = http_build_query([
            'client_id' => $cfg['client_id'],
            'redirect_uri' => $cfg['redirect_uri'],
            'response_type' => 'code',
            'scope' => 'openid email profile',
            'state' => $state,
            'code_challenge' => $challenge,
            'code_challenge_method' => 'S256',
            'access_type' => 'online',
            'prompt' => 'select_account',
        ]);

        return "https://accounts.google.com/o/oauth2/v2/auth?$params";
    }

    public static function callback(string $code, string $state): array
    {
        if (($state ?? '') === '' || !hash_equals((string)($_SESSION['__g_state'] ?? ''), $state)) {
            throw new RuntimeException('Estado OAuth inválido.');
        }

        $verifier = (string)($_SESSION['__g_verifier'] ?? '');
        if ($verifier === '') throw new RuntimeException('PKCE verifier faltante.');

        $cfg = $GLOBALS['APP_CONFIG']['google_oauth'] ?? [];

        $token = self::postForm('https://oauth2.googleapis.com/token', [
            'client_id' => $cfg['client_id'],
            'client_secret' => $cfg['client_secret'],
            'code' => $code,
            'grant_type' => 'authorization_code',
            'redirect_uri' => $cfg['redirect_uri'],
            'code_verifier' => $verifier,
        ]);

        $accessToken = $token['access_token'] ?? '';
        if ($accessToken === '') throw new RuntimeException('Access token inválido.');

        $info = self::getJson('https://openidconnect.googleapis.com/v1/userinfo', $accessToken);

        return $info;
    }

    private static function postForm(string $url, array $fields): array
    {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($fields),
            CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
            CURLOPT_TIMEOUT => 20,
        ]);
        $raw = curl_exec($ch);
        $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $json = $raw ? json_decode($raw, true) : null;
        if ($code < 200 || $code >= 300 || !is_array($json)) {
            throw new RuntimeException('Error token OAuth.');
        }
        return $json;
    }

    private static function getJson(string $url, string $bearer): array
    {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => ['Authorization: Bearer ' . $bearer],
            CURLOPT_TIMEOUT => 20,
        ]);
        $raw = curl_exec($ch);
        $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $json = $raw ? json_decode($raw, true) : null;
        if ($code < 200 || $code >= 300 || !is_array($json)) {
            throw new RuntimeException('Error userinfo OAuth.');
        }
        return $json;
    }

    private static function base64url(string $bin): string
    {
        return rtrim(strtr(base64_encode($bin), '+/', '-_'), '=');
    }
}
