<?php
declare(strict_types=1);

final class ModuleService
{
    /**
     * Regla:
     * 1) modules.estado inactivo => false
     * 2) override manual (user_modules) => manda
     * 3) override auto (user_modules) => manda
     * 4) por plan (users.plan_id -> plan_modules.enabled) => manda
     */
    public static function isEnabled(int $userId, string $moduleCode): bool
    {
        $pdo = DB::pdo();

        // IMPORTANTE:
        // Con PDO::ATTR_EMULATE_PREPARES=false, MySQL no permite reutilizar el mismo :uid varias veces.
        // Por eso usamos :uid1, :uid2, :uid3.
        $sql = "
            SELECT
              m.estado AS module_estado,
              um_m.enabled AS um_manual,
              um_a.enabled AS um_auto,
              pm.enabled AS plan_enabled
            FROM modules m
            LEFT JOIN user_modules um_m
              ON um_m.module_id = m.id
             AND um_m.user_id = :uid1
             AND um_m.override_mode = 'manual'
            LEFT JOIN user_modules um_a
              ON um_a.module_id = m.id
             AND um_a.user_id = :uid2
             AND um_a.override_mode = 'auto'
            LEFT JOIN users u
              ON u.id = :uid3
            LEFT JOIN plan_modules pm
              ON pm.plan_id = u.plan_id
             AND pm.module_id = m.id
            WHERE m.code = :code
            LIMIT 1
        ";

        $st = $pdo->prepare($sql);
        $st->execute([
            ':uid1' => $userId,
            ':uid2' => $userId,
            ':uid3' => $userId,
            ':code' => $moduleCode,
        ]);

        $row = $st->fetch();

        if (!$row) return false;
        if (($row['module_estado'] ?? '') !== 'activo') return false;

        // Overrides mandan si existen (aunque el plan diga otra cosa)
        if ($row['um_manual'] !== null) return (int)$row['um_manual'] === 1;
        if ($row['um_auto'] !== null)   return (int)$row['um_auto'] === 1;

        // plan_enabled puede ser null si el plan no tiene ese módulo
        return (int)($row['plan_enabled'] ?? 0) === 1;
    }
}
