<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/Config/config.php';
require_once dirname(__DIR__) . '/Support/Logger.php';
require_once dirname(__DIR__) . '/Support/Helpers.php';
require_once dirname(__DIR__) . '/Support/Exceptions.php';

require_once dirname(__DIR__) . '/Http/Request.php';
require_once dirname(__DIR__) . '/Http/Response.php';
require_once dirname(__DIR__) . '/Http/Router.php';

require_once dirname(__DIR__) . '/Security/CsrfToken.php';
require_once dirname(__DIR__) . '/Security/RateLimiter.php';
require_once dirname(__DIR__) . '/Security/Captcha.php';

require_once dirname(__DIR__) . '/Http/Middlewares/SecurityHeaders.php';
require_once dirname(__DIR__) . '/Http/Middlewares/Csrf.php';
require_once dirname(__DIR__) . '/Http/Middlewares/RateLimit.php';
require_once dirname(__DIR__) . '/Http/Middlewares/RequireAuth.php';
require_once dirname(__DIR__) . '/Http/Middlewares/RequireModule.php';
require_once dirname(__DIR__) . '/Http/Middlewares/RequirePermission.php';

require_once dirname(__DIR__) . '/DB/DB.php';

require_once dirname(__DIR__) . '/Admin/AdminModuleService.php';

require_once dirname(__DIR__) . '/Auth/Passwords.php';
require_once dirname(__DIR__) . '/Auth/AuthService.php';
require_once dirname(__DIR__) . '/Auth/ModuleService.php';

final class App
{
    private Router $router;
    private Logger $logger;

    public function __construct(private array $config)
    {
        $this->logger = new Logger($config['paths']['logs_dir']);
        $this->installErrorHandling();

        $this->router = new Router();

        // Middlewares base (orden IMPORTA)
        $this->router->addMiddleware(new SecurityHeadersMiddleware($config));
        $this->router->addMiddleware(new RateLimitMiddleware($config, $this->logger));
        $this->router->addMiddleware(new CsrfMiddleware($config, $this->logger));
        $this->router->addMiddleware(new RequireAuthMiddleware($config, $this->logger));

        $this->registerRoutes();
    }

    public function run(): void
    {
        $req = Request::fromGlobals();
        $res = $this->router->dispatch($req);
        $res->send();
    }

    /* ===============================
     * Error Handling (HTML/JSON)
     * =============================== */
    private function installErrorHandling(): void
    {
        $logger = $this->logger;

        set_exception_handler(function (Throwable $e) use ($logger): void {
            $id = bin2hex(random_bytes(8));

            $logger->error('Unhandled exception', [
                'id'      => $id,
                'type'    => get_class($e),
                'message' => $e->getMessage(),
                'file'    => $e->getFile(),
                'line'    => $e->getLine(),
            ]);

            $isDebug = (bool)($GLOBALS['APP_CONFIG']['app']['debug'] ?? false);

            $accept = strtolower((string)($_SERVER['HTTP_ACCEPT'] ?? ''));
            $isAjax = (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest');
            $format = strtolower((string)($_GET['format'] ?? ''));

            $wantsJson =
                $isAjax ||
                str_contains($accept, 'application/json') ||
                str_contains($accept, 'text/json') ||
                ($format === 'json');

            if (headers_sent()) {
                if ($wantsJson) {
                    echo json_encode([
                        'ok'       => false,
                        'error'    => $isDebug ? $e->getMessage() : 'Error interno',
                        'error_id' => $id,
                    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                } else {
                    echo "<pre>ERROR {$id}: " . htmlspecialchars($isDebug ? $e->getMessage() : 'Error interno') . "</pre>";
                }
                return;
            }

            if ($wantsJson) {
                Response::json([
                    'ok'       => false,
                    'error'    => $isDebug ? $e->getMessage() : 'Error interno',
                    'error_id' => $id,
                ], 500)->send();
                return;
            }

            $msg = $isDebug
                ? ($e->getMessage() . " @ " . $e->getFile() . ":" . $e->getLine())
                : 'Error interno';

            $html = "<!doctype html><html><head><meta charset='utf-8'>"
                . "<meta name='viewport' content='width=device-width,initial-scale=1'>"
                . "<title>Error</title>"
                . "<style>
                    body{font-family:system-ui;padding:24px;background:#f6f8ff}
                    .card{background:#fff;border:1px solid #e8eefc;border-radius:14px;padding:18px;max-width:860px}
                    h2{margin:0 0 10px}
                    p{margin:0 0 8px}
                  </style>"
                . "</head><body>"
                . "<div class='card'>"
                . "<h2>Ocurrió un error</h2>"
                . "<p>" . htmlspecialchars($msg) . "</p>"
                . "<p><small>ID: " . htmlspecialchars($id) . "</small></p>"
                . "</div>"
                . "</body></html>";

            Response::html($html, 500)->send();
        });

        set_error_handler(function (int $severity, string $message, string $file, int $line) use ($logger): bool {
            $logger->error('PHP error', [
                'severity' => $severity,
                'message'  => $message,
                'file'     => $file,
                'line'     => $line,
            ]);
            return false;
        });
    }

    /* ===============================
     * Rutas con alias /public
     * =============================== */
    private function mapGet(string $path, callable $handler): void
    {
        $this->router->get($path, $handler);

        if ($path === '/') {
            $this->router->get('/public', $handler);
            return;
        }

        if (!str_starts_with($path, '/public/')) {
            $this->router->get('/public' . $path, $handler);
        }
    }

    private function mapPost(string $path, callable $handler): void
    {
        $this->router->post($path, $handler);

        if (!str_starts_with($path, '/public/')) {
            $this->router->post('/public' . $path, $handler);
        }
    }

    /* ===============================
     * Redirect robusto
     * =============================== */
    private function redirect(string $to, int $status = 302): Response
    {
        $to = $to === '' ? (base_path() . '/') : $to;

        $html = '<!doctype html><html><head><meta charset="utf-8">'
            . '<meta http-equiv="refresh" content="0;url=' . htmlspecialchars($to) . '">'
            . '</head><body>Redirigiendo...</body></html>';

        return Response::html($html, $status)->header('Location', $to);
    }

    /* ===============================
     * Gate helper (Admin: módulo + permiso)
     * =============================== */
    private function gateAdmin(Request $req, string $permCode, callable $handler): Response
    {
        $mwModule = new RequireModuleMiddleware($this->config, $this->logger, 'admin_panel');
        $mwPerm   = new RequirePermissionMiddleware($this->config, $this->logger, $permCode);

        return $mwModule->handle($req, function (Request $req) use ($mwPerm, $handler): Response {
            return $mwPerm->handle($req, function (Request $req) use ($handler): Response {
                return $handler($req);
            });
        });
    }

    /* ===============================
     * Routes
     * =============================== */
    private function registerRoutes(): void
    {
        /* ========= HEALTH ========= */
        $this->mapGet('/health', function (Request $req): Response {
            return Response::json(['ok' => true, 'status' => 'UP'], 200);
        });

        $this->mapGet('/db-health', function (Request $req): Response {
            try {
                DB::pdo()->query('SELECT 1');
                return Response::json(['ok' => true, 'db' => 'UP'], 200);
            } catch (Throwable $e) {
                $this->logger->error('DB health failed', ['message' => $e->getMessage()]);
                return Response::json(['ok' => false, 'db' => 'DOWN'], 500);
            }
        });

        /* ========= LANDING ========= */
        $this->mapGet('/', function (Request $req): Response {
            $cfg = $GLOBALS['APP_CONFIG'];

            $html = render_template('layouts/front.php', [
                'pageTitle' => $cfg['seo']['site_name'] ?? 'Plataforma Laboral',
                'content'   => render_template('front/index.php', []),
            ]);

            return Response::html($html, 200);
        });

        /* ========= CSRF ========= */
        $this->mapGet('/csrf', function (Request $req): Response {
            return Response::json(['ok' => true, 'csrf' => CsrfToken::issue()], 200);
        });

        /* ========= CAPTCHA ========= */
        $this->mapGet('/captcha', function (Request $req): Response {
            $code = Captcha::issue(6);
            $png  = Captcha::renderPng($code);

            $res = new Response($png, 200, 'image/png');
            $res->header('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0');
            $res->header('Pragma', 'no-cache');
            $res->header('X-Content-Type-Options', 'nosniff');
            return $res;
        });

        /* ========= LOGIN ========= */
        $this->mapGet('/login', function (Request $req): Response {
            $html = render_template('layouts/front.php', [
                'pageTitle' => 'Login',
                'content'   => render_template('auth/login.php', ['error' => null]),
            ]);
            return Response::html($html, 200);
        });

        $this->mapPost('/login', function (Request $req): Response {
            $email = trim((string)($req->post['email'] ?? ''));
            $pw    = (string)($req->post['password'] ?? '');
            $cap   = (string)($req->post['captcha'] ?? '');

            if (!Captcha::verify($cap)) {
                $this->logger->security('Captcha inválido en login', ['ip' => $req->ip]);

                $html = render_template('layouts/front.php', [
                    'pageTitle' => 'Login',
                    'content'   => render_template('auth/login.php', ['error' => 'Captcha inválido.']),
                ]);
                return Response::html($html, 403);
            }

            $out = AuthService::login($email, $pw);

            if (!$out['ok']) {
                $this->logger->security('Login fallido', ['ip' => $req->ip, 'email' => $email]);

                $html = render_template('layouts/front.php', [
                    'pageTitle' => 'Login',
                    'content'   => render_template('auth/login.php', [
                        'error' => $out['error'] ?? 'Credenciales inválidas.',
                    ]),
                ]);
                return Response::html($html, 401);
            }

            $r = trim((string)($req->query['r'] ?? ''));
            $dest = $r !== '' ? $r : (base_path() . '/feed');
            return $this->redirect($dest, 302);
        });

        /* ========= REGISTRO ========= */
        $this->mapGet('/registro', function (Request $req): Response {
            $html = render_template('layouts/front.php', [
                'pageTitle' => 'Registro',
                'content'   => render_template('auth/registro.php', [
                    'error' => null,
                    'ok'    => null,
                ]),
            ]);
            return Response::html($html, 200);
        });

        $this->mapPost('/registro', function (Request $req): Response {
            $email = trim((string)($req->post['email'] ?? ''));
            $pw    = (string)($req->post['password'] ?? '');
            $tipo  = (string)($req->post['tipo'] ?? 'postulante');
            $cap   = (string)($req->post['captcha'] ?? '');

            if (!Captcha::verify($cap)) {
                $this->logger->security('Captcha inválido en registro', ['ip' => $req->ip]);

                $html = render_template('layouts/front.php', [
                    'pageTitle' => 'Registro',
                    'content'   => render_template('auth/registro.php', [
                        'error' => 'Captcha inválido.',
                        'ok'    => null,
                    ]),
                ]);
                return Response::html($html, 403);
            }

            try {
                AuthService::register($email, $pw, $tipo);
                return $this->redirect(base_path() . '/login', 302);

            } catch (ValidationException $e) {
                $this->logger->security('Registro inválido', ['ip' => $req->ip, 'email' => $email]);

                $html = render_template('layouts/front.php', [
                    'pageTitle' => 'Registro',
                    'content'   => render_template('auth/registro.php', [
                        'error' => $e->getMessage(),
                        'ok'    => null,
                    ]),
                ]);
                return Response::html($html, 422);
            }
        });

        /* ========= FEED (PRIVADO) ========= */
        $this->mapGet('/feed', function (Request $req): Response {
            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Feed',
                'content'   => render_template('front/feed.php', [
                    'userId' => AuthService::currentUserId(),
                ]),
            ]);
            return Response::html($html, 200);
        });

        /* ========= ADMIN: DASHBOARD ========= */
        $this->mapGet('/admin', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.access', function (Request $req): Response {
                $html = render_template('layouts/admin.php', [
                    'pageTitle' => 'Dashboard',
                    'content'   => render_template('admin/dashboard.php', []),
                ]);
                return Response::html($html, 200);
            });
        });

        /* ========= ADMIN: MÓDULOS ========= */
        $this->mapGet('/admin/modules', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $items = AdminModuleService::listAll();

                $html = render_template('layouts/admin.php', [
                    'pageTitle' => 'Módulos',
                    'content'   => render_template('admin/modules/index.php', [
                        'items' => $items,
                        'error' => null,
                    ]),
                ]);
                return Response::html($html, 200);
            });
        });

        $this->mapGet('/admin/modules/create', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $html = render_template('layouts/admin.php', [
                    'pageTitle' => 'Crear módulo',
                    'content'   => render_template('admin/modules/create.php', [
                        'error' => null,
                        'old'   => ['code' => '', 'nombre' => '', 'estado' => 'activo'],
                    ]),
                ]);
                return Response::html($html, 200);
            });
        });

        $this->mapPost('/admin/modules/create', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $old = [
                    'code'   => trim((string)($req->post['code'] ?? '')),
                    'nombre' => trim((string)($req->post['nombre'] ?? '')),
                    'estado' => (string)($req->post['estado'] ?? 'activo'),
                ];

                try {
                    AdminModuleService::create($old['code'], $old['nombre'], $old['estado']);
                    return $this->redirect(base_path() . '/admin/modules', 302);

                } catch (ValidationException $e) {
                    $html = render_template('layouts/admin.php', [
                        'pageTitle' => 'Crear módulo',
                        'content'   => render_template('admin/modules/create.php', [
                            'error' => $e->getMessage(),
                            'old'   => $old,
                        ]),
                    ]);
                    return Response::html($html, 422);
                }
            });
        });

        $this->mapGet('/admin/modules/edit', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $id = (int)($req->query['id'] ?? 0);
                $row = AdminModuleService::find($id);

                if (!$row) {
                    return Response::html("<h3>No existe</h3>", 404);
                }

                $html = render_template('layouts/admin.php', [
                    'pageTitle' => 'Editar módulo',
                    'content'   => render_template('admin/modules/edit.php', [
                        'error' => null,
                        'row'   => $row,
                    ]),
                ]);
                return Response::html($html, 200);
            });
        });

        $this->mapPost('/admin/modules/edit', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $id = (int)($req->post['id'] ?? 0);
                $row = AdminModuleService::find($id);
                if (!$row) return Response::html("<h3>No existe</h3>", 404);

                $new = [
                    'id'     => $id,
                    'code'   => trim((string)($req->post['code'] ?? '')),
                    'nombre' => trim((string)($req->post['nombre'] ?? '')),
                    'estado' => (string)($req->post['estado'] ?? 'activo'),
                ];

                try {
                    AdminModuleService::update($new['id'], $new['code'], $new['nombre'], $new['estado']);
                    return $this->redirect(base_path() . '/admin/modules', 302);

                } catch (ValidationException $e) {
                    $row['code']   = $new['code'];
                    $row['nombre'] = $new['nombre'];
                    $row['estado'] = $new['estado'];

                    $html = render_template('layouts/admin.php', [
                        'pageTitle' => 'Editar módulo',
                        'content'   => render_template('admin/modules/edit.php', [
                            'error' => $e->getMessage(),
                            'row'   => $row,
                        ]),
                    ]);
                    return Response::html($html, 422);
                }
            });
        });

        $this->mapPost('/admin/modules/delete', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $id = (int)($req->post['id'] ?? 0);

                try {
                    AdminModuleService::delete($id);
                    return $this->redirect(base_path() . '/admin/modules', 302);

                } catch (ValidationException $e) {
                    $items = AdminModuleService::listAll();

                    $html = render_template('layouts/admin.php', [
                        'pageTitle' => 'Módulos',
                        'content'   => render_template('admin/modules/index.php', [
                            'items' => $items,
                            'error' => $e->getMessage(),
                        ]),
                    ]);
                    return Response::html($html, 422);
                }
            });
        });

        /* ========= LOGOUT ========= */
        $this->mapGet('/logout', function (Request $req): Response {
            AuthService::logout();
            return $this->redirect(base_path() . '/login', 302);
        });

        /* ========= 404 ========= */
        $this->router->setNotFoundHandler(function (Request $req): Response {
            $accept = strtolower((string)($_SERVER['HTTP_ACCEPT'] ?? ''));
            $wantsHtml = $accept === '' || str_contains($accept, 'text/html') || str_contains($accept, '*/*');

            if ($wantsHtml) {
                $html = "<!doctype html><html><head><meta charset='utf-8'>"
                    . "<meta name='viewport' content='width=device-width,initial-scale=1'>"
                    . "<title>404</title></head><body>"
                    . "<h2>404</h2><p>Ruta no encontrada.</p></body></html>";
                return Response::html($html, 404);
            }

            return Response::json(['ok' => false, 'error' => 'Ruta no encontrada'], 404);
        });
    }
}

/* ===============================
 * APP BOOTSTRAP
 * =============================== */
function app(): App
{
    $config = config();
    $GLOBALS['APP_CONFIG'] = $config;

    if (session_status() !== PHP_SESSION_ACTIVE) {
        session_set_cookie_params([
            'lifetime' => 0,
            'path'     => '/',
            'secure'   => (bool)($config['security']['cookies_secure'] ?? true),
            'httponly' => true,
            'samesite' => 'Strict',
        ]);
        session_name($config['security']['session_name'] ?? 'APPSESSID');
        session_start();
    }

    return new App($config);
}
