<?php
declare(strict_types=1);

final class CsrfMiddleware
{
    public function __construct(private array $config, private Logger $logger) {}

    public function handle(Request $req, callable $next): Response
    {
        $enabled = (bool)($this->config['security']['csrf']['enabled'] ?? true);
        if (!$enabled) {
            return $next($req);
        }

        // CSRF SOLO para métodos con efecto
        $method = strtoupper($req->method);
        if (in_array($method, ['GET', 'HEAD', 'OPTIONS'], true)) {
            return $next($req);
        }

        // Rutas opcionalmente exentas (por si alguna integración lo requiere)
        $path = $req->path;
        $exempt = [
            '/health','/public/health',
            '/captcha','/public/captcha',
        ];
        if (in_array($path, $exempt, true)) {
            return $next($req);
        }

        // Tomar token desde header o POST
        $headerName = (string)($this->config['security']['csrf']['header'] ?? 'X-CSRF-Token');
        $token = '';

        // Header
        $serverKey = 'HTTP_' . strtoupper(str_replace('-', '_', $headerName));
        if (!empty($_SERVER[$serverKey])) {
            $token = (string)$_SERVER[$serverKey];
        }

        // POST fallback
        if ($token === '') {
            $token = (string)($req->post['_csrf'] ?? $req->post['csrf'] ?? '');
        }

        if (!CsrfToken::verify($token)) {
            $this->logger->security('CSRF inválido', [
                'ip'    => $req->ip,
                'path'  => $path,
                'm'     => $method,
            ]);

            // Detectar si espera JSON
            $accept = strtolower((string)($_SERVER['HTTP_ACCEPT'] ?? ''));
            $isAjax = (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest');
            $wantsJson = $isAjax || str_contains($accept, 'application/json') || str_contains($accept, 'text/json');

            if ($wantsJson) {
                return Response::json(['ok' => false, 'error' => 'CSRF inválido'], 403);
            }

            $html = "<!doctype html><html><head><meta charset='utf-8'>"
                . "<meta name='viewport' content='width=device-width,initial-scale=1'>"
                . "<title>CSRF inválido</title></head><body>"
                . "<h3>CSRF inválido</h3>"
                . "<p>Recarga la página e intenta nuevamente.</p>"
                . "</body></html>";

            return Response::html($html, 403);
        }

        return $next($req);
    }
}
