<?php
declare(strict_types=1);

final class RequireAuthMiddleware
{
    public function __construct(private array $config, private Logger $logger) {}

    public function handle(Request $req, callable $next): Response
    {
        $path = $req->path;

        // Rutas públicas (permitidas sin sesión)
        $public = [
            '/', '/public',

            '/health', '/public/health',
            '/db-health', '/public/db-health',

            '/csrf', '/public/csrf',
            '/captcha', '/public/captcha',

            '/login', '/public/login',
            '/registro', '/public/registro',
        ];

        // Si es pública, continúa
        if (in_array($path, $public, true)) {
            return $next($req);
        }

        // Todo lo demás: requiere sesión
        if (!AuthService::check()) {
            $this->logger->security('Acceso no autorizado', [
                'ip'   => $req->ip,
                'path' => $path,
            ]);

            // conservar querystring para retorno exacto
            $qs = $_SERVER['QUERY_STRING'] ?? '';
            $returnTo = $path . ($qs !== '' ? ('?' . $qs) : '');

            $to = base_path() . '/login?r=' . urlencode($returnTo);

            // Si el cliente espera JSON (fetch/XHR), devolver JSON
            $accept = strtolower((string)($_SERVER['HTTP_ACCEPT'] ?? ''));
            $isAjax = (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest');
            $wantsJson = $isAjax || str_contains($accept, 'application/json') || str_contains($accept, 'text/json');

            if ($wantsJson) {
                return Response::json([
                    'ok'      => false,
                    'error'   => 'No autenticado',
                    'redirect'=> $to,
                ], 401);
            }

            // Redirección real con Location + fallback HTML
            return Response::html(
                '<!doctype html><html><head><meta charset="utf-8">'
                . '<meta http-equiv="refresh" content="0;url=' . htmlspecialchars($to) . '">'
                . '</head><body>Redirigiendo...</body></html>',
                302
            )->header('Location', $to);
        }

        return $next($req);
    }
}
