<?php
declare(strict_types=1);

final class Response
{
    private array $headers = [];

    public function __construct(
        private string $body = '',
        private int $status = 200,
        private string $contentType = 'text/html'
    ) {}

    public static function json(array $data, int $status = 200): self
    {
        return new self(
            json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) ?: '{}',
            $status,
            'application/json'
        );
    }

    public static function html(string $html, int $status = 200): self
    {
        return new self($html, $status, 'text/html; charset=utf-8');
    }

    /** ✔ para binarios: imágenes, pdf, etc */
    public static function binary(string $data, string $mime, int $status = 200): self
    {
        return new self($data, $status, $mime);
    }

    public function header(string $name, string $value): self
    {
        $this->headers[$name] = $value;
        return $this;
    }

    public function send(): void
    {
        if (headers_sent()) {
            return;
        }

        http_response_code($this->status);

        // Content-Type
        header('Content-Type: ' . $this->contentType);

        // Content-Length SOLO para binarios
        if (!str_contains($this->contentType, 'text/') && !str_contains($this->contentType, 'json')) {
            header('Content-Length: ' . strlen($this->body));
        }

        foreach ($this->headers as $k => $v) {
            header($k . ': ' . $v);
        }

        echo $this->body;
    }
}
