<?php
declare(strict_types=1);

final class Captcha
{
    private const SKEY = '__captcha';

    public static function issue(int $len = 6): string
    {
        $chars = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
        $code = '';
        for ($i=0; $i<$len; $i++) {
            $code .= $chars[random_int(0, strlen($chars)-1)];
        }

        $_SESSION[self::SKEY] = [
            'hash' => hash('sha256', $code),
            'ts' => time(),
            'used' => false,
        ];

        return $code;
    }

    public static function verify(string $input, int $ttlSeconds = 300): bool
    {
        $row = $_SESSION[self::SKEY] ?? null;
        if (!$row || empty($row['hash']) || empty($row['ts'])) return false;
        if (!empty($row['used'])) return false;

        if ((time() - (int)$row['ts']) > $ttlSeconds) {
            unset($_SESSION[self::SKEY]);
            return false;
        }

        $input = strtoupper(trim($input));
        $ok = hash_equals((string)$row['hash'], hash('sha256', $input));
        $_SESSION[self::SKEY]['used'] = true; // 1 uso
        return $ok;
    }

    public static function renderPng(string $code): string
    {
        if (!extension_loaded('gd') || !function_exists('imagecreatetruecolor')) {
            throw new RuntimeException('GD no disponible en este handler PHP.');
        }

        $w = 220; $h = 70;
        $im = imagecreatetruecolor($w, $h);
        if (!$im) throw new RuntimeException('No se pudo crear imagen GD.');

        // Colores
        $bg  = imagecolorallocate($im, 12, 20, 48);     // #0c1430
        $fg  = imagecolorallocate($im, 233, 239, 255);  // texto claro
        $ln  = imagecolorallocatealpha($im, 233, 239, 255, 85); // lneas con alpha
        $dot = imagecolorallocatealpha($im, 255, 255, 255, 95);

        imagefilledrectangle($im, 0, 0, $w, $h, $bg);

        // Lneas ruido
        for ($i=0; $i<10; $i++) {
            imageline($im, random_int(0,$w), random_int(0,$h), random_int(0,$w), random_int(0,$h), $ln);
        }

        // Puntos ruido
        for ($i=0; $i<120; $i++) {
            imagesetpixel($im, random_int(0,$w-1), random_int(0,$h-1), $dot);
        }

        // Texto (sin TTF: usamos built-in)
        // Escalamos con imagestring para compatibilidad mxima
        $x = 18;
        $yBase = 26;
        $font = 5; // built-in GD font 1..5

        for ($i=0; $i<strlen($code); $i++) {
            $y = $yBase + random_int(-5, 5);
            imagestring($im, $font, $x, $y, $code[$i], $fg);
            $x += 28;
        }

        // Output
        ob_start();
        imagepng($im, null, 6); // compresin 0-9
        $png = (string)ob_get_clean();
        imagedestroy($im);

        return $png;
    }
}
