<?php
declare(strict_types=1);

/**
 * Renderiza un template dentro de /templates.
 * - Protege contra path traversal.
 * - Extrae variables de $data de forma segura.
 * - En debug muestra detalle del error; en producción, mensaje genérico.
 */
function render_template(string $templateRelativePath, array $data = []): string
{
    $cfg = $GLOBALS['APP_CONFIG'] ?? [];
    $tplDir = $cfg['paths']['templates_dir'] ?? null;

    if (!$tplDir || !is_dir($tplDir)) {
        return '<h1>Error interno</h1>';
    }

    // Normalizar path
    $rel = str_replace('\\', '/', $templateRelativePath);
    $rel = ltrim($rel, '/');

    // Bloquear traversal
    if (str_contains($rel, '..')) {
        return '<h1>Acceso inválido</h1>';
    }

    $base = rtrim((string)$tplDir, '/');
    $full = $base . '/' . $rel;

    // Validar que el archivo exista y que realmente esté dentro de templates_dir
    $realBase = realpath($base);
    $realFull = realpath($full);

    if ($realBase === false || $realFull === false || !str_starts_with($realFull, $realBase)) {
        $isDebug = (bool)($cfg['app']['debug'] ?? false);
        return $isDebug
            ? '<h1>Template inválido</h1><pre>' . e($rel) . '</pre>'
            : '<h1>Error interno</h1>';
    }

    if (!is_file($realFull)) {
        $isDebug = (bool)($cfg['app']['debug'] ?? false);
        return $isDebug
            ? '<h1>Template no encontrado</h1><pre>' . e($rel) . '</pre>'
            : '<h1>Template no encontrado</h1>';
    }

    // Exportar variables para el template
    extract($data, EXTR_SKIP);

    ob_start();
    require $realFull;
    return (string) ob_get_clean();
}

/**
 * Base path de la app según el front controller.
 * - Si estás montado en /public => devuelve "/public"
 * - Si cambias el docroot a /public => devuelve ""
 */
function base_path(): string
{
    $script = (string)($_SERVER['SCRIPT_NAME'] ?? '');
    $script = str_replace('\\', '/', $script);

    // /public/index.php => /public
    $dir = rtrim(str_replace('\\', '/', dirname($script)), '/');

    if ($dir === '' || $dir === '.' || $dir === '/') {
        return '';
    }

    return $dir;
}

/** Escape HTML seguro */
function e(string $s): string
{
    return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

/**
 * Construye una URL relativa dentro de la app.
 * Ej: url('/login') => '/public/login' (en tu montaje actual)
 */
function url(string $path = '/'): string
{
    $path = '/' . ltrim($path, '/');
    return base_path() . $path;
}
