<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/DB/DB.php';

final class AdminModerationVacanteService
{
    public static function listAll(string $q = '', string $estado = ''): array
    {
        $pdo = DB::pdo();

        $sql = "
            SELECT
              v.*,
              u.email  AS empresa_email,
              u.nombre AS empresa_nombre,
              (SELECT COUNT(*) FROM postulaciones p WHERE p.vacante_id = v.id) AS postulaciones_count
            FROM vacantes v
            LEFT JOIN users u ON u.id = v.empresa_user_id
            WHERE 1=1
        ";
        $params = [];

        if ($estado !== '') {
            $sql .= " AND v.estado = :estado";
            $params['estado'] = $estado;
        }

        if ($q !== '') {
            $sql .= " AND (
                v.titulo LIKE :q
                OR v.descripcion LIKE :q
                OR u.email LIKE :q
                OR u.nombre LIKE :q
            )";
            $params['q'] = '%' . $q . '%';
        }

        $sql .= " ORDER BY v.id DESC";

        $st = $pdo->prepare($sql);
        $st->execute($params);

        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    public static function find(int $vacanteId): ?array
    {
        if ($vacanteId <= 0) return null;

        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT
              v.*,
              u.email  AS empresa_email,
              u.nombre AS empresa_nombre,
              (SELECT COUNT(*) FROM postulaciones p WHERE p.vacante_id = v.id) AS postulaciones_count
            FROM vacantes v
            LEFT JOIN users u ON u.id = v.empresa_user_id
            WHERE v.id = :id
            LIMIT 1
        ");
        $st->execute(['id' => $vacanteId]);
        $row = $st->fetch(\PDO::FETCH_ASSOC);

        return $row ?: null;
    }

    public static function listPostulaciones(int $vacanteId): array
    {
        if ($vacanteId <= 0) return [];

        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT
              p.id, p.vacante_id, p.postulante_user_id, p.mensaje, p.estado, p.creado_en,
              u.email AS postulante_email,
              u.nombre AS postulante_nombre
            FROM postulaciones p
            LEFT JOIN users u ON u.id = p.postulante_user_id
            WHERE p.vacante_id = :vid
            ORDER BY p.id DESC
        ");
        $st->execute(['vid' => $vacanteId]);

        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    public static function setEstado(int $vacanteId, string $estado): void
    {
        if ($vacanteId <= 0) return;

        $allowed = ['borrador','publicada','cerrada'];
        if (!in_array($estado, $allowed, true)) {
            throw new ValidationException('Estado inválido.');
        }

        $pdo = DB::pdo();

        // Regla de timestamps (sin inventar columnas: se usan las existentes)
        if ($estado === 'publicada') {
            $st = $pdo->prepare("
                UPDATE vacantes
                SET estado='publicada',
                    publicada_en = COALESCE(publicada_en, NOW()),
                    cerrada_en = NULL
                WHERE id=:id
            ");
        } elseif ($estado === 'cerrada') {
            $st = $pdo->prepare("
                UPDATE vacantes
                SET estado='cerrada',
                    cerrada_en = COALESCE(cerrada_en, NOW())
                WHERE id=:id
            ");
        } else { // borrador
            $st = $pdo->prepare("
                UPDATE vacantes
                SET estado='borrador',
                    publicada_en = NULL,
                    cerrada_en = NULL
                WHERE id=:id
            ");
        }

        $st->execute(['id' => $vacanteId]);
    }
}
