<?php
declare(strict_types=1);

final class AdminPlanModuleService
{
    public static function listModulesForPlan(int $planId): array
    {
        $pdo = DB::pdo();

        $st = $pdo->prepare("
            SELECT
                m.id,
                m.code,
                m.nombre,
                m.descripcion,
                m.scope,
                m.estado,
                CASE WHEN pm.module_id IS NULL THEN 0 ELSE pm.enabled END AS enabled
            FROM modules m
            LEFT JOIN plan_modules pm
                ON pm.module_id = m.id
               AND pm.plan_id = :pid
            ORDER BY
                FIELD(m.scope, 'platform','empresa','postulante','public'),
                m.code ASC
        ");
        $st->execute([':pid' => $planId]);

        return $st->fetchAll();
    }

    /**
     * Estrategia: "set completo"
     * - Borra todas las filas del plan
     * - Inserta solo las marcadas (enabled=1)
     *
     * Esto deja "no asignado" = deshabilitado, coherente con el seed actual. :contentReference[oaicite:4]{index=4}
     */
    public static function savePlanModules(int $planId, array $enabledModuleIds): void
    {
        if ($planId <= 0) {
            throw new ValidationException('plan_id inválido.');
        }

        $enabledIds = [];
        foreach ($enabledModuleIds as $mid) {
            $mid = (int)$mid;
            if ($mid > 0) $enabledIds[$mid] = true;
        }
        $enabledIds = array_keys($enabledIds);

        $pdo = DB::pdo();
        $pdo->beginTransaction();

        try {
            $del = $pdo->prepare("DELETE FROM plan_modules WHERE plan_id = :pid");
            $del->execute([':pid' => $planId]);

            if (!empty($enabledIds)) {
                $ins = $pdo->prepare("
                    INSERT INTO plan_modules (plan_id, module_id, enabled)
                    VALUES (:pid, :mid, 1)
                ");

                foreach ($enabledIds as $mid) {
                    $ins->execute([':pid' => $planId, ':mid' => $mid]);
                }
            }

            $pdo->commit();
        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }
}
