<?php
declare(strict_types=1);

final class AdminPlanService
{
    public static function listAll(): array
    {
        $pdo = DB::pdo();

        $sql = "
            SELECT
                id,
                code,
                nombre,
                descripcion,
                precio_mensual,
                precio_anual,
                estado,
                creado_en
            FROM plans
            ORDER BY id DESC
        ";

        return $pdo->query($sql)->fetchAll();
    }

    public static function find(int $id): ?array
    {
        $pdo = DB::pdo();

        $st = $pdo->prepare("
            SELECT
                id,
                code,
                nombre,
                descripcion,
                precio_mensual,
                precio_anual,
                estado,
                creado_en
            FROM plans
            WHERE id = :id
            LIMIT 1
        ");
        $st->execute([':id' => $id]);

        $row = $st->fetch();
        return $row ?: null;
    }

    public static function create(
        string $code,
        string $nombre,
        ?string $descripcion,
        string $precioMensual,
        string $precioAnual,
        string $estado
    ): void {
        $code = trim($code);
        $nombre = trim($nombre);
        $descripcion = $descripcion !== null ? trim($descripcion) : null;

        if ($code === '' || $nombre === '') {
            throw new ValidationException('Code y nombre son obligatorios.');
        }

        if (!preg_match('/^[A-Z0-9_]{2,50}$/', $code)) {
            throw new ValidationException('El code debe ser MAYÚSCULAS, números o _ (2 a 50). Ej: PRO, EMPRESA_10, FREE.');
        }

        $pm = self::parseMoney($precioMensual, 'precio_mensual');
        $pa = self::parseMoney($precioAnual, 'precio_anual');

        if (!in_array($estado, ['activo', 'inactivo'], true)) {
            throw new ValidationException('Estado inválido.');
        }

        $pdo = DB::pdo();

        // Uniqueness: plans.code tiene UNIQUE
        try {
            $st = $pdo->prepare("
                INSERT INTO plans (code, nombre, descripcion, precio_mensual, precio_anual, estado)
                VALUES (:c, :n, :d, :pm, :pa, :e)
            ");
            $st->execute([
                ':c'  => $code,
                ':n'  => $nombre,
                ':d'  => ($descripcion === '' ? null : $descripcion),
                ':pm' => $pm,
                ':pa' => $pa,
                ':e'  => $estado,
            ]);
        } catch (Throwable $e) {
            // Manejo amigable de duplicado sin depender del driver
            $msg = strtolower($e->getMessage());
            if (str_contains($msg, 'uq_plans_code') || str_contains($msg, 'duplicate')) {
                throw new ValidationException('Ya existe un plan con ese code.');
            }
            throw $e;
        }
    }

    public static function update(
        int $id,
        string $code,
        string $nombre,
        ?string $descripcion,
        string $precioMensual,
        string $precioAnual,
        string $estado
    ): void {
        $code = trim($code);
        $nombre = trim($nombre);
        $descripcion = $descripcion !== null ? trim($descripcion) : null;

        if ($id <= 0) {
            throw new ValidationException('ID inválido.');
        }

        if ($code === '' || $nombre === '') {
            throw new ValidationException('Code y nombre son obligatorios.');
        }

        if (!preg_match('/^[A-Z0-9_]{2,50}$/', $code)) {
            throw new ValidationException('El code debe ser MAYÚSCULAS, números o _ (2 a 50).');
        }

        $pm = self::parseMoney($precioMensual, 'precio_mensual');
        $pa = self::parseMoney($precioAnual, 'precio_anual');

        if (!in_array($estado, ['activo', 'inactivo'], true)) {
            throw new ValidationException('Estado inválido.');
        }

        $pdo = DB::pdo();

        // Evitar romper el plan base FREE por accidente (política simple)
        $row = self::find($id);
        if (!$row) {
            throw new ValidationException('El plan no existe.');
        }
        if (($row['code'] ?? '') === 'FREE' && $code !== 'FREE') {
            throw new ValidationException('No se puede cambiar el code del plan FREE.');
        }

        try {
            $st = $pdo->prepare("
                UPDATE plans
                SET
                    code = :c,
                    nombre = :n,
                    descripcion = :d,
                    precio_mensual = :pm,
                    precio_anual = :pa,
                    estado = :e
                WHERE id = :id
            ");
            $st->execute([
                ':id' => $id,
                ':c'  => $code,
                ':n'  => $nombre,
                ':d'  => ($descripcion === '' ? null : $descripcion),
                ':pm' => $pm,
                ':pa' => $pa,
                ':e'  => $estado,
            ]);
        } catch (Throwable $e) {
            $msg = strtolower($e->getMessage());
            if (str_contains($msg, 'uq_plans_code') || str_contains($msg, 'duplicate')) {
                throw new ValidationException('Ya existe un plan con ese code.');
            }
            throw $e;
        }
    }

    public static function delete(int $id): void
    {
        $pdo = DB::pdo();

        $row = self::find($id);
        if (!$row) {
            throw new ValidationException('El plan no existe.');
        }

        // seguridad mínima: no borrar FREE
        if (($row['code'] ?? '') === 'FREE') {
            throw new ValidationException('No se puede eliminar el plan FREE.');
        }

        // FK users.plan_id está ON DELETE SET NULL, así que la eliminación es válida. :contentReference[oaicite:2]{index=2}
        $st = $pdo->prepare("DELETE FROM plans WHERE id = :id");
        $st->execute([':id' => $id]);
    }

    private static function parseMoney(string $raw, string $field): string
    {
        $raw = trim($raw);
        if ($raw === '') $raw = '0';

        // Normalizar coma decimal
        $raw = str_replace(',', '.', $raw);

        if (!preg_match('/^\d+(\.\d{1,2})?$/', $raw)) {
            throw new ValidationException("Formato inválido para {$field}. Use 0, 10, 10.50");
        }

        $val = (float)$raw;
        if ($val < 0) {
            throw new ValidationException("{$field} no puede ser negativo.");
        }

        // Mantener 2 decimales como string
        return number_format($val, 2, '.', '');
    }
}
