<?php
declare(strict_types=1);

final class AdminRolePermissionService
{
    public static function listPermissionsForRole(int $roleId): array
    {
        $pdo = DB::pdo();

        $st = $pdo->prepare("
            SELECT
                p.id,
                p.code,
                p.nombre,
                p.descripcion,
                p.created_at,
                CASE WHEN rp.permission_id IS NULL THEN 0 ELSE 1 END AS assigned
            FROM permissions p
            LEFT JOIN role_permissions rp
                   ON rp.permission_id = p.id
                  AND rp.role_id = :rid
            ORDER BY p.code ASC
        ");
        $st->execute([':rid' => $roleId]);

        return $st->fetchAll();
    }

    /**
     * Estrategia: set completo
     * - DELETE role_permissions por role_id
     * - INSERT solo las permission_id marcadas
     *
     * role_permissions no tiene columna enabled; es pivote puro. :contentReference[oaicite:4]{index=4}
     */
    public static function saveRolePermissions(int $roleId, array $permissionIds): void
    {
        if ($roleId <= 0) {
            throw new ValidationException('role_id inválido.');
        }

        $ids = [];
        foreach ($permissionIds as $pid) {
            $pid = (int)$pid;
            if ($pid > 0) $ids[$pid] = true;
        }
        $ids = array_keys($ids);

        $pdo = DB::pdo();
        $pdo->beginTransaction();

        try {
            $del = $pdo->prepare("DELETE FROM role_permissions WHERE role_id = :rid");
            $del->execute([':rid' => $roleId]);

            if (!empty($ids)) {
                $ins = $pdo->prepare("
                    INSERT INTO role_permissions (role_id, permission_id)
                    VALUES (:rid, :pid)
                ");
                foreach ($ids as $pid) {
                    $ins->execute([':rid' => $roleId, ':pid' => $pid]);
                }
            }

            $pdo->commit();
        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }
}
