<?php
declare(strict_types=1);

final class GoogleOAuthService
{
    private const AUTH_URL  = 'https://accounts.google.com/o/oauth2/v2/auth';
    private const TOKEN_URL = 'https://oauth2.googleapis.com/token';
    private const USERINFO  = 'https://openidconnect.googleapis.com/v1/userinfo';

    public static function isEnabled(): bool
    {
        $cfg = $GLOBALS['APP_CONFIG']['google_oauth'] ?? [];
        return (bool)($cfg['enabled'] ?? false)
            && (string)($cfg['client_id'] ?? '') !== ''
            && (string)($cfg['client_secret'] ?? '') !== ''
            && (string)($cfg['redirect_uri'] ?? '') !== '';
    }

    /** Genera URL de login Google con state anti-CSRF */
    public static function buildAuthUrl(): string
    {
        if (!self::isEnabled()) {
            throw new RuntimeException('Google OAuth no está habilitado o falta configuración.');
        }

        $cfg = $GLOBALS['APP_CONFIG']['google_oauth'];

        $state = bin2hex(random_bytes(16));
        $_SESSION['__google_oauth_state'] = $state;

        $params = [
            'client_id'     => (string)$cfg['client_id'],
            'redirect_uri'  => (string)$cfg['redirect_uri'],
            'response_type' => 'code',
            'scope'         => 'openid email profile',
            'access_type'   => 'online',
            'prompt'        => 'select_account',
            'state'         => $state,
        ];

        return self::AUTH_URL . '?' . http_build_query($params);
    }

    /**
     * Intercambia code por access_token y devuelve userinfo:
     * - sub, email, email_verified, name, picture, ...
     */
    public static function fetchUserInfoFromCode(string $code, string $stateFromQuery): array
    {
        if (!self::isEnabled()) {
            throw new RuntimeException('Google OAuth no está habilitado.');
        }

        $state = (string)($_SESSION['__google_oauth_state'] ?? '');
        unset($_SESSION['__google_oauth_state']);

        if ($state === '' || !hash_equals($state, $stateFromQuery)) {
            throw new ValidationException('OAuth state inválido.');
        }

        $cfg = $GLOBALS['APP_CONFIG']['google_oauth'];

        // 1) token
        $token = self::postForm(self::TOKEN_URL, [
            'code'          => $code,
            'client_id'     => (string)$cfg['client_id'],
            'client_secret' => (string)$cfg['client_secret'],
            'redirect_uri'  => (string)$cfg['redirect_uri'],
            'grant_type'    => 'authorization_code',
        ]);

        $accessToken = (string)($token['access_token'] ?? '');
        if ($accessToken === '') {
            throw new RuntimeException('No se pudo obtener access_token de Google.');
        }

        // 2) userinfo
        $info = self::getJson(self::USERINFO, [
            'Authorization: Bearer ' . $accessToken,
        ]);

        // mínimo requerido
        if (!isset($info['sub']) || !is_string($info['sub']) || $info['sub'] === '') {
            throw new RuntimeException('Google userinfo inválido (sin sub).');
        }

        return $info;
    }

    private static function postForm(string $url, array $data): array
    {
        $ch = curl_init($url);
        if ($ch === false) throw new RuntimeException('cURL no disponible.');

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => http_build_query($data),
            CURLOPT_HTTPHEADER     => ['Content-Type: application/x-www-form-urlencoded'],
            CURLOPT_TIMEOUT        => 20,
        ]);

        $raw = (string)curl_exec($ch);
        $err = curl_error($ch);
        $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($raw === '' || $err) {
            throw new RuntimeException('Error cURL token: ' . ($err ?: 'respuesta vacía'));
        }

        $json = json_decode($raw, true);
        if (!is_array($json)) {
            throw new RuntimeException('Token JSON inválido.');
        }

        if ($code >= 400) {
            $msg = (string)($json['error_description'] ?? $json['error'] ?? 'Error OAuth');
            throw new RuntimeException('Google token error: ' . $msg);
        }

        return $json;
    }

    private static function getJson(string $url, array $headers = []): array
    {
        $ch = curl_init($url);
        if ($ch === false) throw new RuntimeException('cURL no disponible.');

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_TIMEOUT        => 20,
        ]);

        $raw = (string)curl_exec($ch);
        $err = curl_error($ch);
        $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($raw === '' || $err) {
            throw new RuntimeException('Error cURL userinfo: ' . ($err ?: 'respuesta vacía'));
        }

        $json = json_decode($raw, true);
        if (!is_array($json)) {
            throw new RuntimeException('Userinfo JSON inválido.');
        }

        if ($code >= 400) {
            $msg = (string)($json['error_description'] ?? $json['error'] ?? 'Error userinfo');
            throw new RuntimeException('Google userinfo error: ' . $msg);
        }

        return $json;
    }
}
