<?php
declare(strict_types=1);

final class PlanLimitService
{
    public static function getUserPlanId(int $userId): ?int
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("SELECT plan_id FROM users WHERE id = :id LIMIT 1");
        $st->execute([':id' => $userId]);
        $row = $st->fetch();
        if (!$row) return null;
        $pid = $row['plan_id'] ?? null;
        return $pid === null ? null : (int)$pid;
    }

    /**
     * value_int = 0 => ilimitado (si enabled=1)
     * si no existe el feature => retorna $default
     */
    public static function getInt(int $planId, string $featureKey, int $default): int
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT value_int, enabled
            FROM plan_limits
            WHERE plan_id = :pid AND feature_key = :k
            LIMIT 1
        ");
        $st->execute([':pid' => $planId, ':k' => $featureKey]);
        $row = $st->fetch();

        if (!$row) return $default;
        if ((int)($row['enabled'] ?? 0) !== 1) return $default;

        return (int)($row['value_int'] ?? $default);
    }

    /**
     * Regla: limita cantidad de vacantes en estado 'publicada' por empresa.
     */
    public static function assertCanPublishVacante(int $empresaUserId): void
    {
        $planId = self::getUserPlanId($empresaUserId);
        if (!$planId) return; // si no tiene plan, no bloqueamos aquí (tu sistema ya controla módulos)

        $limit = self::getInt($planId, 'max_vacantes_publicadas', 0);
        if ($limit <= 0) return; // ilimitado

        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT COUNT(*) AS c
            FROM vacantes
            WHERE empresa_user_id = :eid
              AND estado = 'publicada'
        ");
        $st->execute([':eid' => $empresaUserId]);
        $count = (int)($st->fetch()['c'] ?? 0);

        if ($count >= $limit) {
            throw new ValidationException(
                "Límite de plan alcanzado: solo puedes tener {$limit} vacante(s) publicada(s). " .
                "Cierra una vacante o mejora tu plan."
            );
        }
    }
}
