<?php
declare(strict_types=1);

/**
 * PERFIL PÚBLICO DEL POSTULANTE (sin login)
 * URL: /p?id=7
 *
 * Este archivo NO debe ejecutar $this->mapGet directamente.
 * Define una función para registrar rutas y así evita "Using $this when not in object context"
 * incluso si el archivo se incluye fuera de App::registerRoutes().
 */

if (!function_exists('register_public_postulante_profile_routes')) {

    /**
     * @param object $app Instancia de App (debe exponer mapGet y redirect si aplica)
     */
    function register_public_postulante_profile_routes(object $app): void
    {
        // Requisito: App debe tener mapGet
        if (!method_exists($app, 'mapGet')) {
            return;
        }

        $app->mapGet('/p', function (Request $req): Response {

            $id = (int)($req->get('id') ?? 0);
            if ($id <= 0) {
                return Response::html('Perfil no encontrado', 404);
            }

            $pdo = DB::pdo();

            // Solo postulantes
            $st = $pdo->prepare("SELECT id,nombre,tipo FROM users WHERE id=:id LIMIT 1");
            $st->execute([':id' => $id]);
            $user = $st->fetch(PDO::FETCH_ASSOC) ?: null;

            if (!$user || (string)($user['tipo'] ?? '') !== 'postulante') {
                return Response::html('Perfil no encontrado', 404);
            }

            // Perfil + CV
            $perfil = PostulanteProfileService::getOrCreate($id);
            $cv     = CvService::latestActiveForUser($id);

            // Categorías seleccionadas
            $selectedIds = [];
            try {
                $selectedIds = user_selected_categoria_ids($id);
            } catch (Throwable $e) {
                $selectedIds = [];
            }

            $selectedCats = [];
            if ($selectedIds) {
                $in = implode(',', array_fill(0, count($selectedIds), '?'));
                $q = $pdo->prepare("SELECT id,nombre,codigo,parent_id FROM categorias_laborales WHERE id IN ($in) ORDER BY nombre ASC");
                foreach ($selectedIds as $i => $cid) {
                    $q->bindValue($i + 1, (int)$cid, PDO::PARAM_INT);
                }
                $q->execute();
                $selectedCats = $q->fetchAll(PDO::FETCH_ASSOC) ?: [];
            }

            // Stats reales de postulaciones
            $stats = ['total'=>0,'enviada'=>0,'vista'=>0,'preseleccion'=>0,'rechazada'=>0];
            try {
                $q = $pdo->prepare("
                    SELECT estado, COUNT(*) c
                    FROM postulaciones
                    WHERE postulante_user_id = :uid
                    GROUP BY estado
                ");
                $q->execute([':uid' => $id]);
                $rows = $q->fetchAll(PDO::FETCH_ASSOC) ?: [];

                foreach ($rows as $r) {
                    $k = (string)($r['estado'] ?? '');
                    if (isset($stats[$k])) $stats[$k] = (int)($r['c'] ?? 0);
                }
                $stats['total'] = (int)$stats['enviada'] + (int)$stats['vista'] + (int)$stats['preseleccion'] + (int)$stats['rechazada'];
            } catch (Throwable $e) {
                // defaults
            }

            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Perfil público',
                'content'   => render_template('postulante/public.php', [
                    'user'         => $user,
                    'perfil'       => $perfil,
                    'cv'           => $cv,
                    'selectedCats' => $selectedCats,
                    'stats'        => $stats,
                ]),
            ]);

            return Response::html($html, 200);
        });
    }
}
