<?php
declare(strict_types=1);

function load_env_file(string $path): void
{
    if (!is_file($path) || !is_readable($path)) return;

    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if (!$lines) return;

    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || str_starts_with($line, '#') || str_starts_with($line, ';')) continue;
        if (!str_contains($line, '=')) continue;

        [$k, $v] = explode('=', $line, 2);
        $k = trim($k);
        $v = trim($v);

        // quitar comillas si vienen
        if ((str_starts_with($v, '"') && str_ends_with($v, '"')) || (str_starts_with($v, "'") && str_ends_with($v, "'"))) {
            $v = substr($v, 1, -1);
        }

        if ($k === '') continue;

        $_ENV[$k] = $v;
        $_SERVER[$k] = $v;
        @putenv($k . '=' . $v);
    }
}

function config(): array
{
    $root = dirname(__DIR__, 2);

    // Cargar .env SIEMPRE antes de leer getenv()
    load_env_file($root . '/.env');

    $env = static function (string $key, string $default = ''): string {
        $v = getenv($key);
        return ($v === false || $v === null) ? $default : (string)$v;
    };

    $envBool = static function (string $key, bool $default = false): bool {
        $v = getenv($key);
        if ($v === false || $v === null || $v === '') return $default;
        $v = strtolower((string)$v);
        return in_array($v, ['1','true','yes','on'], true);
    };

    $isHttps = (
        (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
        || (($_SERVER['SERVER_PORT'] ?? '') === '443')
        || (strtolower((string)($_SERVER['HTTP_X_FORWARDED_PROTO'] ?? '')) === 'https')
        || (isset($_SERVER['HTTP_CF_VISITOR']) && str_contains((string)$_SERVER['HTTP_CF_VISITOR'], 'https'))
    );

    $host   = (string)($_SERVER['HTTP_HOST'] ?? '');
    $scheme = $isHttps ? 'https' : 'http';
    $baseUrl = $env('APP_BASE_URL', $host !== '' ? ($scheme . '://' . $host) : '');

    $cookiesSecure = $envBool('APP_COOKIES_SECURE', $isHttps);

    return [
        'app' => [
            'env'      => $env('APP_ENV', 'production'),
            'debug'    => $envBool('APP_DEBUG', false),
            'base_url' => $baseUrl,
        ],

        'paths' => [
            'root'          => $root,
            'logs_dir'      => $root . '/storage/logs',
            'cache_dir'     => $root . '/storage/cache',
            'tmp_dir'       => $root . '/storage/tmp',
            'templates_dir' => $root . '/templates',
        ],

        'security' => [
            'session_name'    => $env('APP_SESSION_NAME', 'LABSESSID'),
            'cookies_secure'  => $cookiesSecure,

            'csrf' => [
                'enabled'     => $envBool('APP_CSRF_ENABLED', true),
                'header'      => $env('APP_CSRF_HEADER', 'X-CSRF-Token'),
                'ttl_seconds' => (int)$env('APP_CSRF_TTL', '1800'),
            ],

            'rate_limit' => [
                'enabled' => $envBool('APP_RATELIMIT_ENABLED', true),
                'default' => [
                    'window_seconds' => (int)$env('APP_RATELIMIT_WINDOW', '60'),
                    'max'            => (int)$env('APP_RATELIMIT_MAX', '120'),
                ],
                'routes' => [
                    '/login'    => ['window_seconds' => 60, 'max' => 10],
                    '/registro' => ['window_seconds' => 60, 'max' => 6],
                    '/captcha'  => ['window_seconds' => 60, 'max' => 60],
                ],
            ],

            'headers' => [
                'csp'  => $env(
                    'APP_CSP',
                    "default-src 'self'; img-src 'self' data:; style-src 'self' 'unsafe-inline' https://cdn.jsdelivr.net; script-src 'self' 'unsafe-inline' https://cdn.jsdelivr.net;"
                ),
                'hsts' => $envBool('APP_HSTS', true),
            ],
        ],

        'seo' => [
            'site_name'           => $env('SEO_SITE_NAME', 'Red Laboral Segura'),
            'default_description' => $env('SEO_DEFAULT_DESCRIPTION', 'Vacantes verificadas y perfiles reales.'),
        ],

        'db' => [
            'dsn'  => $env('DB_DSN', ''),
            'user' => $env('DB_USER', ''),
            'pass' => $env('DB_PASS', ''),
        ],

        'auth' => [
            'session_user_key'  => $env('AUTH_SESSION_USER_KEY', '__uid'),
            'max_failed_logins' => (int)$env('AUTH_MAX_FAILED_LOGINS', '8'),
            'lock_minutes'      => (int)$env('AUTH_LOCK_MINUTES', '15'),
        ],

        'google_oauth' => [
            'enabled'       => $envBool('GOOGLE_OAUTH_ENABLED', false),
            'client_id'     => $env('GOOGLE_CLIENT_ID', ''),
            'client_secret' => $env('GOOGLE_CLIENT_SECRET', ''),
            'redirect_uri'  => $env('GOOGLE_REDIRECT_URI', ''),
        ],
    ];
}
