<?php
declare(strict_types=1);

final class DB
{
    private static ?PDO $pdo = null;

    public static function pdo(): PDO
    {
        if (self::$pdo) return self::$pdo;

        $cfg = $GLOBALS['APP_CONFIG']['db'] ?? [];
        $dsn = (string)($cfg['dsn'] ?? '');
        $usr = (string)($cfg['user'] ?? '');
        $pwd = (string)($cfg['pass'] ?? '');

        if ($dsn === '') {
            throw new RuntimeException('DB_DSN no configurado.');
        }

        // Asegurar utf8mb4 si es MySQL y no viene en el DSN
        if (str_starts_with($dsn, 'mysql:') && !str_contains($dsn, 'charset=')) {
            $dsn .= ';charset=utf8mb4';
        }

        $pdo = new PDO($dsn, $usr, $pwd, [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
        ]);

        self::$pdo = $pdo;
        return $pdo;
    }

    public static function ping(): bool
    {
        $pdo = self::pdo();
        $pdo->query('SELECT 1');
        return true;
    }

    public static function exec(string $sql, array $params = []): int
    {
        $st = self::pdo()->prepare($sql);
        $st->execute($params);
        return $st->rowCount();
    }

    public static function query(string $sql, array $params = []): array
    {
        $st = self::pdo()->prepare($sql);
        $st->execute($params);
        return $st->fetchAll();
    }

    public static function one(string $sql, array $params = []): ?array
    {
        $st = self::pdo()->prepare($sql);
        $st->execute($params);
        $row = $st->fetch();
        return $row === false ? null : $row;
    }

    public static function tx(callable $fn)
    {
        $pdo = self::pdo();
        $pdo->beginTransaction();
        try {
            $out = $fn($pdo);
            $pdo->commit();
            return $out;
        } catch (Throwable $e) {
            if ($pdo->inTransaction()) $pdo->rollBack();
            throw $e;
        }
    }
}
