<?php
declare(strict_types=1);

final class RateLimitMiddleware
{
    public function __construct(private array $config, private Logger $logger) {}

    public function __invoke(Request $req, callable $next): Response
    {
        if (empty($this->config['security']['rate_limit']['enabled'])) {
            return $next($req);
        }

        $policy = $this->config['security']['rate_limit']['default'] ?? ['window_seconds' => 60, 'max' => 120];
        $routePolicies = $this->config['security']['rate_limit']['routes'] ?? [];
        if (isset($routePolicies[$req->path])) {
            $policy = $routePolicies[$req->path];
        }

        $limiter = new RateLimiter($this->config['paths']['cache_dir']);
        $key = 'rl:' . $req->path . ':' . $req->ip;

        $allowed = $limiter->hit($key, (int)$policy['window_seconds'], (int)$policy['max']);
        if (!$allowed) {
            $this->logger->security('Rate limit exceeded', [
                'path' => $req->path,
                'ip' => $req->ip,
                'ua' => $req->userAgent,
            ]);
            return Response::json(['ok' => false, 'error' => 'Demasiadas solicitudes'], 429);
        }

        return $next($req);
    }
}
