<?php
declare(strict_types=1);

final class Router
{
    /** @var array<int, callable> */
    private array $middlewares = [];

    /** @var array<string, array<string, callable>> */
    private array $routes = [
        'GET'  => [],
        'POST' => [],
    ];

    /** @var callable|null */
    private $notFound = null;

    /**
     * Acepta:
     * - callable(Request $req, callable $next): Response
     * - objeto con método handle(Request $req, callable $next): Response
     */
    public function addMiddleware(callable|object $mw): void
    {
        // Si es callable directo, lo guardamos tal cual
        if (is_callable($mw)) {
            $this->middlewares[] = $mw;
            return;
        }

        // Si es objeto, debe tener handle()
        if (is_object($mw) && method_exists($mw, 'handle')) {
            $this->middlewares[] = function (Request $req, callable $next) use ($mw): Response {
                /** @var object $mw */
                return $mw->handle($req, $next);
            };
            return;
        }

        throw new TypeError('Middleware inválido. Debe ser callable o tener método handle().');
    }

    public function get(string $path, callable $handler): void
    {
        $this->routes['GET'][$path] = $handler;
    }

    public function post(string $path, callable $handler): void
    {
        $this->routes['POST'][$path] = $handler;
    }

    public function setNotFoundHandler(callable $handler): void
    {
        $this->notFound = $handler;
    }

    public function dispatch(Request $req): Response
    {
        $method = strtoupper($req->method);
        $path   = $req->path;

        $handler = $this->routes[$method][$path] ?? null;

        if (!$handler) {
            if ($this->notFound) {
                return ($this->notFound)($req);
            }
            return Response::json(['ok' => false, 'error' => 'Ruta no encontrada'], 404);
        }

        // Pipeline: middlewares -> handler
        $next = function (Request $r) use ($handler): Response {
            return $handler($r);
        };

        // envolver en orden inverso
        for ($i = count($this->middlewares) - 1; $i >= 0; $i--) {
            $mw = $this->middlewares[$i];
            $prev = $next;
            $next = function (Request $r) use ($mw, $prev): Response {
                return $mw($r, $prev);
            };
        }

        return $next($req);
    }
}
