<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/DB/DB.php';

final class EmpresaPublicService
{
    /**
     * Lista pública: SOLO empresas que tienen vacantes publicadas (como dice tu UI).
     * Trae: logo_path, ubicacion, total vacantes publicadas y reputación (score 0..10).
     */
    public static function listEmpresasPublicas(int $limit = 24, int $offset = 0): array
    {
        $pdo = DB::pdo();

        $limit  = max(1, min(60, $limit));
        $offset = max(0, $offset);

        // reputación en SQL (misma lógica que EmpresaReputationService: P/(P+N)*10)
        $sql = "
            SELECT
              u.id,
              u.nombre,
              u.email,
              u.email_verificado,

              ep.ubicacion,
              ep.logo_path,

              (SELECT COUNT(*)
                 FROM vacantes v
                WHERE v.empresa_user_id = u.id
                  AND v.estado = 'publicada'
              ) AS vacantes_publicadas,

              COALESCE((
                SELECT ROUND(
                  (
                    SUM(CASE WHEN r.reaccion IN ('like','recommend','excellent') THEN 1 ELSE 0 END)
                    / NULLIF(SUM(CASE WHEN r.reaccion IN ('like','recommend','excellent','bad') THEN 1 ELSE 0 END),0)
                  ) * 10
                ,2)
                FROM vacantes v2
                LEFT JOIN vacante_reacciones r ON r.vacante_id = v2.id
                WHERE v2.empresa_user_id = u.id
              ), 0) AS reputacion_score

            FROM users u
            LEFT JOIN empresa_perfiles ep ON ep.user_id = u.id
            WHERE u.tipo = 'empresa'
            HAVING vacantes_publicadas > 0
            ORDER BY vacantes_publicadas DESC, reputacion_score DESC, u.id DESC
            LIMIT {$limit} OFFSET {$offset}
        ";

        $st = $pdo->query($sql);
        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    public static function findEmpresa(int $empresaId): ?array
    {
        if ($empresaId <= 0) return null;

        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT id, nombre, email, creado_en
            FROM users
            WHERE id = :id AND tipo = 'empresa'
            LIMIT 1
        ");
        $st->execute(['id' => $empresaId]);
        $row = $st->fetch(\PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public static function listVacantesPublicadas(int $empresaId): array
    {
        if ($empresaId <= 0) return [];

        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT id, titulo, ubicacion, modalidad, tipo_empleo, salario_min, salario_max, moneda, estado
            FROM vacantes
            WHERE empresa_user_id = :eid AND estado = 'publicada'
            ORDER BY id DESC
        ");
        $st->execute(['eid' => $empresaId]);
        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    public static function countVacantesPublicadas(int $empresaId): int
    {
        if ($empresaId <= 0) return 0;

        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT COUNT(*) AS c
            FROM vacantes
            WHERE empresa_user_id = :eid AND estado = 'publicada'
        ");
        $st->execute(['eid' => $empresaId]);
        $row = $st->fetch(\PDO::FETCH_ASSOC);
        return (int)($row['c'] ?? 0);
    }
}
