<?php
declare(strict_types=1);

final class FeedService
{
    /* =========================
       IDEALES / PATROCINADAS
       - FIX CLAVE: si hay categorías y NO sale nada, cae a global
       ========================= */
    public static function idealesParaTi(int $userId, array $categoriaIds = [], int $limit = 5): array
    {
        $pdo = DB::pdo();
        $limit = max(1, min(10, (int)$limit));

        $catIds = array_values(array_filter(array_map('intval', $categoriaIds), fn($v) => $v > 0));
        $catIds = array_slice(array_unique($catIds), 0, 20);

        // 1) Intento: para_ti + match
        if (!empty($catIds)) {
            $rows = self::idealesQuery($pdo, $catIds, 'para_ti', $limit);
            if (!empty($rows)) return $rows;
        }

        // 2) Fallback: global
        return self::idealesQuery($pdo, [], 'global', $limit);
    }

    private static function idealesQuery(PDO $pdo, array $catIds, string $scope, int $limit): array
    {
        $inCats = !empty($catIds) ? implode(',', $catIds) : '';

        $matchCountExpr = '0';
        $catMatchExpr   = "''";
        $whereExtra     = " AND fb.scope='global' ";

        if ($scope === 'para_ti' && $inCats !== '') {
            $matchCountExpr = "COUNT(DISTINCT CASE WHEN vc.categoria_id IN ($inCats) THEN vc.categoria_id END)";
            $catMatchExpr   = "GROUP_CONCAT(DISTINCT CASE WHEN vc.categoria_id IN ($inCats) THEN cl.nombre ELSE NULL END ORDER BY cl.nombre SEPARATOR ', ')";
            // importante: solo pedimos que la VACANTE haga match con mis cats
            $whereExtra = " AND fb.scope='para_ti' AND vc.categoria_id IN ($inCats) ";
        }

        $sql = "
            SELECT
              v.id,
              v.empresa_user_id,
              v.titulo,
              v.descripcion,
              v.ubicacion,
              v.modalidad,
              v.tipo_empleo,
              v.salario_min,
              v.salario_max,
              v.moneda,
              v.imagen_path,
              COALESCE(v.publicada_en, v.creado_en) AS fecha_feed,

              u.nombre AS empresa_nombre,
              u.email_verificado AS empresa_verificada,
              ep.logo_path AS empresa_logo_path,

              1 AS is_ideal,
              MAX(fb.score_boost) AS boost_max,

              GROUP_CONCAT(DISTINCT cl.nombre ORDER BY cl.nombre SEPARATOR ', ') AS categorias_vacante,
              {$catMatchExpr} AS categorias_match,
              {$matchCountExpr} AS match_count

            FROM vacantes v
            INNER JOIN feed_boosts fb
              ON fb.actor_user_id = v.empresa_user_id
             AND fb.actor_tipo = 'empresa'
             AND fb.estado = 'activo'
             AND NOW() BETWEEN fb.starts_at AND fb.ends_at

            LEFT JOIN users u
              ON u.id = v.empresa_user_id
            LEFT JOIN empresa_perfiles ep
              ON ep.user_id = v.empresa_user_id

            LEFT JOIN vacante_categorias vc
              ON vc.vacante_id = v.id
            LEFT JOIN categorias_laborales cl
              ON cl.id = vc.categoria_id

            WHERE v.estado='publicada'
              {$whereExtra}

            GROUP BY v.id
            HAVING boost_max > 0
            ORDER BY
              match_count DESC,
              boost_max DESC,
              fecha_feed DESC,
              v.id DESC
            LIMIT {$limit}
        ";

        $st = $pdo->prepare($sql);
        $st->execute();
        return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    /* =========================
       FEED POSTULANTE (CENTRAL)
       3 niveles RANDOM:
       1) patrocinadas MATCH (con cap/slots)
       2) recientes <= 15 días MATCH (random ponderado)
       3) antiguas >= 16 días MATCH (random)
       Fallback si no hay categorías
       + integra anuncios RRHH (con reacciones)
       ========================= */
    public static function feedForPostulante(
        int $userId,
        string $q = '',
        int $limit = 20,
        int $offset = 0,
        array $filters = []
    ): array {
        $pdo = DB::pdo();

        $q = trim($q);
        $limit  = max(1, min(100, (int)$limit));
        $offset = max(0, (int)$offset);

        $onlyMatching = ((int)($filters['only_matching'] ?? 0) === 1);

        $catIds = array_values(array_filter(array_map('intval', (array)($filters['categoria_ids'] ?? [])), fn($v) => $v > 0));
        $catIds = array_slice(array_unique($catIds), 0, 20);
        $hasCats = !empty($catIds);
        $inCats = $hasCats ? implode(',', $catIds) : '';

        $fetch = 260; // pool grande para random estable
        $take  = $limit + $offset;

        /* ---------- VACANTES pool ---------- */
        $matchCountExpr = $hasCats
            ? "COUNT(DISTINCT CASE WHEN vc.categoria_id IN ($inCats) THEN vc.categoria_id END)"
            : "0";

        $categoriasMatchExpr = $hasCats
            ? "GROUP_CONCAT(DISTINCT CASE WHEN vc.categoria_id IN ($inCats) THEN cl.nombre ELSE NULL END ORDER BY cl.nombre SEPARATOR ', ')"
            : "''";

        $whereParts = ["v.estado = 'publicada'"];
        $params = [
            ':uid1' => $userId, // postulaciones join
            ':uid2' => $userId, // my reaction
        ];

        if ($q !== '') {
            // NO reutilizar placeholder (HY093)
            $whereParts[] = "(v.titulo LIKE :q1 OR v.ubicacion LIKE :q2 OR v.descripcion LIKE :q3)";
            $like = '%' . $q . '%';
            $params[':q1'] = $like;
            $params[':q2'] = $like;
            $params[':q3'] = $like;
        }

        $where = implode(' AND ', $whereParts);

        $sqlVac = "
            SELECT
                'vacante' AS kind,

                v.id,
                v.empresa_user_id,
                v.titulo,
                v.descripcion,
                v.ubicacion,
                v.modalidad,
                v.tipo_empleo,
                v.salario_min,
                v.salario_max,
                v.moneda,
                v.estado,

                v.imagen_path AS imagen_path,
                COALESCE(v.publicada_en, v.creado_en) AS fecha_feed,
                DATEDIFF(NOW(), COALESCE(v.publicada_en, v.creado_en)) AS age_days,

                u.nombre,
                u.email,
                u.email_verificado,

                ep.logo_path,

                {$matchCountExpr} AS match_count,
                {$categoriasMatchExpr} AS categorias_match,

                COALESCE(p.id, 0) AS postulacion_id,
                COALESCE(p.estado, '') AS postulacion_estado,
                p.creado_en AS postulacion_creado_en,

                COALESCE((
                    SELECT r0.reaccion
                    FROM vacante_reacciones r0
                    WHERE r0.vacante_id = v.id AND r0.user_id = :uid2
                    LIMIT 1
                ), '') AS my_reaction,

                (SELECT COUNT(*) FROM vacante_reacciones r1 WHERE r1.vacante_id = v.id AND r1.reaccion='like') AS reac_like,
                (SELECT COUNT(*) FROM vacante_reacciones r2 WHERE r2.vacante_id = v.id AND r2.reaccion='recommend') AS reac_recommend,
                (SELECT COUNT(*) FROM vacante_reacciones r3 WHERE r3.vacante_id = v.id AND r3.reaccion='excellent') AS reac_excellent,
                (SELECT COUNT(*) FROM vacante_reacciones r4 WHERE r4.vacante_id = v.id AND r4.reaccion='bad') AS reac_bad,
                (SELECT COUNT(*) FROM vacante_compartidos s WHERE s.vacante_id = v.id) AS shares_count,

                MAX(COALESCE(fb.score_boost,0)) AS boost_max

            FROM vacantes v
            INNER JOIN users u ON u.id = v.empresa_user_id
            LEFT JOIN empresa_perfiles ep ON ep.user_id = v.empresa_user_id
            LEFT JOIN postulaciones p ON p.vacante_id = v.id AND p.postulante_user_id = :uid1

            LEFT JOIN vacante_categorias vc ON vc.vacante_id = v.id
            LEFT JOIN categorias_laborales cl ON cl.id = vc.categoria_id

            LEFT JOIN feed_boosts fb
              ON fb.actor_user_id = v.empresa_user_id
             AND fb.actor_tipo = 'empresa'
             AND fb.estado = 'activo'
             AND NOW() BETWEEN fb.starts_at AND fb.ends_at
             AND (
                fb.categoria_id IS NULL
                OR fb.categoria_id = vc.categoria_id
             )
             AND (
                fb.scope = 'global'
                OR (fb.scope = 'para_ti' AND " . ($hasCats ? "vc.categoria_id IN ($inCats)" : "0") . ")
             )

            WHERE {$where}
            GROUP BY
              v.id, v.empresa_user_id, v.titulo, v.descripcion, v.ubicacion, v.modalidad, v.tipo_empleo,
              v.salario_min, v.salario_max, v.moneda, v.estado,
              v.imagen_path, fecha_feed, age_days,
              u.nombre, u.email, u.email_verificado,
              ep.logo_path,
              p.id, p.estado, p.creado_en
            LIMIT {$fetch} OFFSET 0
        ";

        $st = $pdo->prepare($sqlVac);

        // bind SOLO placeholders usados
        $used = [];
        if (preg_match_all('/:\w+/', $sqlVac, $m)) foreach ($m[0] as $ph) $used[$ph] = true;
        $paramsVac = array_filter($params, fn($v, $k) => isset($used[$k]), ARRAY_FILTER_USE_BOTH);

        $st->execute($paramsVac);
        $vacItems = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

        // Si only=1 y tengo cats: filtro match>0 (antes de random)
        if ($onlyMatching && $hasCats) {
            $vacItems = array_values(array_filter($vacItems, fn($r) => (int)($r['match_count'] ?? 0) > 0));
        }

        /* ---------- ANUNCIOS RRHH pool ---------- */
        $paramsA = [];
        $whereA = ["a.estado = 'publicado'"];

        if ($q !== '') {
            $whereA[] = "(a.titulo LIKE :qa1 OR a.descripcion LIKE :qa2)";
            $like = '%' . $q . '%';
            $paramsA[':qa1'] = $like;
            $paramsA[':qa2'] = $like;
        }

        // NOTA: requiere tablas rrhh_anuncio_reacciones / rrhh_anuncio_compartidos (te dejo SQL abajo)
        $sqlA = "
            SELECT
                'rrhh_anuncio' AS kind,

                a.id,
                a.rrhh_user_id,
                a.titulo,
                a.descripcion,
                a.tipo AS anuncio_tipo,
                a.imagen_path,
                COALESCE(a.publicado_en, a.creado_en) AS fecha_feed,
                DATEDIFF(NOW(), COALESCE(a.publicado_en, a.creado_en)) AS age_days,

                COALESCE(rp.display_name, u.nombre) AS autor_nombre,
                rp.foto_path AS autor_foto_path,

                COALESCE((
                    SELECT r0.reaccion
                    FROM rrhh_anuncio_reacciones r0
                    WHERE r0.anuncio_id = a.id AND r0.user_id = :uidA
                    LIMIT 1
                ), '') AS my_reaction,

                (SELECT COUNT(*) FROM rrhh_anuncio_reacciones r1 WHERE r1.anuncio_id = a.id AND r1.reaccion='like') AS reac_like,
                (SELECT COUNT(*) FROM rrhh_anuncio_reacciones r2 WHERE r2.anuncio_id = a.id AND r2.reaccion='recommend') AS reac_recommend,
                (SELECT COUNT(*) FROM rrhh_anuncio_reacciones r3 WHERE r3.anuncio_id = a.id AND r3.reaccion='excellent') AS reac_excellent,
                (SELECT COUNT(*) FROM rrhh_anuncio_reacciones r4 WHERE r4.anuncio_id = a.id AND r4.reaccion='bad') AS reac_bad,
                (SELECT COUNT(*) FROM rrhh_anuncio_compartidos s WHERE s.anuncio_id = a.id) AS shares_count,

                MAX(COALESCE(fb.score_boost,0)) AS boost_max,
                0 AS match_count,
                '' AS categorias_match

            FROM rrhh_anuncios a
            INNER JOIN users u ON u.id = a.rrhh_user_id
            LEFT JOIN rrhh_profiles rp ON rp.user_id = a.rrhh_user_id

            LEFT JOIN feed_boosts fb
              ON fb.actor_user_id = a.rrhh_user_id
             AND fb.actor_tipo = 'rrhh'
             AND fb.estado = 'activo'
             AND NOW() BETWEEN fb.starts_at AND fb.ends_at
             AND fb.scope = 'global'
             AND fb.categoria_id IS NULL

            WHERE " . implode(' AND ', $whereA) . "
            GROUP BY
              a.id, a.rrhh_user_id, a.titulo, a.descripcion, a.tipo, a.imagen_path, fecha_feed, age_days,
              autor_nombre, autor_foto_path
            LIMIT {$fetch} OFFSET 0
        ";

        $stA = $pdo->prepare($sqlA);
        $stA->execute(array_merge($paramsA, [':uidA' => $userId]));
        $aItems = $stA->fetchAll(PDO::FETCH_ASSOC) ?: [];

        // Sanitizar HTML anuncios (interpretar HTML sin XSS)
        foreach ($aItems as &$row) {
            $html = (string)($row['descripcion'] ?? '');
            if ($html !== '') {
                $html = preg_replace('~<(script|iframe|object|embed|style)\b[^>]*>.*?</\1>~is', '', $html) ?? '';
                $allowed = '<p><br><b><strong><i><em><ul><ol><li><a><blockquote><code><pre><h1><h2><h3><h4><h5><h6>';
                $html = strip_tags($html, $allowed);
                $html = preg_replace('~\son\w+\s*=\s*(".*?"|\'.*?\'|[^\s>]+)~i', '', $html) ?? '';
                $html = preg_replace('~\sstyle\s*=\s*(".*?"|\'.*?\')~i', '', $html) ?? '';
                $row['descripcion'] = $html;
            }
        }
        unset($row);

        /* ---------- RANDOM 3 NIVELES (VACANTES) ---------- */
        $sponsored = [];
        $recent    = [];
        $old       = [];
        $vacRest   = [];

        foreach ($vacItems as $it) {
            $boost = (int)($it['boost_max'] ?? 0);
            $mc    = (int)($it['match_count'] ?? 0);
            $age   = (int)($it['age_days'] ?? 999);

            $isMatch = $hasCats ? ($mc > 0) : true;

            // Si tengo cats, solo considero "match" para prioridad.
            if ($hasCats && !$isMatch) {
                $vacRest[] = $it;
                continue;
            }

            if ($boost > 0 && (!$hasCats || $mc > 0)) {
                $sponsored[] = $it;
            } elseif ($age <= 15) {
                $recent[] = $it;
            } else {
                $old[] = $it;
            }
        }

        // slots/cap (patrocinadas random)
        $sponsored = self::shuffleSimple($sponsored);
        $sponsored = array_slice($sponsored, 0, 6);

        // recientes: random ponderado (más reciente + más match_count = más probabilidad)
        $recent = self::shuffleWeighted($recent, function (array $r) use ($hasCats): float {
            $age = max(0, (int)($r['age_days'] ?? 99));
            $mc  = (int)($r['match_count'] ?? 0);

            $wAge = 1.0 / (1.0 + $age); // 0 días => 1, 15 => ~0.0625
            $wMc  = $hasCats ? (1.0 + min(8, $mc) * 0.35) : 1.0;

            return max(0.05, $wAge * $wMc);
        });

        // antiguas: random simple
        $old = self::shuffleSimple($old);

        // anuncios: metelos entre recientes/antiguas según edad; patrocinados RRHH arriba si boost>0
        $anSponsored = [];
        $anRecent = [];
        $anOld = [];

        foreach ($aItems as $a) {
            $boost = (int)($a['boost_max'] ?? 0);
            $age   = (int)($a['age_days'] ?? 999);
            if ($boost > 0) $anSponsored[] = $a;
            elseif ($age <= 15) $anRecent[] = $a;
            else $anOld[] = $a;
        }

        $anSponsored = self::shuffleSimple($anSponsored);
        $anRecent    = self::shuffleSimple($anRecent);
        $anOld       = self::shuffleSimple($anOld);

        // Merge final por niveles:
        // 1) patrocinadas vacantes + patrocinados anuncios
        // 2) recientes vacantes + recientes anuncios
        // 3) antiguas vacantes + antiguas anuncios
        // 4) fallback (no match) si aplica y hace falta llenar
        $all = array_merge(
            $sponsored,
            $anSponsored,
            $recent,
            $anRecent,
            $old,
            $anOld
        );

        // fallback si hay cats: si faltan items, meto no-match al final (sin prioridad)
        if ($hasCats && count($all) < $take) {
            $vacRest = self::shuffleSimple($vacRest);
            $all = array_merge($all, $vacRest);
        }

        // recorte paginado
        $items = array_slice($all, $offset, $limit);

        $stats = self::postulanteStats($userId);
        return ['items' => $items, 'stats' => $stats];
    }

    public static function feedForEmpresa(
        int $empresaUserId,
        string $q = '',
        int $limit = 20,
        int $offset = 0
    ): array {
        $pdo = DB::pdo();

        $q = trim($q);
        $limit  = max(1, min(100, (int)$limit));
        $offset = max(0, (int)$offset);

        $where = "v.empresa_user_id = :eid";
        $params = [':eid' => $empresaUserId];

        if ($q !== '') {
            $like = '%' . $q . '%';
            $where .= " AND (
                v.titulo LIKE :q1
                OR v.ubicacion LIKE :q2
                OR v.descripcion LIKE :q3
                OR v.slug LIKE :q4
            )";
            $params[':q1'] = $like;
            $params[':q2'] = $like;
            $params[':q3'] = $like;
            $params[':q4'] = $like;

            if (ctype_digit($q)) {
                $where .= " AND v.id = :qid";
                $params[':qid'] = (int)$q;
            }
        }

        $sql = "
            SELECT
                v.id,
                v.empresa_user_id,
                v.titulo,
                v.descripcion,
                v.ubicacion,
                v.modalidad,
                v.tipo_empleo,
                v.salario_min,
                v.salario_max,
                v.moneda,
                v.estado,

                v.imagen_path AS imagen_path,
                COALESCE(v.publicada_en, v.creado_en) AS fecha_feed,

                u.nombre AS empresa_nombre,
                u.email  AS empresa_email,
                ep.logo_path AS empresa_logo_path,

                (SELECT COUNT(*) FROM postulaciones p WHERE p.vacante_id = v.id) AS postulaciones_count,

                (SELECT COUNT(*) FROM vacante_reacciones r1 WHERE r1.vacante_id = v.id AND r1.reaccion='like')      AS reac_like,
                (SELECT COUNT(*) FROM vacante_reacciones r2 WHERE r2.vacante_id = v.id AND r2.reaccion='recommend') AS reac_recommend,
                (SELECT COUNT(*) FROM vacante_reacciones r3 WHERE r3.vacante_id = v.id AND r3.reaccion='excellent') AS reac_excellent,
                (SELECT COUNT(*) FROM vacante_reacciones r4 WHERE r4.vacante_id = v.id AND r4.reaccion='bad')       AS reac_bad,
                (SELECT COUNT(*) FROM vacante_compartidos s WHERE s.vacante_id = v.id)                               AS shares_count,

                '' AS my_reaction
            FROM vacantes v
            LEFT JOIN users u ON u.id = v.empresa_user_id
            LEFT JOIN empresa_perfiles ep ON ep.user_id = v.empresa_user_id
            WHERE {$where}
            ORDER BY fecha_feed DESC, v.id DESC
            LIMIT {$limit} OFFSET {$offset}
        ";

        $st = $pdo->prepare($sql);

        $used = [];
        if (preg_match_all('/:\w+/', $sql, $m)) foreach ($m[0] as $ph) $used[$ph] = true;
        $params = array_filter($params, fn($v, $k) => isset($used[$k]), ARRAY_FILTER_USE_BOTH);

        $st->execute($params);

        $items = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
        $stats = self::empresaStats($empresaUserId);

        return ['items' => $items, 'stats' => $stats];
    }

    private static function postulanteStats(int $uid): array
    {
        $pdo = DB::pdo();

        $sql = "
            SELECT estado, COUNT(*) c
            FROM postulaciones
            WHERE postulante_user_id = :uid
            GROUP BY estado
        ";

        $st = $pdo->prepare($sql);
        $st->execute([':uid' => $uid]);

        $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $map = [
            'enviada'       => 0,
            'vista'         => 0,
            'preseleccion'  => 0,
            'rechazada'     => 0,
        ];

        foreach ($rows as $r) {
            $k = (string)($r['estado'] ?? '');
            if (isset($map[$k])) $map[$k] = (int)($r['c'] ?? 0);
        }

        $total = array_sum($map);
        return ['total' => $total] + $map;
    }

    private static function empresaStats(int $eid): array
    {
        $pdo = DB::pdo();

        $sql = "
            SELECT
                (SELECT COUNT(*) FROM vacantes v1 WHERE v1.empresa_user_id = :eid1) AS vacantes_total,
                (SELECT COUNT(*) FROM vacantes v2 WHERE v2.empresa_user_id = :eid2 AND v2.estado = 'publicada') AS vacantes_publicadas,
                (SELECT COUNT(*) FROM postulaciones p
                   INNER JOIN vacantes v3 ON v3.id = p.vacante_id
                  WHERE v3.empresa_user_id = :eid3
                ) AS postulaciones_total
        ";

        $st = $pdo->prepare($sql);
        $st->execute([
            ':eid1' => $eid,
            ':eid2' => $eid,
            ':eid3' => $eid,
        ]);

        $row = $st->fetch(PDO::FETCH_ASSOC) ?: [];

        return [
            'vacantes_total'      => (int)($row['vacantes_total'] ?? 0),
            'vacantes_publicadas' => (int)($row['vacantes_publicadas'] ?? 0),
            'postulaciones_total' => (int)($row['postulaciones_total'] ?? 0),
        ];
    }

    /* =========================
       RANDOM HELPERS
       ========================= */

    private static function shuffleSimple(array $items): array
    {
        if (count($items) <= 1) return $items;
        // Fisher–Yates con random_int
        for ($i = count($items) - 1; $i > 0; $i--) {
            $j = random_int(0, $i);
            [$items[$i], $items[$j]] = [$items[$j], $items[$i]];
        }
        return $items;
    }

    /**
     * Weighted random permutation (Efraimidis-Spirakis):
     * key = -log(U) / w
     * mayor w => tiende a quedar primero
     */
    private static function shuffleWeighted(array $items, callable $weightFn): array
    {
        if (count($items) <= 1) return $items;

        $keys = [];
        foreach ($items as $idx => $it) {
            $w = (float)$weightFn($it);
            $w = max(0.0001, $w);
            $u = max(0.0000001, (random_int(1, PHP_INT_MAX - 1) / PHP_INT_MAX));
            $keys[$idx] = (-log($u)) / $w;
        }

        uasort($items, function ($a, $b) use ($keys) {
            // no tenemos idx aquí, así que reindexamos abajo
            return 0;
        });

        // reordenar por keys usando indices originales
        $pairs = [];
        foreach ($items as $idx => $it) {
            $pairs[] = ['k' => $keys[$idx] ?? 999999.0, 'it' => $it];
        }

        usort($pairs, fn($x, $y) => $x['k'] <=> $y['k']);

        $out = [];
        foreach ($pairs as $p) $out[] = $p['it'];
        return $out;
    }
}
