<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/DB/DB.php';
require_once dirname(__DIR__) . '/Support/Exceptions.php';

final class PostulacionService
{
    /** Crear postulación (una por vacante por usuario) */
    public static function create(int $vacanteId, int $postulanteUserId, ?string $mensaje): void
    {
        if ($vacanteId <= 0) {
            throw new ValidationException('Vacante inválida.');
        }
        if ($postulanteUserId <= 0) {
            throw new ValidationException('Usuario inválido.');
        }

        $msg = $mensaje !== null ? trim($mensaje) : null;
        if ($msg !== null && $msg === '') $msg = null;

        // (Opcional) Validación simple del texto
        if ($msg !== null && mb_strlen($msg) > 5000) {
            throw new ValidationException('El mensaje es demasiado largo.');
        }

        $pdo = DB::pdo();

        // Verifica si ya postuló (índice único debe existir, pero lo manejamos elegantemente)
        $st = $pdo->prepare("SELECT id FROM postulaciones WHERE vacante_id = :vid AND postulante_user_id = :uid LIMIT 1");
        $st->execute(['vid' => $vacanteId, 'uid' => $postulanteUserId]);
        if ($st->fetchColumn()) {
            throw new ValidationException('Ya postulaste a esta vacante.');
        }

        try {
            $ins = $pdo->prepare("
                INSERT INTO postulaciones (vacante_id, postulante_user_id, mensaje, estado)
                VALUES (:vid, :uid, :msg, 'enviada')
            ");
            $ins->execute([
                'vid' => $vacanteId,
                'uid' => $postulanteUserId,
                'msg' => $msg,
            ]);
        } catch (\PDOException $e) {
            // Por si cae por uq_postulacion
            if ((string)$e->getCode() === '23000') {
                throw new ValidationException('Ya postulaste a esta vacante.');
            }
            throw $e;
        }
    }

    /** Listar postulaciones para una vacante (empresa) */
    public static function listForVacante(int $vacanteId): array
    {
        if ($vacanteId <= 0) return [];

        $pdo = DB::pdo();

        // Solo usamos columnas conocidas de postulaciones + email del user
        $st = $pdo->prepare("
            SELECT
              p.id,
              p.vacante_id,
              p.postulante_user_id,
              p.mensaje,
              p.estado,
              p.creado_en,
              p.actualizado_en,
              u.email AS postulante_email
            FROM postulaciones p
            LEFT JOIN users u ON u.id = p.postulante_user_id
            WHERE p.vacante_id = :vid
            ORDER BY p.id DESC
        ");
        $st->execute(['vid' => $vacanteId]);
        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    /** Cambiar estado (empresa) */
    public static function updateEstado(int $postulacionId, string $estado): void
    {
        $estado = trim($estado);
        $allowed = ['enviada', 'vista', 'preseleccion', 'rechazada'];
        if (!in_array($estado, $allowed, true)) {
            throw new ValidationException('Estado inválido.');
        }

        $pdo = DB::pdo();

        $st = $pdo->prepare("UPDATE postulaciones SET estado = :e WHERE id = :id");
        $st->execute(['e' => $estado, 'id' => $postulacionId]);

        if ($st->rowCount() === 0) {
            throw new ValidationException('Postulación no existe o no hubo cambios.');
        }
    }

    /** Buscar una postulación (empresa) */
    public static function find(int $id): ?array
    {
        if ($id <= 0) return null;

        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT id, vacante_id, postulante_user_id, mensaje, estado, creado_en, actualizado_en
            FROM postulaciones
            WHERE id = :id
            LIMIT 1
        ");
        $st->execute(['id' => $id]);
        $row = $st->fetch(\PDO::FETCH_ASSOC);
        return $row ?: null;
    }

        /** Listar postulaciones de un postulante */
    public static function listForPostulante(int $postulanteUserId): array
    {
        if ($postulanteUserId <= 0) return [];

        $pdo = DB::pdo();

        $st = $pdo->prepare("
            SELECT
              p.id,
              p.vacante_id,
              p.postulante_user_id,
              p.mensaje,
              p.estado AS postulacion_estado,
              p.creado_en AS postulacion_creado_en,

              v.titulo,
              v.ubicacion,
              v.modalidad,
              v.tipo_empleo,
              v.salario_min,
              v.salario_max,
              v.moneda,
              v.estado AS vacante_estado,
              v.publicada_en,

              u.email AS empresa_email
            FROM postulaciones p
            JOIN vacantes v ON v.id = p.vacante_id
            LEFT JOIN users u ON u.id = v.empresa_user_id
            WHERE p.postulante_user_id = :uid
            ORDER BY p.id DESC
        ");
        $st->execute(['uid' => $postulanteUserId]);

        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }
}
