<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/DB/DB.php';

final class VacantePublicService
{
    /**
     * Listado público.
     * Si $viewerUserId > 0 (postulante logueado) calcula match por categorías y ordena patrocinadas primero.
     */
    public static function listPublicadas(int $viewerUserId = 0, int $limit = 60, int $offset = 0): array
    {
        $pdo = DB::pdo();

        $limit  = max(1, min(200, $limit));
        $offset = max(0, $offset);

        $sql = "
            SELECT
              v.id,
              v.slug,
              v.empresa_user_id,
              v.titulo,
              v.descripcion,
              v.ubicacion,
              v.modalidad,
              v.tipo_empleo,
              v.salario_min,
              v.salario_max,
              v.moneda,
              v.estado,
              v.imagen_path,
              v.publicada_en,
              v.creado_en,

              u.nombre AS empresa_nombre,
              u.email  AS empresa_email,
              u.email_verificado AS empresa_verificada,
              ep.logo_path AS empresa_logo_path,
              ep.website_url,
              ep.linkedin_url,
              ep.whatsapp,

              /* rating aproximado por reacciones históricas de vacantes de la empresa */
              COALESCE((
                SELECT ROUND(
                  (
                    SUM(CASE WHEN r.reaccion IN ('like','recommend','excellent') THEN 1 ELSE 0 END)
                    / NULLIF(COUNT(*),0)
                  ) * 5
                ,1)
                FROM vacante_reacciones r
                INNER JOIN vacantes v2 ON v2.id = r.vacante_id
                WHERE v2.empresa_user_id = v.empresa_user_id
              ), 0) AS empresa_rating,

              GROUP_CONCAT(DISTINCT cl.nombre ORDER BY cl.nombre SEPARATOR ', ') AS categorias_vacante,
              GROUP_CONCAT(
                DISTINCT CASE WHEN uc.categoria_id IS NOT NULL THEN cl.nombre ELSE NULL END
                ORDER BY cl.nombre SEPARATOR ', '
              ) AS categorias_match,

              COUNT(DISTINCT uc.categoria_id) AS match_count,

              MAX(COALESCE(fb.score_boost, 0)) AS boost_max,
              (CASE WHEN MAX(COALESCE(fb.score_boost, 0)) > 0 THEN 1 ELSE 0 END) AS is_ideal,

              COALESCE(v.publicada_en, v.creado_en) AS fecha_feed

            FROM vacantes v
            LEFT JOIN users u
              ON u.id = v.empresa_user_id
            LEFT JOIN empresa_perfiles ep
              ON ep.user_id = v.empresa_user_id

            LEFT JOIN vacante_categorias vc
              ON vc.vacante_id = v.id
            LEFT JOIN categorias_laborales cl
              ON cl.id = vc.categoria_id

            LEFT JOIN user_categorias uc
              ON uc.user_id = :uid
             AND uc.categoria_id = vc.categoria_id

            LEFT JOIN feed_boosts fb
              ON fb.actor_user_id = v.empresa_user_id
             AND fb.actor_tipo = 'empresa'
             AND fb.estado = 'activo'
             AND NOW() BETWEEN fb.starts_at AND fb.ends_at

            WHERE v.estado = 'publicada'

            GROUP BY v.id
            ORDER BY is_ideal DESC, match_count DESC, fecha_feed DESC
            LIMIT {$limit} OFFSET {$offset}
        ";

        $st = $pdo->prepare($sql);
        $st->execute([':uid' => $viewerUserId]);
        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * Detalle público por ID.
     * $viewerUserId sirve para mostrar match por categorías y etiqueta “Match para ti”.
     */
    public static function findPublicada(int $id, int $viewerUserId = 0): ?array
    {
        if ($id <= 0) return null;

        $pdo = DB::pdo();

        $sql = "
            SELECT
              v.id,
              v.slug,
              v.empresa_user_id,
              v.titulo,
              v.descripcion,
              v.ubicacion,
              v.modalidad,
              v.tipo_empleo,
              v.salario_min,
              v.salario_max,
              v.moneda,
              v.estado,
              v.imagen_path,
              v.publicada_en,
              v.creado_en,

              u.nombre AS empresa_nombre,
              u.email  AS empresa_email,
              u.email_verificado AS empresa_verificada,
              ep.logo_path AS empresa_logo_path,
              ep.website_url,
              ep.linkedin_url,
              ep.whatsapp,

              COALESCE((
                SELECT ROUND(
                  (
                    SUM(CASE WHEN r.reaccion IN ('like','recommend','excellent') THEN 1 ELSE 0 END)
                    / NULLIF(COUNT(*),0)
                  ) * 5
                ,1)
                FROM vacante_reacciones r
                INNER JOIN vacantes v2 ON v2.id = r.vacante_id
                WHERE v2.empresa_user_id = v.empresa_user_id
              ), 0) AS empresa_rating,

              GROUP_CONCAT(DISTINCT cl.nombre ORDER BY cl.nombre SEPARATOR ', ') AS categorias_vacante,
              GROUP_CONCAT(
                DISTINCT CASE WHEN uc.categoria_id IS NOT NULL THEN cl.nombre ELSE NULL END
                ORDER BY cl.nombre SEPARATOR ', '
              ) AS categorias_match,
              COUNT(DISTINCT uc.categoria_id) AS match_count,

              MAX(COALESCE(fb.score_boost, 0)) AS boost_max,
              (CASE WHEN MAX(COALESCE(fb.score_boost, 0)) > 0 THEN 1 ELSE 0 END) AS is_ideal,

              COALESCE(v.publicada_en, v.creado_en) AS fecha_feed

            FROM vacantes v
            LEFT JOIN users u
              ON u.id = v.empresa_user_id
            LEFT JOIN empresa_perfiles ep
              ON ep.user_id = v.empresa_user_id

            LEFT JOIN vacante_categorias vc
              ON vc.vacante_id = v.id
            LEFT JOIN categorias_laborales cl
              ON cl.id = vc.categoria_id

            LEFT JOIN user_categorias uc
              ON uc.user_id = :uid
             AND uc.categoria_id = vc.categoria_id

            LEFT JOIN feed_boosts fb
              ON fb.actor_user_id = v.empresa_user_id
             AND fb.actor_tipo = 'empresa'
             AND fb.estado = 'activo'
             AND NOW() BETWEEN fb.starts_at AND fb.ends_at

            WHERE v.id = :id
              AND v.estado = 'publicada'

            GROUP BY v.id
            LIMIT 1
        ";

        $st = $pdo->prepare($sql);
        $st->execute([
            ':id'  => $id,
            ':uid' => $viewerUserId,
        ]);

        $row = $st->fetch(\PDO::FETCH_ASSOC);
        return $row ?: null;
    }
}
