<?php
declare(strict_types=1);

final class VacanteSocialService
{
    private const ALLOWED = ['like','recommend','excellent','bad'];

    public static function setReaction(int $userId, int $vacanteId, string $reaction): void
    {
        if ($userId <= 0 || $vacanteId <= 0) return;
        if (!in_array($reaction, self::ALLOWED, true)) return;

        $pdo = DB::pdo();

        // Upsert por (vacante_id,user_id) => permite 1 reacción por usuario POR VACANTE.
        $sql = "
            INSERT INTO vacante_reacciones (vacante_id, user_id, reaction, creado_en, actualizado_en)
            VALUES (:vid, :uid, :reac, NOW(), NOW())
            ON DUPLICATE KEY UPDATE
                reaction = VALUES(reaction),
                actualizado_en = NOW()
        ";

        $st = $pdo->prepare($sql);
        $st->execute([
            ':vid'  => $vacanteId,
            ':uid'  => $userId,
            ':reac' => $reaction,
        ]);
    }

    public static function clearReaction(int $userId, int $vacanteId): void
    {
        if ($userId <= 0 || $vacanteId <= 0) return;

        $pdo = DB::pdo();
        $st = $pdo->prepare("DELETE FROM vacante_reacciones WHERE vacante_id = :vid AND user_id = :uid");
        $st->execute([':vid' => $vacanteId, ':uid' => $userId]);
    }

    public static function myReaction(int $userId, int $vacanteId): string
    {
        if ($userId <= 0 || $vacanteId <= 0) return '';

        $pdo = DB::pdo();
        $st = $pdo->prepare("SELECT reaction FROM vacante_reacciones WHERE vacante_id = :vid AND user_id = :uid LIMIT 1");
        $st->execute([':vid' => $vacanteId, ':uid' => $userId]);
        $r = $st->fetchColumn();
        return is_string($r) ? $r : '';
    }

    public static function share(int $userId, int $vacanteId): void
    {
        if ($userId <= 0 || $vacanteId <= 0) return;

        $pdo = DB::pdo();
        $st = $pdo->prepare("INSERT INTO vacante_shares (vacante_id, user_id, creado_en) VALUES (:vid, :uid, NOW())");
        $st->execute([':vid' => $vacanteId, ':uid' => $userId]);
    }

    public static function counts(int $vacanteId): array
    {
        if ($vacanteId <= 0) {
            return ['like'=>0,'recommend'=>0,'excellent'=>0,'bad'=>0,'share'=>0];
        }

        $pdo = DB::pdo();

        $st1 = $pdo->prepare("
            SELECT
              SUM(reaction='like')      AS c_like,
              SUM(reaction='recommend') AS c_recommend,
              SUM(reaction='excellent') AS c_excellent,
              SUM(reaction='bad')       AS c_bad
            FROM vacante_reacciones
            WHERE vacante_id = :vid
        ");
        $st1->execute([':vid' => $vacanteId]);
        $a = $st1->fetch(PDO::FETCH_ASSOC) ?: [];

        $st2 = $pdo->prepare("SELECT COUNT(*) FROM vacante_shares WHERE vacante_id = :vid");
        $st2->execute([':vid' => $vacanteId]);
        $shares = (int)$st2->fetchColumn();

        return [
            'like'      => (int)($a['c_like'] ?? 0),
            'recommend' => (int)($a['c_recommend'] ?? 0),
            'excellent' => (int)($a['c_excellent'] ?? 0),
            'bad'       => (int)($a['c_bad'] ?? 0),
            'share'     => $shares,
        ];
    }
    
    public static function getMyReaction(int $userId, int $vacanteId): string
{
    $pdo = DB::pdo();
    $st = $pdo->prepare("SELECT reaccion FROM vacante_reacciones WHERE vacante_id=:v AND user_id=:u LIMIT 1");
    $st->execute([':v'=>$vacanteId, ':u'=>$userId]);
    $r = $st->fetchColumn();
    return is_string($r) ? $r : '';
}

public static function getCounts(int $vacanteId): array
{
    $pdo = DB::pdo();

    $st = $pdo->prepare("
        SELECT
            SUM(reaccion='like') AS `like`,
            SUM(reaccion='recommend') AS `recommend`,
            SUM(reaccion='excellent') AS `excellent`,
            SUM(reaccion='bad') AS `bad`
        FROM vacante_reacciones
        WHERE vacante_id = :v
    ");
    $st->execute([':v' => $vacanteId]);
    $row = $st->fetch(PDO::FETCH_ASSOC) ?: [];

    $st2 = $pdo->prepare("SELECT COUNT(*) FROM vacante_compartidos WHERE vacante_id=:v");
    $st2->execute([':v' => $vacanteId]);
    $share = (int)$st2->fetchColumn();

    return [
        'like' => (int)($row['like'] ?? 0),
        'recommend' => (int)($row['recommend'] ?? 0),
        'excellent' => (int)($row['excellent'] ?? 0),
        'bad' => (int)($row['bad'] ?? 0),
        'share' => $share,
    ];
}

}
