<?php
declare(strict_types=1);

final class CvService
{
    public static function latestActiveForUser(int $userId): ?array
    {
        $pdo = DB::pdo();
        $stmt = $pdo->prepare("SELECT * FROM postulante_cvs WHERE user_id = :uid AND estado='activo' ORDER BY id DESC LIMIT 1");
        $stmt->execute(['uid' => $userId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public static function upload(int $userId, array $file, array $config): void
    {
        if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
            throw new ValidationException('Archivo inválido.');
        }

        $maxBytes = 5 * 1024 * 1024; // 5MB (ajustable después por plan)
        if (($file['size'] ?? 0) > $maxBytes) {
            throw new ValidationException('El CV excede 5MB.');
        }

        $orig = (string)($file['name'] ?? 'cv');
        $mime = (string)($file['type'] ?? 'application/octet-stream');

        // Validación por extensión + mime (simple, sin frameworks)
        $ext = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
        $allowedExt  = ['pdf','doc','docx'];
        $allowedMime = [
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        ];

        if (!in_array($ext, $allowedExt, true)) {
            throw new ValidationException('Formato no permitido. Solo PDF/DOC/DOCX.');
        }
        if (!in_array($mime, $allowedMime, true)) {
            // algunos servers reportan mime genérico; no bloqueamos duro si la extensión es válida
            $mime = 'application/octet-stream';
        }

        $sha = hash_file('sha256', $file['tmp_name']);
        if (!is_string($sha) || $sha === '') {
            throw new ValidationException('No se pudo validar el archivo.');
        }

        $root = (string)($config['paths']['root'] ?? dirname(__DIR__, 2));
        $dir  = $root . '/storage/uploads/cv/' . $userId;
        if (!is_dir($dir)) {
            @mkdir($dir, 0775, true);
        }

        $storedName = 'cv_' . date('Ymd_His') . '_' . bin2hex(random_bytes(6)) . '.' . $ext;
        $storedPath = $dir . '/' . $storedName;

        if (!move_uploaded_file($file['tmp_name'], $storedPath)) {
            throw new ValidationException('No se pudo guardar el archivo.');
        }

        $pdo = DB::pdo();
        $pdo->beginTransaction();

        // Solo un CV activo: marcamos cualquier activo previo como eliminado
        $pdo->prepare("UPDATE postulante_cvs SET estado='eliminado', eliminado_en=NOW() WHERE user_id=:uid AND estado='activo'")
            ->execute(['uid' => $userId]);

        $pdo->prepare("INSERT INTO postulante_cvs (user_id, original_name, stored_name, stored_path, mime, size_bytes, sha256)
                       VALUES (:uid, :o, :sn, :sp, :m, :sz, :sha)")
            ->execute([
                'uid' => $userId,
                'o'   => $orig,
                'sn'  => $storedName,
                'sp'  => $storedPath,
                'm'   => $mime,
                'sz'  => (int)($file['size'] ?? 0),
                'sha' => $sha,
            ]);

        $pdo->commit();
    }

    public static function deleteActive(int $userId): void
    {
        $pdo = DB::pdo();
        $pdo->prepare("UPDATE postulante_cvs SET estado='eliminado', eliminado_en=NOW() WHERE user_id=:uid AND estado='activo'")
            ->execute(['uid' => $userId]);
    }

    public static function sendFile(string $absPath, string $downloadName, string $mime): Response
    {
        if (!is_file($absPath) || !is_readable($absPath)) {
            return Response::html("<h3>Archivo no disponible</h3>", 404);
        }

        $data = file_get_contents($absPath);
        if ($data === false) {
            return Response::html("<h3>No se pudo leer el archivo</h3>", 500);
        }

        $res = new Response($data, 200, $mime !== '' ? $mime : 'application/octet-stream');
        $res->header('Content-Disposition', 'attachment; filename="' . str_replace('"', '', $downloadName) . '"');
        $res->header('X-Content-Type-Options', 'nosniff');
        return $res;
    }
}
