<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/DB/DB.php';

final class RrhhProfileService
{
    public static function ensure(int $userId): void
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("SELECT user_id FROM rrhh_profiles WHERE user_id=:id LIMIT 1");
        $st->execute([':id'=>$userId]);
        if ($st->fetch(\PDO::FETCH_ASSOC)) return;

        $ins = $pdo->prepare("INSERT INTO rrhh_profiles (user_id, estado, creado_en) VALUES (:id,'activo',NOW())");
        $ins->execute([':id'=>$userId]);
    }

    public static function get(int $userId): array
    {
        self::ensure($userId);

        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT rp.*, u.email, u.nombre, u.tipo, u.estado AS user_estado
            FROM rrhh_profiles rp
            INNER JOIN users u ON u.id = rp.user_id
            WHERE rp.user_id=:id
            LIMIT 1
        ");
        $st->execute([':id'=>$userId]);
        return $st->fetch(\PDO::FETCH_ASSOC) ?: [];
    }

    public static function update(int $userId, array $in): void
    {
        self::ensure($userId);

        $display = trim((string)($in['display_name'] ?? ''));
        $titular = trim((string)($in['titular'] ?? ''));
        $bio     = trim((string)($in['bio'] ?? ''));
        $ubi     = trim((string)($in['ubicacion'] ?? ''));
        $tel     = trim((string)($in['telefono'] ?? ''));
        $web     = trim((string)($in['website'] ?? ''));
        $lnk     = trim((string)($in['linkedin'] ?? ''));

        $empresaUserId = (int)($in['empresa_user_id'] ?? 0);
        if ($empresaUserId <= 0) $empresaUserId = null;

        $slug = trim((string)($in['slug'] ?? ''));
        if ($slug === '') {
            $base = $display !== '' ? $display : 'rrhh-' . $userId;
            $slug = self::slugify($base);
        } else {
            $slug = self::slugify($slug);
        }

        $pdo = DB::pdo();

        // slug único
        $stU = $pdo->prepare("
            SELECT user_id FROM rrhh_profiles
            WHERE slug = :slug AND user_id <> :uid
            LIMIT 1
        ");
        $stU->execute([':slug'=>$slug, ':uid'=>$userId]);
        if ($stU->fetch(\PDO::FETCH_ASSOC)) {
            $slug = $slug . '-' . $userId;
        }

        $st = $pdo->prepare("
            UPDATE rrhh_profiles
            SET empresa_user_id=:eid,
                slug=:slug,
                display_name=:display,
                titular=:titular,
                bio=:bio,
                ubicacion=:ubi,
                telefono=:tel,
                website=:web,
                linkedin=:lnk
            WHERE user_id=:uid
            LIMIT 1
        ");
        $st->bindValue(':eid', $empresaUserId, $empresaUserId === null ? \PDO::PARAM_NULL : \PDO::PARAM_INT);
        $st->bindValue(':slug', $slug);
        $st->bindValue(':display', $display !== '' ? $display : null, $display !== '' ? \PDO::PARAM_STR : \PDO::PARAM_NULL);
        $st->bindValue(':titular', $titular !== '' ? $titular : null, $titular !== '' ? \PDO::PARAM_STR : \PDO::PARAM_NULL);
        $st->bindValue(':bio', $bio !== '' ? $bio : null, $bio !== '' ? \PDO::PARAM_STR : \PDO::PARAM_NULL);
        $st->bindValue(':ubi', $ubi !== '' ? $ubi : null, $ubi !== '' ? \PDO::PARAM_STR : \PDO::PARAM_NULL);
        $st->bindValue(':tel', $tel !== '' ? $tel : null, $tel !== '' ? \PDO::PARAM_STR : \PDO::PARAM_NULL);
        $st->bindValue(':web', $web !== '' ? $web : null, $web !== '' ? \PDO::PARAM_STR : \PDO::PARAM_NULL);
        $st->bindValue(':lnk', $lnk !== '' ? $lnk : null, $lnk !== '' ? \PDO::PARAM_STR : \PDO::PARAM_NULL);
        $st->bindValue(':uid', $userId, \PDO::PARAM_INT);
        $st->execute();
    }

    public static function setPhoto(int $userId, ?string $path): void
    {
        self::ensure($userId);
        $pdo = DB::pdo();
        $st = $pdo->prepare("UPDATE rrhh_profiles SET foto_path=:p WHERE user_id=:uid LIMIT 1");
        $st->bindValue(':p', $path, $path === null ? \PDO::PARAM_NULL : \PDO::PARAM_STR);
        $st->bindValue(':uid', $userId, \PDO::PARAM_INT);
        $st->execute();
    }

    public static function getLinkedEmpresaId(int $rrhhUserId): ?int
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("SELECT empresa_user_id FROM rrhh_profiles WHERE user_id=:uid LIMIT 1");
        $st->execute([':uid'=>$rrhhUserId]);
        $r = $st->fetch(\PDO::FETCH_ASSOC);
        $eid = (int)($r['empresa_user_id'] ?? 0);
        return $eid > 0 ? $eid : null;
    }

    /**
     * Perfil público (para /rrhh/ver?id=)
     * Devuelve estructura lista para template:
     * - nombre
     * - titulo
     * - bio (HTML permitido, saneado)
     * - skills[] (flat: categoria,nombre,porcentaje)
     * - rating_avg / rating_count
     */
    public static function publicById(int $rrhhUserId): array
    {
        $pdo = DB::pdo();

        $st = $pdo->prepare("
            SELECT rp.*, u.nombre, u.tipo, u.estado AS user_estado
            FROM rrhh_profiles rp
            INNER JOIN users u ON u.id = rp.user_id
            WHERE rp.user_id=:id
            LIMIT 1
        ");
        $st->execute([':id'=>$rrhhUserId]);
        $row = $st->fetch(\PDO::FETCH_ASSOC) ?: [];
        if (!$row) return [];

        // Rating
        $rating = ['avg'=>0,'count'=>0];
        if (class_exists('RrhhRatingService') && method_exists('RrhhRatingService', 'summary')) {
            $rating = (array)RrhhRatingService::summary($rrhhUserId);
        }

        // Skills agrupadas → flatten para template
        $skillsFlat = [];
        if (class_exists('RrhhSkillService') && method_exists('RrhhSkillService', 'listActiveGrouped')) {
            $grouped = (array)RrhhSkillService::listActiveGrouped($rrhhUserId);
            foreach ($grouped as $g) {
                $cat = (string)($g['categoria'] ?? '');
                $items = $g['items'] ?? [];
                if (!is_array($items)) $items = [];
                foreach ($items as $it) {
                    $skillsFlat[] = [
                        'categoria'   => $cat,
                        'nombre'      => (string)($it['nombre'] ?? ''),
                        'porcentaje'  => (int)($it['porcentaje'] ?? 0),
                    ];
                }
            }
        }

        // Bio: permitir HTML básico (sin XSS). Tu feed ya sanea anuncios; acá saneo bio.
        $bio = (string)($row['bio'] ?? '');
        if ($bio !== '') {
            $bio = preg_replace('~<(script|iframe|object|embed|style)\b[^>]*>.*?</\1>~is', '', $bio) ?? '';
            $allowed = '<p><br><b><strong><i><em><ul><ol><li><a><blockquote><code><pre>';
            $bio = strip_tags($bio, $allowed);
            $bio = preg_replace('~\son\w+\s*=\s*(".*?"|\'.*?\'|[^\s>]+)~i', '', $bio) ?? '';
            $bio = preg_replace('~\sstyle\s*=\s*(".*?"|\'.*?\')~i', '', $bio) ?? '';
        }

        return [
            // lo que tu template necesita
            'nombre'       => (string)($row['display_name'] ?? '') !== '' ? (string)$row['display_name'] : (string)($row['nombre'] ?? ''),
            'titular'      => (string)($row['titular'] ?? ''),
            'ubicacion'    => (string)($row['ubicacion'] ?? ''),
            'bio'          => $bio,
            'foto_path'    => (string)($row['foto_path'] ?? ''),

            'rating_avg'   => (float)($rating['avg'] ?? 0),
            'rating_count' => (int)($rating['count'] ?? 0),

            'skills'       => $skillsFlat,

            // extra útil por si lo ocupás luego
            'id'           => (int)$rrhhUserId,
            'slug'         => (string)($row['slug'] ?? ''),
        ];
    }

    private static function slugify(string $s): string
    {
        $s = mb_strtolower(trim($s), 'UTF-8');
        $s = preg_replace('~[^\pL\d]+~u', '-', $s) ?? $s;
        $s = trim($s, '-');
        $s = preg_replace('~[^-\w]+~', '', $s) ?? $s;
        $s = preg_replace('~-+~', '-', $s) ?? $s;
        return $s !== '' ? $s : 'rrhh';
    }
}
