<?php
/** @var array $empresa */   // users
/** @var array $perfil */    // empresa_perfiles
/** @var string $csrf */
/** @var string|null $ok */
/** @var string|null $error */

$bp = base_path();

$empresaId = (int)($empresa['id'] ?? 0);
$nombreActual = (string)($empresa['nombre'] ?? ($empresa['name'] ?? ''));

$telefono  = (string)($perfil['telefono'] ?? '');
$whatsapp  = (string)($perfil['whatsapp'] ?? '');
$ubicacion = (string)($perfil['ubicacion'] ?? '');
$bio       = (string)($perfil['bio'] ?? '');
if (mb_strlen($bio) > 500) $bio = mb_substr($bio, 0, 500);

$website        = (string)($perfil['website_url'] ?? '');
$googleBusiness = (string)($perfil['google_business_url'] ?? '');

// Paths desde DB (tu DB guarda /upload/... )
$logoPath   = (string)($perfil['logo_path'] ?? '');
$bannerPath = (string)($perfil['banner_path'] ?? '');

/**
 * En TU hosting, la URL pública correcta es:
 *   /public/upload/...
 * DB guarda:
 *   /upload/...
 *
 * Esta función devuelve SIEMPRE:
 *   /public/upload/...
 * (sin duplicar /public)
 */
function normalize_public_upload_path(string $p): string {
    $p = trim($p);
    if ($p === '') return '';

    $p = str_replace('\\', '/', $p);

    // quitar dominio si alguien guardó URL completa
    $p = preg_replace('~^https?://[^/]+~i', '', $p) ?? $p;

    // asegurar leading slash
    if (!str_starts_with($p, '/')) $p = '/' . $p;

    // si ya viene /public/public/... lo arreglamos
    while (str_starts_with($p, '/public/public/')) {
        $p = substr($p, 7); // quita un /public
    }

    // si ya viene correcto
    if (str_starts_with($p, '/public/upload/')) return $p;

    // si viene /upload/...
    if (str_starts_with($p, '/upload/')) return '/public' . $p;

    // si contiene /upload/ en otra posición, forzar prefijo /public
    if (str_contains($p, '/upload/')) {
        if (!str_starts_with($p, '/public/')) return '/public' . $p;
        return $p;
    }

    // si no es upload, no lo usamos
    return '';
}

$logoPublicPath   = normalize_public_upload_path($logoPath);
$bannerPublicPath = normalize_public_upload_path($bannerPath);

/**
 * URL final:
 * OJO: NO concatenar con $bp, porque $bp ya es "/public".
 * Aquí devolvemos URL absoluta desde raíz del dominio.
 */
$logoUrl   = $logoPublicPath   !== '' ? $logoPublicPath   : '';
$bannerUrl = $bannerPublicPath !== '' ? $bannerPublicPath : '';
?>

<style>
  .img-box{
    width:100%;
    border:1px dashed #cfd8ee;
    border-radius:14px;
    overflow:hidden;
    background:#fbfcff;
    position:relative;
  }
  .img-box.logo{
    max-width:260px;
    aspect-ratio: 1/1;
    display:flex;
    align-items:center;
    justify-content:center;
  }
  .img-box.banner{
    aspect-ratio: 16/5;
    display:flex;
    align-items:center;
    justify-content:center;
  }
  .img-box img{width:100%;height:100%;object-fit:cover;display:block}
  .img-empty{padding:14px;text-align:center;color:#7a88a6;font-size:13px}
  .hint{font-size:12px;color:#6b7a99}
  .counter{font-size:12px;color:#6b7a99}
  .small-actions{display:flex;gap:8px;flex-wrap:wrap;margin-top:8px}
</style>

<div class="container py-4">
  <div class="d-flex align-items-center justify-content-between mb-3">
    <div>
      <h3 class="mb-0">Perfil público (Empresa)</h3>
      <div class="text-muted small">Esto se mostrará en tu perfil público.</div>
    </div>

    <a class="btn btn-outline-secondary btn-sm"
       href="<?= e($bp . '/empresas/ver?id=' . $empresaId) ?>">
      Ver perfil público
    </a>
  </div>

  <?php if (!empty($ok)): ?>
    <div class="alert alert-success"><?= e((string)$ok) ?></div>
  <?php endif; ?>

  <?php if (!empty($error)): ?>
    <div class="alert alert-danger"><?= e((string)$error) ?></div>
  <?php endif; ?>

  <div class="card">
    <div class="card-body">
      <form method="post"
            action="<?= e($bp . '/empresa/perfil') ?>"
            enctype="multipart/form-data">
        <input type="hidden" name="_csrf" value="<?= e($csrf) ?>">

        <div class="row g-3">

          <div class="col-12">
            <label class="form-label">Nombre de la empresa</label>
            <input class="form-control" name="nombre"
                   value="<?= e($nombreActual) ?>"
                   placeholder="Nombre público de tu empresa" required>
            <div class="hint">Este campo se guarda en la tabla <strong>users</strong>.</div>
          </div>

          <!-- Logo -->
          <div class="col-12 col-lg-6">
            <label class="form-label">Logo (carga de imagen)</label>

            <div class="img-box logo mb-2">
              <?php if ($logoUrl !== ''): ?>
                <img id="logoPreview" src="<?= e($logoUrl) ?>" alt="Logo actual">
                <div id="logoEmpty" class="img-empty" style="display:none;">Sin logo</div>
              <?php else: ?>
                <img id="logoPreview" src="" alt="" style="display:none;">
                <div id="logoEmpty" class="img-empty">Sin logo</div>
              <?php endif; ?>
            </div>

            <input class="form-control" type="file" name="logo" accept="image/*" id="logoInput">
            <div class="hint">Se guarda en <code>/public/upload/empresas/logo/</code>.</div>

            <div class="small-actions">
              <button type="button" class="btn btn-sm btn-outline-secondary" id="logoClearBtn">Quitar selección</button>
            </div>
          </div>

          <!-- Banner -->
          <div class="col-12 col-lg-6">
            <label class="form-label">Banner (carga de imagen)</label>

            <div class="img-box banner mb-2">
              <?php if ($bannerUrl !== ''): ?>
                <img id="bannerPreview" src="<?= e($bannerUrl) ?>" alt="Banner actual">
                <div id="bannerEmpty" class="img-empty" style="display:none;">Sin banner</div>
              <?php else: ?>
                <img id="bannerPreview" src="" alt="" style="display:none;">
                <div id="bannerEmpty" class="img-empty">Sin banner</div>
              <?php endif; ?>
            </div>

            <input class="form-control" type="file" name="banner" accept="image/*" id="bannerInput">
            <div class="hint">Se guarda en <code>/public/upload/empresas/banner/</code>.</div>

            <div class="small-actions">
              <button type="button" class="btn btn-sm btn-outline-secondary" id="bannerClearBtn">Quitar selección</button>
            </div>
          </div>

          <div class="col-12 col-lg-6">
            <label class="form-label">Teléfono</label>
            <input class="form-control" name="telefono" value="<?= e($telefono) ?>" placeholder="Ej: +506 8xxx xxxx">
          </div>

          <div class="col-12 col-lg-6">
            <label class="form-label">WhatsApp</label>
            <input class="form-control" name="whatsapp" value="<?= e($whatsapp) ?>" placeholder="Ej: +506 8xxx xxxx">
          </div>

          <div class="col-12 col-lg-6">
            <label class="form-label">Ubicación</label>
            <input class="form-control" name="ubicacion" value="<?= e($ubicacion) ?>">
          </div>

          <div class="col-12 col-lg-6">
            <label class="form-label">Website (https://...)</label>
            <input class="form-control" name="website_url" value="<?= e($website) ?>">
          </div>

          <div class="col-12">
            <label class="form-label">Google Business (link)</label>
            <input class="form-control" name="google_business_url" value="<?= e($googleBusiness) ?>"
                   placeholder="https://g.page/tu-negocio o link de Google Maps/Business Profile">
          </div>

          <div class="col-12">
            <label class="form-label">Bio / Descripción (máx. 500)</label>
            <textarea class="form-control" rows="4" name="bio" id="bio"
                      maxlength="500"><?= e($bio) ?></textarea>
            <div class="d-flex justify-content-between mt-1">
              <div class="hint">Máximo 500 caracteres.</div>
              <div class="counter"><span id="bioCount"><?= (int)mb_strlen($bio) ?></span>/500</div>
            </div>
          </div>

        </div>

        <div class="mt-3 d-flex gap-2">
          <button class="btn btn-primary" type="submit">Guardar</button>
          <a class="btn btn-outline-secondary"
             href="<?= e($bp . '/empresas/ver?id=' . $empresaId) ?>">
            Ver perfil público
          </a>
        </div>
      </form>
    </div>
  </div>
</div>

<script>
(function(){
  const bio = document.getElementById('bio');
  const bioCount = document.getElementById('bioCount');
  if (bio && bioCount) bio.addEventListener('input', ()=> bioCount.textContent = String(bio.value.length));

  function bindPreview(inputId, imgId, emptyId, clearBtnId, initialSrc){
    const input = document.getElementById(inputId);
    const img = document.getElementById(imgId);
    const empty = document.getElementById(emptyId);
    const clearBtn = document.getElementById(clearBtnId);
    if (!input || !img) return;

    function showEmpty(){
      img.src = '';
      img.style.display = 'none';
      if (empty) empty.style.display = 'block';
    }
    function showImg(src){
      img.src = src;
      img.style.display = 'block';
      if (empty) empty.style.display = 'none';
    }

    if (initialSrc) showImg(initialSrc); else showEmpty();

    input.addEventListener('change', ()=>{
      const f = input.files && input.files[0];
      if (!f) return;
      const reader = new FileReader();
      reader.onload = (ev)=> showImg(String(ev.target.result || ''));
      reader.readAsDataURL(f);
    });

    if (clearBtn) {
      clearBtn.addEventListener('click', ()=>{
        input.value = '';
        if (initialSrc) showImg(initialSrc); else showEmpty();
      });
    }
  }

  bindPreview('logoInput', 'logoPreview', 'logoEmpty', 'logoClearBtn', <?= $logoUrl !== '' ? "'" . e($logoUrl) . "'" : "''" ?>);
  bindPreview('bannerInput', 'bannerPreview', 'bannerEmpty', 'bannerClearBtn', <?= $bannerUrl !== '' ? "'" . e($bannerUrl) . "'" : "''" ?>);
})();
</script>
