<?php
declare(strict_types=1);
/** @var string $pageTitle */
/** @var string $content */

$bp = base_path();

function badge_count(int $n): string {
  if ($n <= 0) return '';
  $txt = $n > 99 ? '99+' : (string)$n;
  return '<span class="badge-pill">'.$txt.'</span>';
}

/**
 * Obtener userId y tipo sin depender de métodos inexistentes.
 */
$userId = 0;
if (class_exists('AuthService') && method_exists('AuthService', 'currentUserId')) {
  try { $userId = (int)AuthService::currentUserId(); } catch (Throwable $e) { $userId = 0; }
}
if ($userId <= 0) {
  $userId = (int)($_SESSION['user_id'] ?? ($_SESSION['uid'] ?? 0));
}

$userTipo = (string)($_SESSION['user_tipo'] ?? ($_SESSION['tipo'] ?? ''));

if ($userId > 0 && $userTipo === '' && class_exists('DB')) {
  try {
    $pdo = DB::pdo();
    $st = $pdo->prepare("SELECT tipo FROM users WHERE id = :id LIMIT 1");
    $st->execute([':id' => $userId]);
    $userTipo = (string)($st->fetch(PDO::FETCH_ASSOC)['tipo'] ?? '');
  } catch (Throwable $e) {
    $userTipo = '';
  }
}

/**
 * Conteo de notificaciones no leídas (si existe tabla notifications).
 */
$notifCount = 0;
if ($userId > 0 && class_exists('NotificationService') && method_exists('NotificationService', 'countUnread')) {
  try { $notifCount = (int)NotificationService::countUnread($userId); } catch (Throwable $e) { $notifCount = 0; }
} elseif ($userId > 0 && class_exists('DB')) {
  try {
    $pdo = DB::pdo();
    $st = $pdo->prepare("SELECT COUNT(*) c FROM notifications WHERE user_id = :uid AND is_read = 0");
    $st->execute([':uid' => $userId]);
    $notifCount = (int)($st->fetch(PDO::FETCH_ASSOC)['c'] ?? 0);
  } catch (Throwable $e) {
    $notifCount = 0;
  }
}

/**
 * Rutas principales
 */
$feedUrl    = $bp . '/feed';
$searchUrl  = $bp . '/vacantes';
$publishUrl = $bp . '/feed';
$panelUrl   = $bp . '/feed';
$notifUrl   = $bp . '/feed';
$profileUrl = $bp . '/feed';

if ($userTipo === 'empresa') {
  $panelUrl   = $bp . '/empresa';
  $notifUrl   = $bp . '/empresa/notificaciones';
  $profileUrl = $bp . '/empresas/ver?id=' . $userId;

} elseif ($userTipo === 'postulante') {
  $panelUrl   = $bp . '/postulante/panel';
  $notifUrl   = $bp . '/postulante/notificaciones';
  $profileUrl = $bp . '/postulante/perfil';

} elseif ($userTipo === 'rrhh') {
  $panelUrl   = $bp . '/rrhh';
  $notifUrl   = $bp . '/rrhh/notificaciones';
  $profileUrl = $bp . '/rrhh/perfil';

} elseif ($userTipo === 'admin') {
  $panelUrl   = $bp . '/admin';
  $notifUrl   = $bp . '/admin';
  $profileUrl = $bp . '/admin';
}


/**
 * CSS/JS local (cache buster por filemtime)
 * Nota: base_path() YA incluye "/public", por lo tanto:
 * CSS => $bp . '/assets/css/app.css'
 * NO => $bp . '/public/assets/...'
 */
$publicDir = dirname(__DIR__, 3) . '/public'; // ajusta si tu layout está en otra ruta
$appCssAbs = $publicDir . '/assets/css/app.css';
$appCssVer = is_file($appCssAbs) ? (string)filemtime($appCssAbs) : (string)time();

// si tienes bootstrap local, puedes cambiarlo luego; por ahora dejas CDN como en tu código.
?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title><?= e((string)$pageTitle) ?></title>

  <link rel="icon" href="<?= e((string)$bp) ?>/assets/img/favicon.ico">

  <!-- Bootstrap -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">

  <!-- App CSS (PROYECTO) -->
  <link rel="stylesheet" href="<?= e((string)($bp . '/assets/css/app.css?v=' . $appCssVer)) ?>">

  <style>
    :root{
      --app-bg:#f6f8ff;
      --stroke:#e8eefc;
      --ink:#0b1220;
      --muted:#5b6b86;
      --primary:#245bff;
      --danger:#ff2d55;
    }
    body{ background:var(--app-bg); color:var(--ink); }
    .app-topbar{
      position:sticky; top:0; z-index:20;
      background:rgba(255,255,255,0.92);
      backdrop-filter: blur(8px);
      border-bottom:1px solid var(--stroke);
    }
    .app-shell{ padding-top: 14px; padding-bottom: 92px; }
    .app-muted{ color:var(--muted); }
    .top-actions{ display:flex; gap:10px; align-items:center; }

    .icon-link{
      text-decoration:none;
      color:var(--muted);
      display:flex; flex-direction:column;
      align-items:center; justify-content:center;
      gap:4px;
      padding:6px 8px;
      border-radius:12px;
      border:1px solid transparent;
      min-width:64px;
    }
    .icon-link:hover{
      background:#f1f5ff;
      border-color:var(--stroke);
      color:var(--ink);
    }
    .icon-link .ico{
      width:40px;height:40px;border-radius:14px;
      background:#f1f5ff;border:1px solid var(--stroke);
      display:flex;align-items:center;justify-content:center;
      color:var(--primary);
      font-weight:800;
      position:relative;
    }
    .icon-link small{
      font-size:12px;
      line-height:1;
      color:var(--muted);
      white-space:nowrap;
    }
    .badge-pill{
      position:absolute;
      top:-6px;
      right:-6px;
      background:var(--danger);
      color:#fff;
      border-radius:999px;
      padding:2px 6px;
      font-size:11px;
      font-weight:800;
      border:2px solid #fff;
      min-width:22px;
      text-align:center;
    }

    .bottom-nav{
      position:fixed; bottom:0; left:0; right:0; z-index:30;
      background:rgba(255,255,255,0.96);
      backdrop-filter: blur(8px);
      border-top:1px solid var(--stroke);
      padding:10px 14px;
      display:none;
    }
    .bottom-nav a{
      text-decoration:none;
      color:var(--muted);
      font-size:12px;
      display:flex;
      flex-direction:column;
      align-items:center;
      gap:4px;
      min-width:64px;
    }
    .bottom-nav .dot{
      width:44px;height:44px;border-radius:14px;
      background:#f1f5ff;border:1px solid var(--stroke);
      display:flex;align-items:center;justify-content:center;
      color:var(--primary);
      font-weight:800;
      position:relative;
    }
    .bottom-nav a:hover{ color:var(--ink); }

    @media (max-width: 992px){
      .top-actions{ display:none; }
      .bottom-nav{ display:flex; justify-content:space-between; overflow-x:auto; gap:6px; }
    }
  </style>
</head>
<body>

  <div class="app-topbar">
    <div class="container py-3 d-flex align-items-center justify-content-between">
      <div class="d-flex align-items-center gap-2">
        <a href="<?= e((string)$feedUrl) ?>" class="rounded-3 px-3 py-2"
           style="text-decoration:none;background:#eef3ff;border:1px solid var(--stroke);font-weight:800;color:var(--primary);">
          JOB
        </a>
        <div>
          <div class="fw-semibold" style="line-height:1.1;">Red Laboral</div>
          <div class="small app-muted">Feed seguro y rápido</div>
        </div>
      </div>

      <div class="top-actions">
        <a class="icon-link" href="<?= e((string)$feedUrl) ?>" title="Feed">
          <span class="ico">🏠</span>
          <small>Feed</small>
        </a>

        <a class="icon-link" href="<?= e((string)$searchUrl) ?>" title="Buscar">
          <span class="ico">🔎</span>
          <small>Buscar</small>
        </a>

        <a class="icon-link" href="<?= e((string)$publishUrl) ?>" title="Publicar">
          <span class="ico">＋</span>
          <small>Publicar</small>
        </a>

        <a class="icon-link" href="<?= e((string)$notifUrl) ?>" title="Notificaciones">
          <span class="ico">🔔<?= badge_count($notifCount) ?></span>
          <small>Alertas</small>
        </a>

        <a class="icon-link" href="<?= e((string)$profileUrl) ?>" title="Perfil">
          <span class="ico">👤</span>
          <small>Perfil</small>
        </a>

        <a class="icon-link" href="<?= e((string)$panelUrl) ?>" title="Panel">
          <span class="ico">📊</span>
          <small>Panel</small>
        </a>

        <a class="btn btn-outline-primary btn-sm" href="<?= e((string)$bp) ?>/logout">Salir</a>
      </div>

      <div class="d-flex d-lg-none">
        <a class="btn btn-outline-primary btn-sm" href="<?= e((string)$bp) ?>/logout">Salir</a>
      </div>
    </div>
  </div>

  <main class="app-shell">
    <?= $content ?>
  </main>

  <nav class="bottom-nav">
    <a href="<?= e((string)$feedUrl) ?>" title="Feed">
      <span class="dot">🏠</span>
      <span>Feed</span>
    </a>

    <a href="<?= e((string)$searchUrl) ?>" title="Buscar">
      <span class="dot">🔎</span>
      <span>Buscar</span>
    </a>

    <a href="<?= e((string)$publishUrl) ?>" title="Publicar">
      <span class="dot">＋</span>
      <span>Publicar</span>
    </a>

    <a href="<?= e((string)$notifUrl) ?>" title="Alertas">
      <span class="dot">🔔<?= badge_count($notifCount) ?></span>
      <span>Alertas</span>
    </a>

    <a href="<?= e((string)$profileUrl) ?>" title="Perfil">
      <span class="dot">👤</span>
      <span>Perfil</span>
    </a>

    <a href="<?= e((string)$panelUrl) ?>" title="Panel">
      <span class="dot">📊</span>
      <span>Panel</span>
    </a>
  </nav>

  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
