<?php
declare(strict_types=1);

final class AdminRoleService
{
    public static function listAll(): array
    {
        $pdo = DB::pdo();

        $sql = "
            SELECT
                id,
                code,
                nombre,
                descripcion,
                estado,
                creado_en
            FROM roles
            ORDER BY id DESC
        ";

        return $pdo->query($sql)->fetchAll();
    }

    public static function find(int $id): ?array
    {
        $pdo = DB::pdo();

        $st = $pdo->prepare("
            SELECT
                id,
                code,
                nombre,
                descripcion,
                estado,
                creado_en
            FROM roles
            WHERE id = :id
            LIMIT 1
        ");
        $st->execute([':id' => $id]);

        $row = $st->fetch();
        return $row ?: null;
    }

    public static function create(string $code, string $nombre, ?string $descripcion, string $estado): void
    {
        $code = trim($code);
        $nombre = trim($nombre);
        $descripcion = $descripcion !== null ? trim($descripcion) : null;

        if ($code === '' || $nombre === '') {
            throw new ValidationException('Código y nombre son obligatorios.');
        }

        // Convención conservadora: code tipo "admin", "super_admin", "empresa_hr"
        if (!preg_match('/^[a-z0-9_]{2,60}$/', $code)) {
            throw new ValidationException('El code debe ser minúsculas/números/underscore (2 a 60 caracteres).');
        }

        if (!in_array($estado, ['activo', 'inactivo'], true)) {
            throw new ValidationException('Estado inválido.');
        }

        $pdo = DB::pdo();

        // Evitar duplicados de code (aunque no haya UNIQUE en el dump, lo tratamos como regla de dominio)
        $st = $pdo->prepare("SELECT id FROM roles WHERE code = :c LIMIT 1");
        $st->execute([':c' => $code]);
        if ($st->fetch()) {
            throw new ValidationException('Ya existe un rol con ese code.');
        }

        $st = $pdo->prepare("
            INSERT INTO roles (code, nombre, descripcion, estado)
            VALUES (:c, :n, :d, :e)
        ");
        $st->execute([
            ':c' => $code,
            ':n' => $nombre,
            ':d' => ($descripcion === '' ? null : $descripcion),
            ':e' => $estado,
        ]);
    }

    public static function update(int $id, string $code, string $nombre, ?string $descripcion, string $estado): void
    {
        $code = trim($code);
        $nombre = trim($nombre);
        $descripcion = $descripcion !== null ? trim($descripcion) : null;

        if ($id <= 0) {
            throw new ValidationException('ID inválido.');
        }

        if ($code === '' || $nombre === '') {
            throw new ValidationException('Código y nombre son obligatorios.');
        }

        if (!preg_match('/^[a-z0-9_]{2,60}$/', $code)) {
            throw new ValidationException('El code debe ser minúsculas/números/underscore (2 a 60 caracteres).');
        }

        if (!in_array($estado, ['activo', 'inactivo'], true)) {
            throw new ValidationException('Estado inválido.');
        }

        $pdo = DB::pdo();

        // code único (distinto al propio)
        $st = $pdo->prepare("SELECT id FROM roles WHERE code = :c AND id <> :id LIMIT 1");
        $st->execute([':c' => $code, ':id' => $id]);
        if ($st->fetch()) {
            throw new ValidationException('Ya existe otro rol con ese code.');
        }

        $st = $pdo->prepare("
            UPDATE roles
            SET code = :c,
                nombre = :n,
                descripcion = :d,
                estado = :e
            WHERE id = :id
        ");
        $st->execute([
            ':id' => $id,
            ':c'  => $code,
            ':n'  => $nombre,
            ':d'  => ($descripcion === '' ? null : $descripcion),
            ':e'  => $estado,
        ]);
    }

    public static function delete(int $id): void
    {
        $pdo = DB::pdo();

        $st = $pdo->prepare("SELECT code FROM roles WHERE id = :id LIMIT 1");
        $st->execute([':id' => $id]);
        $row = $st->fetch();

        if (!$row) {
            throw new ValidationException('El rol no existe.');
        }

        // protección mínima (por seed)
        if (($row['code'] ?? '') === 'super_admin') {
            throw new ValidationException('No se puede eliminar el rol super_admin.');
        }

        // Si hay usuarios asignados, no permitir borrar (limpieza explícita primero)
        $st = $pdo->prepare("SELECT 1 FROM user_roles WHERE role_id = :id LIMIT 1");
        $st->execute([':id' => $id]);
        if ($st->fetch()) {
            throw new ValidationException('No se puede eliminar: existen usuarios asignados a este rol.');
        }

        $st = $pdo->prepare("DELETE FROM roles WHERE id = :id");
        $st->execute([':id' => $id]);
    }
}
