<?php
declare(strict_types=1);

final class AdminUserRoleService
{
    public static function listUsers(string $q = ''): array
    {
        $pdo = DB::pdo();

        $sql = "
            SELECT
                id,
                email,
                nombre,
                tipo,
                plan_id,
                estado,
                email_verificado,
                creado_en
            FROM users
        ";

        $params = [];
        if ($q !== '') {
            $sql .= " WHERE email LIKE :q OR nombre LIKE :q ";
            $params[':q'] = '%' . $q . '%';
        }

        $sql .= " ORDER BY id DESC";

        $st = $pdo->prepare($sql);
        $st->execute($params);

        return $st->fetchAll();
    }

    public static function findUser(int $id): ?array
    {
        $pdo = DB::pdo();

        $st = $pdo->prepare("
            SELECT
                id,
                email,
                nombre,
                tipo,
                plan_id,
                estado,
                email_verificado,
                creado_en
            FROM users
            WHERE id = :id
            LIMIT 1
        ");
        $st->execute([':id' => $id]);

        $row = $st->fetch();
        return $row ?: null;
    }

    public static function listRolesForUser(int $userId): array
    {
        $pdo = DB::pdo();

        $st = $pdo->prepare("
            SELECT
                r.id,
                r.code,
                r.nombre,
                r.descripcion,
                r.estado,
                CASE WHEN ur.role_id IS NULL THEN 0 ELSE 1 END AS assigned
            FROM roles r
            LEFT JOIN user_roles ur
                   ON ur.role_id = r.id
                  AND ur.user_id = :uid
            ORDER BY r.code ASC
        ");
        $st->execute([':uid' => $userId]);

        return $st->fetchAll();
    }

    /**
     * Estrategia: set completo:
     * - DELETE user_roles por user_id
     * - INSERT solo roles seleccionados (assigned_at default current_timestamp())
     *
     * user_roles tiene assigned_at con DEFAULT current_timestamp(). :contentReference[oaicite:4]{index=4}
     */
    public static function saveUserRoles(int $userId, array $roleIds): void
    {
        if ($userId <= 0) {
            throw new ValidationException('user_id inválido.');
        }

        // Normalizar role_ids
        $ids = [];
        foreach ($roleIds as $rid) {
            $rid = (int)$rid;
            if ($rid > 0) $ids[$rid] = true;
        }
        $ids = array_keys($ids);

        $pdo = DB::pdo();
        $pdo->beginTransaction();

        try {
            // Protección mínima: no permitir quitar super_admin al usuario seed (user_id=1, role_id=1)
            // (en tu dump ya existe esa asignación). :contentReference[oaicite:5]{index=5}
            if ($userId === 1) {
                if (!in_array(1, $ids, true)) {
                    throw new ValidationException('No se puede quitar el rol super_admin al usuario #1.');
                }
            }

            $del = $pdo->prepare("DELETE FROM user_roles WHERE user_id = :uid");
            $del->execute([':uid' => $userId]);

            if (!empty($ids)) {
                $ins = $pdo->prepare("
                    INSERT INTO user_roles (user_id, role_id)
                    VALUES (:uid, :rid)
                ");
                foreach ($ids as $rid) {
                    $ins->execute([':uid' => $userId, ':rid' => $rid]);
                }
            }

            $pdo->commit();
        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }
}
