<?php
declare(strict_types=1);

final class AuthService
{
    public static function register(string $email, string $password, string $tipo = 'postulante'): array
{
    $email = strtolower(trim($email));
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new ValidationException('Email inválido.');
    }

    $allowed = ['postulante', 'empresa', 'rrhh', 'orador', 'profesional'];
    if (!in_array($tipo, $allowed, true)) {
        throw new ValidationException('Tipo inválido.');
    }

    $pwErrors = Passwords::validatePolicy($password);
    if ($pwErrors) {
        throw new ValidationException(implode(' ', $pwErrors));
    }

    $pdo = DB::pdo();

    // evitar enumeración: mismo mensaje si existe
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = :e LIMIT 1");
    $stmt->execute([':e' => $email]);
    if ($stmt->fetch()) {
        throw new ValidationException('No se pudo completar el registro.');
    }

    $hash = Passwords::hash($password);

    // Resolver rol por tipo (RBAC manda)
    $roleCode = match ($tipo) {
        'empresa' => 'empresa',
        'rrhh' => 'rrhh',
        default => $tipo, // postulante/orador/profesional -> deben existir como roles o caer al de postulante
    };

    $pdo->beginTransaction();
    try {
        // Plan FREE por defecto (id=1 en tu dump)
        $stmt = $pdo->prepare("
            INSERT INTO users (email, password_hash, estado, email_verificado, tipo, plan_id)
            VALUES (:e, :h, 'activo', 0, :t, 1)
        ");
        $stmt->execute([':e' => $email, ':h' => $hash, ':t' => $tipo]);
        $uid = (int)$pdo->lastInsertId();

        $pdo->prepare("INSERT INTO user_security (user_id) VALUES (:uid)")
            ->execute([':uid' => $uid]);

        $pdo->prepare("INSERT INTO user_twofactor (user_id, enabled) VALUES (:uid, 0)")
            ->execute([':uid' => $uid]);

        // Asegurar que el rol exista; si no existe, caemos a postulante (para no romper registro)
        $stRole = $pdo->prepare("SELECT id FROM roles WHERE code = :c LIMIT 1");
        $stRole->execute([':c' => $roleCode]);
        $rid = (int)($stRole->fetchColumn() ?: 0);

        if ($rid <= 0) {
            $stRole->execute([':c' => 'postulante']);
            $rid = (int)($stRole->fetchColumn() ?: 0);
        }
        if ($rid <= 0) {
            throw new RuntimeException("No existe rol base en tabla roles (postulante).");
        }

        $pdo->prepare("INSERT INTO user_roles (user_id, role_id) VALUES (:u, :r)")
            ->execute([':u' => $uid, ':r' => $rid]);

        // Si es RRHH, creamos el perfil base (tabla nueva; ver SQL en sección 3)
        if ($tipo === 'rrhh') {
            $pdo->prepare("
                INSERT INTO rrhh_profiles (user_id, display_name, estado)
                VALUES (:uid, NULL, 'activo')
            ")->execute([':uid' => $uid]);
        }

        $pdo->commit();
        return ['ok' => true];

    } catch (Throwable $e) {
        $pdo->rollBack();
        throw $e;
    }
}

    public static function login(string $email, string $password): array
    {
        $email = strtolower(trim($email));
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new ValidationException('Credenciales inválidas.');
        }

        $pdo = DB::pdo();

        $stmt = $pdo->prepare("
          SELECT u.id, u.password_hash, u.estado, u.tipo,
                 us.failed_logins, us.locked_until
          FROM users u
          LEFT JOIN user_security us ON us.user_id = u.id
          WHERE u.email = :e
          LIMIT 1
        ");
        $stmt->execute([':e' => $email]);
        $row = $stmt->fetch();

        // Respuesta uniforme
        $genericFail = fn() => ['ok' => false, 'error' => 'Credenciales inválidas.'];

        if (!$row) return $genericFail();

        if (($row['estado'] ?? '') === 'bloqueado') return $genericFail();

        $lockedUntil = $row['locked_until'] ?? null;
        if ($lockedUntil && strtotime((string)$lockedUntil) > time()) {
            return ['ok' => false, 'error' => 'Cuenta temporalmente bloqueada. Intenta más tarde.'];
        }

        $hash = (string)($row['password_hash'] ?? '');
        if ($hash === '' || !Passwords::verify($password, $hash)) {
            self::onFailedLogin((int)$row['id']);
            return $genericFail();
        }

        // Login OK: reset fail counters + auditoría
        self::onSuccessfulLogin((int)$row['id']);

        // Sesión segura
        session_regenerate_id(true);

        $k = $GLOBALS['APP_CONFIG']['auth']['session_user_key'] ?? '__uid';
        $_SESSION[$k] = (int)$row['id'];

        // NOTA: __ut puede quedar por compatibilidad, pero RBAC manda
        $_SESSION['__ut'] = (string)($row['tipo'] ?? '');

        return ['ok' => true];
    }

    public static function logout(): void
    {
        $_SESSION = [];
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(
                session_name(),
                '',
                time() - 42000,
                $params["path"],
                $params["domain"],
                $params["secure"],
                $params["httponly"]
            );
        }
        session_destroy();
    }

    /* =========================
     * Session helpers (middlewares)
     * ========================= */

    public static function currentUserId(): ?int
    {
        $k = $GLOBALS['APP_CONFIG']['auth']['session_user_key'] ?? '__uid';
        $uid = $_SESSION[$k] ?? null;
        return $uid ? (int)$uid : null;
    }

    public static function check(): bool
    {
        return self::currentUserId() !== null;
    }

    public static function currentUserType(): ?string
    {
        $t = $_SESSION['__ut'] ?? null;
        return $t ? (string)$t : null;
    }

    /**
     * Admin por RBAC (NO por users.tipo)
     * Mantengo la firma sin parámetros para que no rompa tu código actual.
     */
    public static function isAdmin(): bool
    {
        $uid = self::currentUserId();
        if (!$uid) return false;
        return self::hasPermission($uid, 'admin.access');
    }

    public static function requireLogin(): int
    {
        $uid = self::currentUserId();
        if (!$uid) {
            throw new UnauthorizedException('No autenticado.');
        }
        return $uid;
    }

    /* =========================
     * RBAC
     * ========================= */

    public static function hasPermission(int $userId, string $permCode): bool
    {
        $pdo = DB::pdo();

        $sql = "
          SELECT 1
          FROM user_roles ur
          JOIN role_permissions rp ON rp.role_id = ur.role_id
          JOIN permissions p ON p.id = rp.permission_id
          JOIN roles r ON r.id = ur.role_id
          WHERE ur.user_id = :uid
            AND p.code = :pc
            AND r.estado = 'activo'
          LIMIT 1
        ";

        $st = $pdo->prepare($sql);
        $st->execute([
            ':uid' => $userId,
            ':pc'  => $permCode,
        ]);

        return (bool)$st->fetch();
    }

    /** Conveniencia: permiso para el usuario logueado */
    public static function hasPermissionCurrent(string $permCode): bool
    {
        $uid = self::currentUserId();
        if (!$uid) return false;
        return self::hasPermission($uid, $permCode);
    }

    /* =========================
     * Seguridad login
     * ========================= */

    private static function onFailedLogin(int $userId): void
    {
        $pdo = DB::pdo();
        $cfg = $GLOBALS['APP_CONFIG']['auth'] ?? [];
        $maxFails = (int)($cfg['max_failed_logins'] ?? 8);
        $lockMinutes = (int)($cfg['lock_minutes'] ?? 15);

        $pdo->beginTransaction();
        try {
            $stmt = $pdo->prepare("SELECT failed_logins FROM user_security WHERE user_id = :uid FOR UPDATE");
            $stmt->execute([':uid' => $userId]);
            $row = $stmt->fetch();
            $fails = (int)($row['failed_logins'] ?? 0) + 1;

            $lockedUntil = null;
            if ($fails >= $maxFails) {
                $lockedUntil = date('Y-m-d H:i:s', time() + ($lockMinutes * 60));
                $fails = 0; // reset tras lock
            }

            $stmt = $pdo->prepare("
              UPDATE user_security
              SET failed_logins = :f,
                  locked_until = :lu,
                  updated_at = NOW()
              WHERE user_id = :uid
            ");
            $stmt->execute([
                ':f'   => $fails,
                ':lu'  => $lockedUntil,
                ':uid' => $userId,
            ]);

            $pdo->commit();
        } catch (Throwable $e) {
            $pdo->rollBack();
        }
    }

    private static function onSuccessfulLogin(int $userId): void
    {
        $pdo = DB::pdo();
        $stmt = $pdo->prepare("
          UPDATE user_security
          SET failed_logins = 0,
              locked_until = NULL,
              last_login_at = NOW(),
              last_login_ip = :ip,
              updated_at = NOW()
          WHERE user_id = :uid
        ");
        $stmt->execute([
            ':uid' => $userId,
            ':ip'  => $_SERVER['REMOTE_ADDR'] ?? '',
        ]);
    }
    
        /* ===============================
     * GOOGLE OAUTH: login/registro
     * =============================== */

    public static function loginWithGoogle(string $providerSub, ?string $email, ?string $name): array
    {
        $pdo = DB::pdo();

        $pdo->beginTransaction();
        try {
            // 1) ¿Ya existe identidad?
            $st = $pdo->prepare("
                SELECT u.id, u.estado, u.tipo
                FROM oauth_identities oi
                JOIN users u ON u.id = oi.user_id
                WHERE oi.provider = 'google'
                  AND oi.provider_sub = :sub
                LIMIT 1
            ");
            $st->execute([':sub' => $providerSub]);
            $row = $st->fetch();

            if ($row) {
                if (($row['estado'] ?? '') === 'bloqueado') {
                    $pdo->commit();
                    return ['ok' => false, 'error' => 'Cuenta bloqueada.'];
                }

                $uid = (int)$row['id'];
                self::startSessionForUser($uid, (string)($row['tipo'] ?? 'postulante'));
                $pdo->commit();
                return ['ok' => true];
            }

            // 2) Si no existe identidad, intentamos linkear por email (si viene)
            $uid = 0;
            $tipo = 'postulante';

            if ($email) {
                $st = $pdo->prepare("SELECT id, estado, tipo FROM users WHERE email = :e LIMIT 1");
                $st->execute([':e' => strtolower(trim($email))]);
                $u = $st->fetch();

                if ($u) {
                    if (($u['estado'] ?? '') === 'bloqueado') {
                        $pdo->commit();
                        return ['ok' => false, 'error' => 'Cuenta bloqueada.'];
                    }
                    $uid = (int)$u['id'];
                    $tipo = (string)($u['tipo'] ?? 'postulante');
                }
            }

            // 3) Si no existe usuario, crearlo (password_hash NULL)
            if ($uid === 0) {
                $st = $pdo->prepare("
                    INSERT INTO users (email, password_hash, estado, email_verificado, nombre, tipo)
                    VALUES (:e, NULL, 'activo', 1, :n, 'postulante')
                ");
                $st->execute([
                    ':e' => $email ? strtolower(trim($email)) : ('google_' . $providerSub . '@noemail.local'),
                    ':n' => $name ? trim($name) : null,
                ]);
                $uid = (int)$pdo->lastInsertId();
                $tipo = 'postulante';

                // filas auxiliares
                $st = $pdo->prepare("INSERT INTO user_security (user_id) VALUES (:uid)");
                $st->execute([':uid' => $uid]);

                $st = $pdo->prepare("INSERT INTO user_twofactor (user_id, enabled) VALUES (:uid, 0)");
                $st->execute([':uid' => $uid]);
            }

            // 4) Registrar identidad oauth
            $st = $pdo->prepare("
                INSERT INTO oauth_identities (user_id, provider, provider_sub, provider_email)
                VALUES (:uid, 'google', :sub, :pe)
            ");
            $st->execute([
                ':uid' => $uid,
                ':sub' => $providerSub,
                ':pe'  => $email,
            ]);

            self::startSessionForUser($uid, $tipo);

            $pdo->commit();
            return ['ok' => true];

        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }

    private static function startSessionForUser(int $userId, string $userType): void
    {
        session_regenerate_id(true);

        $k = $GLOBALS['APP_CONFIG']['auth']['session_user_key'] ?? '__uid';
        $_SESSION[$k] = $userId;
        $_SESSION['__ut'] = $userType;
    }

}
