<?php
declare(strict_types=1);

final class TwoFactor
{
    public static function generateSecret(int $bytes = 10): string
    {
        $raw = random_bytes($bytes);
        return self::base32Encode($raw);
    }

    public static function verifyCode(string $secretBase32, string $code, int $window = 1): bool
    {
        $code = preg_replace('/\s+/', '', $code);
        if (!preg_match('/^\d{6}$/', (string)$code)) return false;

        $secret = self::base32Decode($secretBase32);
        if ($secret === '') return false;

        $timeSlice = (int)floor(time() / 30);

        for ($i = -$window; $i <= $window; $i++) {
            $calc = self::totp($secret, $timeSlice + $i);
            if (hash_equals($calc, $code)) return true;
        }
        return false;
    }

    private static function totp(string $secret, int $timeSlice): string
    {
        $time = pack('N*', 0) . pack('N*', $timeSlice);
        $hash = hash_hmac('sha1', $time, $secret, true);
        $offset = ord($hash[19]) & 0x0F;
        $truncated = (
            ((ord($hash[$offset]) & 0x7F) << 24) |
            ((ord($hash[$offset + 1]) & 0xFF) << 16) |
            ((ord($hash[$offset + 2]) & 0xFF) << 8) |
            (ord($hash[$offset + 3]) & 0xFF)
        );
        $otp = $truncated % 1000000;
        return str_pad((string)$otp, 6, '0', STR_PAD_LEFT);
    }

    private static function base32Encode(string $data): string
    {
        $alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
        $bits = '';
        foreach (str_split($data) as $c) {
            $bits .= str_pad(decbin(ord($c)), 8, '0', STR_PAD_LEFT);
        }
        $chunks = str_split($bits, 5);
        $out = '';
        foreach ($chunks as $chunk) {
            if (strlen($chunk) < 5) $chunk = str_pad($chunk, 5, '0', STR_PAD_RIGHT);
            $out .= $alphabet[bindec($chunk)];
        }
        return $out;
    }

    private static function base32Decode(string $b32): string
    {
        $b32 = strtoupper(preg_replace('/[^A-Z2-7]/', '', $b32));
        $alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
        $bits = '';
        for ($i=0; $i<strlen($b32); $i++) {
            $v = strpos($alphabet, $b32[$i]);
            if ($v === false) return '';
            $bits .= str_pad(decbin($v), 5, '0', STR_PAD_LEFT);
        }
        $bytes = str_split($bits, 8);
        $out = '';
        foreach ($bytes as $byte) {
            if (strlen($byte) < 8) continue;
            $out .= chr(bindec($byte));
        }
        return $out;
    }
}
