<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/Config/config.php';

require_once dirname(__DIR__) . '/Support/Logger.php';
require_once dirname(__DIR__) . '/Support/Helpers.php';
require_once dirname(__DIR__) . '/Support/Exceptions.php';
require_once dirname(__DIR__) . '/Support/AuditLogService.php';
require_once dirname(__DIR__) . '/Support/Seo.php';
require_once dirname(__DIR__) . '/Support/NotificationService.php';
require_once dirname(__DIR__) . '/Support/ImageUpload.php';
require_once dirname(__DIR__) . '/Support/EmpresaProfileUpload.php';

require_once dirname(__DIR__) . '/Http/Request.php';
require_once dirname(__DIR__) . '/Http/Response.php';
require_once dirname(__DIR__) . '/Http/Router.php';

require_once dirname(__DIR__) . '/Security/CsrfToken.php';
require_once dirname(__DIR__) . '/Security/RateLimiter.php';
require_once dirname(__DIR__) . '/Security/Captcha.php';

require_once dirname(__DIR__) . '/Http/Middlewares/SecurityHeaders.php';
require_once dirname(__DIR__) . '/Http/Middlewares/Csrf.php';
require_once dirname(__DIR__) . '/Http/Middlewares/RateLimit.php';
require_once dirname(__DIR__) . '/Http/Middlewares/RequireAuth.php';
require_once dirname(__DIR__) . '/Http/Middlewares/RequireModule.php';
require_once dirname(__DIR__) . '/Http/Middlewares/RequirePermission.php';

require_once dirname(__DIR__) . '/DB/DB.php';

require_once dirname(__DIR__) . '/Taxonomy/CategoriaLaboralService.php';

require_once dirname(__DIR__) . '/Admin/AdminModuleService.php';
require_once dirname(__DIR__) . '/Admin/AdminPlanService.php';
require_once dirname(__DIR__) . '/Admin/AdminPlanModuleService.php';
require_once dirname(__DIR__) . '/Admin/AdminRoleService.php';
require_once dirname(__DIR__) . '/Admin/AdminRolePermissionService.php';
require_once dirname(__DIR__) . '/Admin/AdminUserRoleService.php';
require_once dirname(__DIR__) . '/Admin/AdminMenu.php';
require_once dirname(__DIR__) . '/Admin/AdminMenuBuilder.php';
require_once dirname(__DIR__) . '/Admin/AdminModerationVacanteService.php';

require_once dirname(__DIR__) . '/Auth/Passwords.php';
require_once dirname(__DIR__) . '/Auth/AuthService.php';
require_once dirname(__DIR__) . '/Auth/ModuleService.php';
require_once dirname(__DIR__) . '/Auth/PlanLimitService.php';
require_once dirname(__DIR__) . '/Auth/GoogleOAuthService.php';

require_once dirname(__DIR__) . '/Empresa/EmpresaVacanteService.php';
require_once dirname(__DIR__) . '/Empresa/EmpresaProfileService.php';
require_once dirname(__DIR__) . '/Empresa/EmpresaReputationService.php';

require_once dirname(__DIR__) . '/Jobs/VacantePublicService.php';
require_once dirname(__DIR__) . '/Jobs/PostulacionService.php';
require_once dirname(__DIR__) . '/Jobs/FeedService.php';
require_once dirname(__DIR__) . '/Jobs/EmpresaPublicService.php';
require_once dirname(__DIR__) . '/Jobs/VacanteSlugService.php';
require_once dirname(__DIR__) . '/Jobs/VacanteSocialService.php';

require_once dirname(__DIR__) . '/Postulante/PostulanteProfileService.php';
require_once dirname(__DIR__) . '/Postulante/CvService.php';

require_once dirname(__DIR__) . '/Monetizacion/FeedBoostService.php';

require_once dirname(__DIR__) . '/Rrhh/RrhhProfileService.php';
require_once dirname(__DIR__) . '/Rrhh/RrhhSkillService.php';
require_once dirname(__DIR__) . '/Rrhh/RrhhAnuncioService.php';
require_once dirname(__DIR__) . '/Rrhh/RrhhRatingService.php';

/**
 * Devuelve IDs de categorías/roles seleccionadas por el usuario (postulante),
 * leyendo desde user_categorias.
 * No depende de CategoriaLaboralService::userSelectedIds() (que no existe).
 */
/**
 * Devuelve IDs de categorías/roles seleccionadas por el usuario (postulante),
 * leyendo desde user_categorias.
 */
function user_selected_categoria_ids(int $uid): array
{
    if ($uid <= 0) return [];

    // Lee desde user_categorias (tabla real)
    try {
        $pdo = DB::pdo();
        $st = $pdo->prepare("SELECT categoria_id FROM user_categorias WHERE user_id = :uid");
        $st->execute([':uid' => $uid]);
        $ids = [];
        while ($r = $st->fetch(PDO::FETCH_ASSOC)) {
            $ids[] = (int)($r['categoria_id'] ?? 0);
        }
        return array_values(array_unique(array_filter($ids)));
    } catch (Throwable $e) {
        return [];
    }
}


final class App
{
    private function gateRRHH(Request $req, string $permCode, callable $handler): Response
{
    // 1) Tipo de usuario (tolerante a builds donde AuthService no tiene currentUserType)
    $tipo = '';
    if (class_exists('AuthService') && method_exists('AuthService', 'currentUserType')) {
        try { $tipo = (string)AuthService::currentUserType(); } catch (Throwable $e) { $tipo = ''; }
    }
    if ($tipo === '') {
        $tipo = (string)($_SESSION['user_tipo'] ?? ($_SESSION['tipo'] ?? ''));
    }
    if ($tipo === '' && class_exists('AuthService') && method_exists('AuthService', 'currentUser')) {
        try {
            $u = AuthService::currentUser();
            $tipo = (string)($u['tipo'] ?? $u['user_tipo'] ?? '');
        } catch (Throwable $e) {}
    }

    // 2) Solo RRHH (y admin para pruebas)
    if ($tipo !== 'rrhh' && $tipo !== 'admin') {
        return Response::html("<h3>No autorizado</h3>", 403);
    }

    // 3) Middleware por módulo (si existe la clase)
    if (class_exists('RequireModuleMiddleware')) {
        $mwModule = new RequireModuleMiddleware($this->config, $this->logger, 'rrhh_panel');
        return $mwModule->handle($req, function (Request $req) use ($permCode, $handler): Response {
            // 4) Permiso RBAC (si existe la clase)
            if (class_exists('RequirePermissionMiddleware')) {
                $mwPerm = new RequirePermissionMiddleware($this->config, $this->logger, $permCode);
                return $mwPerm->handle($req, function (Request $req) use ($handler): Response {
                    return $handler($req);
                });
            }
            // Si no existe middleware RBAC por clase (build viejo), deja pasar
            return $handler($req);
        });
    }

    // 5) Si no hay middleware de módulos, al menos aplica RBAC si existe
    if (class_exists('RequirePermissionMiddleware')) {
        $mwPerm = new RequirePermissionMiddleware($this->config, $this->logger, $permCode);
        return $mwPerm->handle($req, function (Request $req) use ($handler): Response {
            return $handler($req);
        });
    }

    // 6) Build mínimo: deja pasar
    return $handler($req);
}

    private Router $router;
    private Logger $logger;

    public function __construct(private array $config)
    {
        $this->logger = new Logger($config['paths']['logs_dir']);
        $this->installErrorHandling();

        $this->router = new Router();

        // Middlewares base (orden IMPORTA)
        $this->router->addMiddleware(new SecurityHeadersMiddleware($config));
        $this->router->addMiddleware(new RateLimitMiddleware($config, $this->logger));
        $this->router->addMiddleware(new CsrfMiddleware($config, $this->logger));
        $this->router->addMiddleware(new RequireAuthMiddleware($config, $this->logger));

        $this->registerRoutes();
    }

    public function run(): void
    {
        $req = Request::fromGlobals();
        $res = $this->router->dispatch($req);
        $res->send();
    }

    /* ===============================
     * Error Handling (HTML/JSON)
     * =============================== */
    private function installErrorHandling(): void
    {
        $logger = $this->logger;

        set_exception_handler(function (Throwable $e) use ($logger): void {
            $id = bin2hex(random_bytes(8));

            $logger->error('Unhandled exception', [
                'id'      => $id,
                'type'    => get_class($e),
                'message' => $e->getMessage(),
                'file'    => $e->getFile(),
                'line'    => $e->getLine(),
            ]);

            $isDebug = (bool)($GLOBALS['APP_CONFIG']['app']['debug'] ?? false);

            $accept = strtolower((string)($_SERVER['HTTP_ACCEPT'] ?? ''));
            $isAjax = (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest');
            $format = strtolower((string)($_GET['format'] ?? ''));

            $wantsJson =
                $isAjax ||
                str_contains($accept, 'application/json') ||
                str_contains($accept, 'text/json') ||
                ($format === 'json');

            if (headers_sent()) {
                if ($wantsJson) {
                    echo json_encode([
                        'ok'       => false,
                        'error'    => $isDebug ? $e->getMessage() : 'Error interno',
                        'error_id' => $id,
                    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                } else {
                    echo "<pre>ERROR {$id}: " . htmlspecialchars($isDebug ? $e->getMessage() : 'Error interno') . "</pre>";
                }
                return;
            }

            if ($wantsJson) {
                Response::json([
                    'ok'       => false,
                    'error'    => $isDebug ? $e->getMessage() : 'Error interno',
                    'error_id' => $id,
                ], 500)->send();
                return;
            }

            $msg = $isDebug
                ? ($e->getMessage() . " @ " . $e->getFile() . ":" . $e->getLine())
                : 'Error interno';

            $html = "<!doctype html><html><head><meta charset='utf-8'>"
                . "<meta name='viewport' content='width=device-width,initial-scale=1'>"
                . "<title>Error</title>"
                . "<style>
                    body{font-family:system-ui;padding:24px;background:#f6f8ff}
                    .card{background:#fff;border:1px solid #e8eefc;border-radius:14px;padding:18px;max-width:860px}
                    h2{margin:0 0 10px}
                    p{margin:0 0 8px}
                  </style>"
                . "</head><body>"
                . "<div class='card'>"
                . "<h2>Ocurrió un error</h2>"
                . "<p>" . htmlspecialchars($msg) . "</p>"
                . "<p><small>ID: " . htmlspecialchars($id) . "</small></p>"
                . "</div>"
                . "</body></html>";

            Response::html($html, 500)->send();
        });

        set_error_handler(function (int $severity, string $message, string $file, int $line) use ($logger): bool {
            $logger->error('PHP error', [
                'severity' => $severity,
                'message'  => $message,
                'file'     => $file,
                'line'     => $line,
            ]);
            return false;
        });
    }

    /* ===============================
     * Rutas con alias /public
     * =============================== */
    private function mapGet(string $path, callable $handler): void
    {
        $this->router->get($path, $handler);

        if ($path === '/') {
            $this->router->get('/public', $handler);
            return;
        }

        if (!str_starts_with($path, '/public/')) {
            $this->router->get('/public' . $path, $handler);
        }
    }

    private function mapPost(string $path, callable $handler): void
    {
        $this->router->post($path, $handler);

        if (!str_starts_with($path, '/public/')) {
            $this->router->post('/public' . $path, $handler);
        }
    }

    /* ===============================
     * Redirect robusto
     * =============================== */
    private function redirect(string $to, int $status = 302): Response
    {
        $to = $to === '' ? (base_path() . '/') : $to;

        $html = '<!doctype html><html><head><meta charset="utf-8">'
            . '<meta http-equiv="refresh" content="0;url=' . htmlspecialchars($to) . '">'
            . '</head><body>Redirigiendo...</body></html>';

        return Response::html($html, $status)->header('Location', $to);
    }

    /* ===============================
     * Render layout admin con menú dinámico
     * =============================== */
    private function renderAdminLayout(string $pageTitle, string $contentHtml): string
    {
        $menu = $GLOBALS['__admin_menu'] ?? [];

        return render_template('layouts/admin.php', [
            'pageTitle' => $pageTitle,
            'menu'      => is_array($menu) ? $menu : [],
            'content'   => $contentHtml,
        ]);
    }

    /* ===============================
     * Gate helper (Admin: módulo + permiso)
     * =============================== */
    private function gateAdmin(Request $req, string $permCode, callable $handler): Response
    {
        $mwModule = new RequireModuleMiddleware($this->config, $this->logger, 'admin_panel');
        $mwPerm   = new RequirePermissionMiddleware($this->config, $this->logger, $permCode);

        return $mwModule->handle($req, function (Request $req) use ($mwPerm, $handler): Response {
            return $mwPerm->handle($req, function (Request $req) use ($handler): Response {

                // Inyección global del menú para el layout admin (una vez por request)
                $GLOBALS['__admin_menu'] = AdminMenuBuilder::buildForUser(AuthService::currentUserId());

                return $handler($req);
            });
        });
    }
    
    private function gateEmpresa(Request $req, string $permCode, callable $handler): Response
    {
        // 1) Solo usuarios tipo empresa (o admin si quieres permitir pruebas)
        $tipo = AuthService::currentUserType();
        if ($tipo !== 'empresa' && $tipo !== 'admin') {
            return Response::html("<h3>No autorizado</h3>", 403);
        }
    
        // 2) Requiere módulo job_board (por plan / override)
        $mwModule = new RequireModuleMiddleware($this->config, $this->logger, 'job_board');
    
        // 3) Requiere permiso empresarial
        $mwPerm = new RequirePermissionMiddleware($this->config, $this->logger, $permCode);
    
        return $mwModule->handle($req, function (Request $req) use ($mwPerm, $handler): Response {
            return $mwPerm->handle($req, function (Request $req) use ($handler): Response {
                return $handler($req);
            });
        });
    }
    
    private function gatePostulante(Request $req, callable $handler): Response
    {
        $tipo = AuthService::currentUserType();
        if ($tipo !== 'postulante' && $tipo !== 'admin') {
            return Response::html("<h3>No autorizado</h3>", 403);
        }
    
        $mwModule = new RequireModuleMiddleware($this->config, $this->logger, 'job_board');
    
        return $mwModule->handle($req, function (Request $req) use ($handler): Response {
            return $handler($req);
        });
    }
    

    /* ===============================
     * Routes
     * =============================== */
    private function registerRoutes(): void
    {
        /* ========= HEALTH ========= */
        $this->mapGet('/health', function (Request $req): Response {
            return Response::json(['ok' => true, 'status' => 'UP'], 200);
        });

        $this->mapGet('/db-health', function (Request $req): Response {
            try {
                DB::pdo()->query('SELECT 1');
                return Response::json(['ok' => true, 'db' => 'UP'], 200);
            } catch (Throwable $e) {
                $this->logger->error('DB health failed', ['message' => $e->getMessage()]);
                return Response::json(['ok' => false, 'db' => 'DOWN'], 500);
            }
        });

        /* ========= LANDING ========= */
        $this->mapGet('/', function (Request $req): Response {
            $cfg = $GLOBALS['APP_CONFIG'];

            $html = render_template('layouts/front.php', [
                'pageTitle' => $cfg['seo']['site_name'] ?? 'Plataforma Laboral',
                'content'   => render_template('front/index.php', []),
            ]);

            return Response::html($html, 200);
        });

        /* ========= CSRF ========= */
        $this->mapGet('/csrf', function (Request $req): Response {
            return Response::json(['ok' => true, 'csrf' => CsrfToken::issue()], 200);
        });

        /* ========= CAPTCHA ========= */
        $this->mapGet('/captcha', function (Request $req): Response {
            $code = Captcha::issue(6);
            $png  = Captcha::renderPng($code);

            $res = new Response($png, 200, 'image/png');
            $res->header('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0');
            $res->header('Pragma', 'no-cache');
            $res->header('X-Content-Type-Options', 'nosniff');
            return $res;
        });

        /* ========= LOGIN ========= */
        $this->mapGet('/login', function (Request $req): Response {
            $html = render_template('layouts/front.php', [
                'pageTitle' => 'Login',
                'content'   => render_template('auth/login.php', ['error' => null]),
            ]);
            return Response::html($html, 200);
        });

        $this->mapPost('/login', function (Request $req): Response {
            $email = trim((string)($req->post['email'] ?? ''));
            $pw    = (string)($req->post['password'] ?? '');
            $cap   = (string)($req->post['captcha'] ?? '');

            if (!Captcha::verify($cap)) {
                $this->logger->security('Captcha inválido en login', ['ip' => $req->ip]);

                $html = render_template('layouts/front.php', [
                    'pageTitle' => 'Login',
                    'content'   => render_template('auth/login.php', ['error' => 'Captcha inválido.']),
                ]);
                return Response::html($html, 403);
            }

            $out = AuthService::login($email, $pw);

            if (!$out['ok']) {
                $this->logger->security('Login fallido', ['ip' => $req->ip, 'email' => $email]);

                $html = render_template('layouts/front.php', [
                    'pageTitle' => 'Login',
                    'content'   => render_template('auth/login.php', [
                        'error' => $out['error'] ?? 'Credenciales inválidas.',
                    ]),
                ]);
                return Response::html($html, 401);
            }

            $r = trim((string)($req->query['r'] ?? ''));
            $r = trim((string)($req->query['r'] ?? ''));
            if ($r !== '') {
                $dest = $r;
            } else {
                $tipo = AuthService::currentUserType();
                $bp   = base_path();
            
                $dest = match ($tipo) {
                    'empresa'     => $bp . '/empresa',
                    'postulante'  => $bp . '/postulante/panel',
                    'rrhh'        => $bp . '/rrhh',
                    'admin'       => $bp . '/admin',
                    default       => $bp . '/feed',
                };
            }
            return $this->redirect($dest, 302);
        });
        
                /* ===============================
         * GOOGLE OAUTH
         * =============================== */
        $this->mapGet('/auth/google', function (Request $req): Response {
            if (!GoogleOAuthService::isEnabled()) {
                return Response::html('<h3>Google OAuth no habilitado</h3>', 503);
            }

            $url = GoogleOAuthService::buildAuthUrl();
            return $this->redirect($url, 302);
        });

        $this->mapGet('/auth/google/callback', function (Request $req): Response {
            if (!GoogleOAuthService::isEnabled()) {
                return Response::html('<h3>Google OAuth no habilitado</h3>', 503);
            }

            $code  = (string)($req->query['code'] ?? '');
            $state = (string)($req->query['state'] ?? '');

            if ($code === '' || $state === '') {
                return Response::html('<h3>Callback OAuth inválido</h3>', 400);
            }

            $info = GoogleOAuthService::fetchUserInfoFromCode($code, $state);

            $sub   = (string)($info['sub'] ?? '');
            $email = isset($info['email']) && is_string($info['email']) ? $info['email'] : null;
            $name  = isset($info['name']) && is_string($info['name']) ? $info['name'] : null;

            $out = AuthService::loginWithGoogle($sub, $email, $name);

            if (!$out['ok']) {
                $html = render_template('layouts/front.php', [
                    'pageTitle' => 'Login',
                    'content'   => render_template('auth/login.php', [
                        'error' => $out['error'] ?? 'No se pudo iniciar sesión con Google.',
                    ]),
                ]);
                return Response::html($html, 401);
            }

            return $this->redirect(base_path() . '/feed', 302);
        });
        
        // ===============================
// PERFIL PÚBLICO POSTULANTE
// URL: /p?id=7  (también /public/p?id=7 por mapGet)
// ===============================
$this->mapGet('/p', function (Request $req): Response {

    $id = (int)($req->query['id'] ?? 0);
    if ($id <= 0) {
        return Response::html('Perfil no encontrado (faltó id)', 404);
    }

    $pdo = DB::pdo();

    // Solo postulantes
    $st = $pdo->prepare("SELECT id,nombre,tipo FROM users WHERE id=:id LIMIT 1");
    $st->execute([':id' => $id]);
    $user = $st->fetch(PDO::FETCH_ASSOC) ?: null;

    if (!$user) {
        return Response::html('Perfil no encontrado (user no existe)', 404);
    }
    if ((string)($user['tipo'] ?? '') !== 'postulante') {
        return Response::html('Perfil no encontrado (no es postulante)', 404);
    }

    $perfil = PostulanteProfileService::getOrCreate($id);
    $cv     = CvService::latestActiveForUser($id);

    // Categorías seleccionadas
    $selectedIds = [];
    try {
        $selectedIds = user_selected_categoria_ids($id);
    } catch (Throwable $e) {
        $selectedIds = [];
    }

    $selectedCats = [];
    if ($selectedIds) {
        $in = implode(',', array_fill(0, count($selectedIds), '?'));
        $q = $pdo->prepare("SELECT id,nombre,codigo,parent_id FROM categorias_laborales WHERE id IN ($in) ORDER BY nombre ASC");
        foreach ($selectedIds as $i => $cid) {
            $q->bindValue($i + 1, (int)$cid, PDO::PARAM_INT);
        }
        $q->execute();
        $selectedCats = $q->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    // Stats reales
    $stats = ['total'=>0,'enviada'=>0,'vista'=>0,'preseleccion'=>0,'rechazada'=>0];
    try {
        $q = $pdo->prepare("
            SELECT estado, COUNT(*) c
            FROM postulaciones
            WHERE postulante_user_id = :uid
            GROUP BY estado
        ");
        $q->execute([':uid' => $id]);
        $rows = $q->fetchAll(PDO::FETCH_ASSOC) ?: [];
        foreach ($rows as $r) {
            $k = (string)($r['estado'] ?? '');
            if (isset($stats[$k])) $stats[$k] = (int)($r['c'] ?? 0);
        }
        $stats['total'] = (int)$stats['enviada'] + (int)$stats['vista'] + (int)$stats['preseleccion'] + (int)$stats['rechazada'];
    } catch (Throwable $e) {
        // defaults
    }

    $html = render_template('layouts/app.php', [
        'pageTitle' => 'Perfil público',
        'content'   => render_template('postulante/public.php', [
            'user'         => $user,
            'perfil'       => $perfil,
            'cv'           => $cv,
            'selectedCats' => $selectedCats,
            'stats'        => $stats,
        ]),
    ]);

    return Response::html($html, 200);
});


        /* ========= REGISTRO ========= */
        $this->mapGet('/registro', function (Request $req): Response {
            $html = render_template('layouts/front.php', [
                'pageTitle' => 'Registro',
                'content'   => render_template('auth/registro.php', [
                    'error' => null,
                    'ok'    => null,
                ]),
            ]);
            return Response::html($html, 200);
        });

        $this->mapPost('/registro', function (Request $req): Response {
            $email = trim((string)($req->post['email'] ?? ''));
            $pw    = (string)($req->post['password'] ?? '');
            $tipo  = (string)($req->post['tipo'] ?? 'postulante');
            $cap   = (string)($req->post['captcha'] ?? '');

            if (!Captcha::verify($cap)) {
                $this->logger->security('Captcha inválido en registro', ['ip' => $req->ip]);

                $html = render_template('layouts/front.php', [
                    'pageTitle' => 'Registro',
                    'content'   => render_template('auth/registro.php', [
                        'error' => 'Captcha inválido.',
                        'ok'    => null,
                    ]),
                ]);
                return Response::html($html, 403);
            }

            try {
                AuthService::register($email, $pw, $tipo);
                return $this->redirect(base_path() . '/login', 302);

            } catch (ValidationException $e) {
                $this->logger->security('Registro inválido', ['ip' => $req->ip, 'email' => $email]);

                $html = render_template('layouts/front.php', [
                    'pageTitle' => 'Registro',
                    'content'   => render_template('auth/registro.php', [
                        'error' => $e->getMessage(),
                        'ok'    => null,
                    ]),
                ]);
                return Response::html($html, 422);
            }
        });
        
        $this->mapGet('/empresa/vacantes/ver', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $id  = (int)($req->query['id'] ?? 0);

        $row = EmpresaVacanteService::findOwned($id, $uid);
        if (!$row) return Response::html("<h3>No existe</h3>", 404);

        $csrf = CsrfToken::issue();

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'Empresa • Ver vacante',
            'content'   => render_template('empresa/vacantes/ver.php', [
                'row'   => $row,
                'error' => null,
                'csrf'  => $csrf,
            ]),
        ]);

        return Response::html($html, 200);
    });
});

$this->mapGet('/empresa/postulaciones', function (Request $req): Response {

    // ✅ Modo RRHH: entra por gateRRHH y opera con empresa vinculada
    if ((string)($req->query['modo'] ?? '') === 'rrhh') {
        return $this->gateRRHH($req, 'rrhh.jobs.manage', function (Request $req): Response {
            $rrhhId = (int)AuthService::currentUserId();
            $empresaId = (int)(RrhhProfileService::getLinkedEmpresaId($rrhhId) ?? 0);

            if ($empresaId <= 0) {
                return Response::html("<h3>Debes vincularte a una empresa para ver postulaciones.</h3>", 403);
            }

            $vacanteId = (int)($req->query['vacante_id'] ?? 0);

            if ($vacanteId <= 0) {
                $vacantes = EmpresaVacanteService::listByEmpresa($empresaId);
                $csrf = CsrfToken::issue();

                $html = render_template('layouts/app.php', [
                    'pageTitle' => 'RRHH • Postulaciones',
                    'content'   => render_template('empresa/postulaciones/select.php', [
                        'vacantes' => $vacantes,
                        'csrf'     => $csrf,
                    ]),
                ]);
                return Response::html($html, 200);
            }

            $vacante = EmpresaVacanteService::findOwned($vacanteId, $empresaId);
            if (!$vacante) return Response::html("<h3>No existe</h3>", 404);

            $items = PostulacionService::listForVacante($vacanteId);
            $csrf  = CsrfToken::issue();

            $html = render_template('layouts/app.php', [
                'pageTitle' => 'RRHH • Postulaciones',
                'content'   => render_template('empresa/postulaciones/index.php', [
                    'vacante' => $vacante,
                    'items'   => $items,
                    'error'   => null,
                    'csrf'    => $csrf,
                ]),
            ]);
            return Response::html($html, 200);
        });
    }

    // ✅ Modo normal empresa
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $vacanteId = (int)($req->query['vacante_id'] ?? 0);

        if ($vacanteId <= 0) {
            $vacantes = EmpresaVacanteService::listByEmpresa($uid);
            $csrf = CsrfToken::issue();

            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Empresa • Postulaciones',
                'content'   => render_template('empresa/postulaciones/select.php', [
                    'vacantes' => $vacantes,
                    'csrf'     => $csrf,
                ]),
            ]);
            return Response::html($html, 200);
        }

        $vacante = EmpresaVacanteService::findOwned($vacanteId, $uid);
        if (!$vacante) return Response::html("<h3>No existe</h3>", 404);

        $items = PostulacionService::listForVacante($vacanteId);
        $csrf  = CsrfToken::issue();

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'Empresa • Postulaciones',
            'content'   => render_template('empresa/postulaciones/index.php', [
                'vacante' => $vacante,
                'items'   => $items,
                'error'   => null,
                'csrf'    => $csrf,
            ]),
        ]);

        return Response::html($html, 200);
    });
});


$this->mapGet('/empresa/postulaciones/vacante', function (Request $req): Response {
    // Alias retrocompatible (antes era /empresa/postulaciones/vacante)
    $vacanteId = (int)($req->query['vacante_id'] ?? 0);
    $to = url_path('empresa/postulaciones');
    if ($vacanteId > 0) $to .= '?vacante_id=' . $vacanteId;
    return $this->redirect($to, 301);
});

$this->mapPost('/empresa/postulaciones/estado', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();

        $vacanteId = (int)($req->post['vacante_id'] ?? 0);
        $postId    = (int)($req->post['postulacion_id'] ?? 0);
        $estado    = (string)($req->post['estado'] ?? '');

        $vacante = EmpresaVacanteService::findOwned($vacanteId, $uid);
        if (!$vacante) return Response::html("<h3>No existe</h3>", 404);

        PostulacionService::updateEstado($postId, $estado);
        
                // ===== NOTIFICACIÓN A POSTULANTE: cambio de estado =====
        $pdo = DB::pdo();
        $st = $pdo->prepare("
          SELECT p.postulante_user_id, v.titulo
          FROM postulaciones p
          JOIN vacantes v ON v.id = p.vacante_id
          WHERE p.id = :pid
          LIMIT 1
        ");
        $st->execute(['pid' => $postId]);
        $r = $st->fetch(\PDO::FETCH_ASSOC);

        if ($r && !empty($r['postulante_user_id'])) {
            $postulanteId = (int)$r['postulante_user_id'];
            $tituloVac = (string)($r['titulo'] ?? 'Vacante');

            NotificationService::create(
                $postulanteId,
                'postulante.postulacion.estado',
                'Tu postulación fue actualizada',
                "Actualización en: {$tituloVac}. Nuevo estado: {$estado}",
                base_path() . '/postulante/postulaciones'
            );
        }
        // ===== FIN NOTIFICACIÓN =====

        return $this->redirect(url_path('empresa/postulaciones') . '?vacante_id=' . $vacanteId, 302);
    });
});

/* ========= EMPRESA: VACANTES ========= */

$this->mapGet('/empresa/vacantes', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid   = (int)AuthService::currentUserId();
        $items = EmpresaVacanteService::listByEmpresa($uid);
        $csrf  = CsrfToken::issue();

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'Empresa • Vacantes',
            'content'   => render_template('empresa/vacantes/index.php', [
                'items' => $items,
                'error' => null,
                'csrf'  => $csrf,
            ]),
        ]);
        return Response::html($html, 200);
    });
});

$this->mapGet('/empresa/vacantes/create', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $csrf = CsrfToken::issue();
        $categorias = CategoriaLaboralService::listActivas(800);

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'Empresa • Nueva vacante',
            'content'   => render_template('empresa/vacantes/create.php', [
                'error' => null,
                'old'   => [
                    'titulo' => '',
                    'descripcion' => '',
                    'ubicacion' => '',
                    'modalidad' => 'presencial',
                    'tipo_empleo' => 'tiempo_completo',
                    'salario_min' => '',
                    'salario_max' => '',
                    'moneda' => 'CRC',
                ],
                'categorias' => $categorias,
                'oldCategorias' => [],
                'csrf'  => $csrf,
            ]),
        ]);
        return Response::html($html, 200);
    });
});

$this->mapPost('/empresa/vacantes/create', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();

        CsrfToken::verify((string)($req->post['_csrf'] ?? ''));

        $old = [
            'titulo'      => trim((string)($req->post['titulo'] ?? '')),
            'descripcion' => trim((string)($req->post['descripcion'] ?? '')),
            'ubicacion'   => trim((string)($req->post['ubicacion'] ?? '')),
            'modalidad'   => (string)($req->post['modalidad'] ?? 'presencial'),
            'tipo_empleo' => (string)($req->post['tipo_empleo'] ?? 'tiempo_completo'),
            'salario_min' => trim((string)($req->post['salario_min'] ?? '')),
            'salario_max' => trim((string)($req->post['salario_max'] ?? '')),
            'moneda'      => (string)($req->post['moneda'] ?? 'CRC'),
        ];

        $oldCategorias = $req->post['categoria_ids'] ?? [];
        $oldCategorias = array_values(array_unique(array_filter(array_map('intval', (array)$oldCategorias), fn($v) => $v > 0)));

        try {
            // ===== Asegurar carga de ImageUpload (no depender de otros requires del bootstrap)
            if (!class_exists('ImageUpload', false)) {
                $imageUploadFile = dirname(__DIR__) . '/Support/ImageUpload.php'; // /src/Support/ImageUpload.php
                if (!is_file($imageUploadFile)) {
                    throw new ValidationException('No se encontró ImageUpload.php en /src/Support/.');
                }
                require_once $imageUploadFile;
            }

            // Procesar imagen (si viene)
            $imgPath = null;
            if (!empty($_FILES['imagen']) && is_array($_FILES['imagen']) && (int)($_FILES['imagen']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK) {
                // /home/fasicat/job.fasicat.com/public
                $publicDir = dirname(__DIR__, 2) . '/public';
                $imgPath = ImageUpload::processVacanteImage($_FILES['imagen'], $publicDir, '/upload/vacantes');
            }

            $data = $old;
            $data['categoria_ids'] = $oldCategorias;
            $data['imagen_path']   = $imgPath;

            EmpresaVacanteService::create($uid, $data);
            return $this->redirect(base_path() . '/empresa/vacantes', 302);

        } catch (ValidationException $e) {
            $csrf = CsrfToken::issue();
            $categorias = CategoriaLaboralService::listActivas(800);

            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Empresa • Nueva vacante',
                'content'   => render_template('empresa/vacantes/create.php', [
                    'error' => $e->getMessage(),
                    'old'   => $old,
                    'categorias' => $categorias,
                    'oldCategorias' => $oldCategorias,
                    'csrf'  => $csrf,
                ]),
            ]);
            return Response::html($html, 422);
        }
    });
});

$this->mapGet('/empresa/vacantes/edit', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $id  = (int)($req->query['id'] ?? 0);

        $row = EmpresaVacanteService::findOwned($id, $uid);
        if (!$row) return Response::html("<h3>No existe</h3>", 404);

        $csrf = CsrfToken::issue();
        $oldCategorias = EmpresaVacanteService::getCategoriaIds($id);
        $categorias = CategoriaLaboralService::listActivas(800);

        // Asegurar que las categorías ya guardadas se vean en el edit aunque no estén "activo"
        if (!empty($oldCategorias)) {
            $existing = [];
            foreach ($categorias as $c) $existing[(int)($c['id'] ?? 0)] = true;

            $missing = [];
            foreach ($oldCategorias as $cid) {
                $cid = (int)$cid;
                if ($cid > 0 && empty($existing[$cid])) $missing[] = $cid;
            }

            if (!empty($missing)) {
                $pdo = DB::pdo();
                $in = implode(',', array_fill(0, count($missing), '?'));
                $st = $pdo->prepare("SELECT id, parent_id, nombre, codigo, orden FROM categorias_laborales WHERE id IN ($in)");
                $st->execute(array_values($missing));
                $extra = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
                if (!empty($extra)) {
                    $categorias = array_merge($categorias, $extra);
                    usort($categorias, function(array $a, array $b): int {
                        $oa = (int)($a['orden'] ?? 0);
                        $ob = (int)($b['orden'] ?? 0);
                        if ($oa !== $ob) return $oa <=> $ob;
                        $na = (string)($a['nombre'] ?? '');
                        $nb = (string)($b['nombre'] ?? '');
                        return strcmp($na, $nb);
                    });
                }
            }
        }

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'Empresa • Editar vacante',
            'content'   => render_template('empresa/vacantes/edit.php', [
                'error' => null,
                'row'   => $row,
                'categorias' => $categorias,
                'oldCategorias' => $oldCategorias,
                'csrf'  => $csrf,
            ]),
        ]);
        return Response::html($html, 200);
    });
});

$this->mapPost('/empresa/vacantes/edit', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $id  = (int)($req->post['id'] ?? 0);

        CsrfToken::verify((string)($req->post['_csrf'] ?? ''));

        $row = EmpresaVacanteService::findOwned($id, $uid);
        if (!$row) return Response::html("<h3>No existe</h3>", 404);

        $data = [
            'titulo'      => trim((string)($req->post['titulo'] ?? '')),
            'descripcion' => trim((string)($req->post['descripcion'] ?? '')),
            'ubicacion'   => trim((string)($req->post['ubicacion'] ?? '')),
            'modalidad'   => (string)($req->post['modalidad'] ?? 'presencial'),
            'tipo_empleo' => (string)($req->post['tipo_empleo'] ?? 'tiempo_completo'),
            'salario_min' => trim((string)($req->post['salario_min'] ?? '')),
            'salario_max' => trim((string)($req->post['salario_max'] ?? '')),
            'moneda'      => (string)($req->post['moneda'] ?? 'CRC'),
        ];

        $oldCategorias = $req->post['categoria_ids'] ?? [];
        $oldCategorias = array_values(array_unique(array_filter(array_map('intval', (array)$oldCategorias), fn($v) => $v > 0)));

        try {
            // ===== Asegurar carga de ImageUpload
            if (!class_exists('ImageUpload', false)) {
                $imageUploadFile = dirname(__DIR__) . '/Support/ImageUpload.php';
                if (!is_file($imageUploadFile)) {
                    throw new ValidationException('No se encontró ImageUpload.php en /src/Support/.');
                }
                require_once $imageUploadFile;
            }

            // Si NO sube imagen nueva, conservamos la actual
            $imgPath = (string)($row['imagen_path'] ?? '');
            if ($imgPath !== '' && str_starts_with($imgPath, '/public/')) {
                $imgPath = substr($imgPath, 7); // quita /public
            }
            $imgPath = $imgPath !== '' ? $imgPath : null;

            if (!empty($_FILES['imagen']) && is_array($_FILES['imagen']) && (int)($_FILES['imagen']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK) {
                $publicDir = dirname(__DIR__, 2) . '/public';
                $imgPath = ImageUpload::processVacanteImage($_FILES['imagen'], $publicDir, '/upload/vacantes');
            }

            $data['categoria_ids'] = $oldCategorias;
            $data['imagen_path']   = $imgPath;

            EmpresaVacanteService::update($id, $uid, $data);
            return $this->redirect(base_path() . '/empresa/vacantes', 302);

        } catch (ValidationException $e) {
            $csrf = CsrfToken::issue();
            $categorias = CategoriaLaboralService::listActivas(800);

            // re-hidratar row visible
            $row['titulo'] = $data['titulo'];
            $row['descripcion'] = $data['descripcion'];
            $row['ubicacion'] = $data['ubicacion'] !== '' ? $data['ubicacion'] : null;
            $row['modalidad'] = $data['modalidad'];
            $row['tipo_empleo'] = $data['tipo_empleo'];
            $row['salario_min'] = $data['salario_min'] === '' ? null : (float)$data['salario_min'];
            $row['salario_max'] = $data['salario_max'] === '' ? null : (float)$data['salario_max'];
            $row['moneda'] = strtoupper($data['moneda']);

            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Empresa • Editar vacante',
                'content'   => render_template('empresa/vacantes/edit.php', [
                    'error' => $e->getMessage(),
                    'row'   => $row,
                    'categorias' => $categorias,
                    'oldCategorias' => $oldCategorias,
                    'csrf'  => $csrf,
                ]),
            ]);
            return Response::html($html, 422);
        }
    });
});

$this->mapPost('/empresa/vacantes/publish', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $id  = (int)($req->post['id'] ?? 0);

        CsrfToken::verify((string)($req->post['_csrf'] ?? ''));

        try {
            EmpresaVacanteService::publish($id, $uid);
            return $this->redirect(base_path() . '/empresa/vacantes', 302);

        } catch (ValidationException $e) {
            $items = EmpresaVacanteService::listByEmpresa($uid);
            $csrf  = CsrfToken::issue();

            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Empresa • Vacantes',
                'content'   => render_template('empresa/vacantes/index.php', [
                    'items' => $items,
                    'error' => $e->getMessage(),
                    'csrf'  => $csrf,
                ]),
            ]);
            return Response::html($html, 422);
        }
    });
});

$this->mapPost('/empresa/vacantes/close', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $id  = (int)($req->post['id'] ?? 0);

        CsrfToken::verify((string)($req->post['_csrf'] ?? ''));

        try {
            EmpresaVacanteService::close($id, $uid);
            return $this->redirect(base_path() . '/empresa/vacantes', 302);

        } catch (ValidationException $e) {
            $items = EmpresaVacanteService::listByEmpresa($uid);
            $csrf  = CsrfToken::issue();

            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Empresa • Vacantes',
                'content'   => render_template('empresa/vacantes/index.php', [
                    'items' => $items,
                    'error' => $e->getMessage(),
                    'csrf'  => $csrf,
                ]),
            ]);
            return Response::html($html, 422);
        }
    });
});

$this->mapPost('/empresa/vacantes/delete', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $id  = (int)($req->post['id'] ?? 0);

        CsrfToken::verify((string)($req->post['_csrf'] ?? ''));

        try {
            EmpresaVacanteService::delete($id, $uid);
            return $this->redirect(base_path() . '/empresa/vacantes', 302);

        } catch (ValidationException $e) {
            $items = EmpresaVacanteService::listByEmpresa($uid);
            $csrf  = CsrfToken::issue();

            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Empresa • Vacantes',
                'content'   => render_template('empresa/vacantes/index.php', [
                    'items' => $items,
                    'error' => $e->getMessage(),
                    'csrf'  => $csrf,
                ]),
            ]);
            return Response::html($html, 422);
        }
    });
});

/* ========= EMPRESA PANEL (nuevo shell) ========= */

$this->mapGet('/empresa', function (Request $req): Response {
    $uid = (int) AuthService::currentUserId();
    if ($uid <= 0) return Response::html("<h3>401</h3>", 401);

    $pdo = DB::pdo();

    $stU = $pdo->prepare("SELECT id, email, tipo, nombre FROM users WHERE id = :id LIMIT 1");
    $stU->execute([':id' => $uid]);
    $user = $stU->fetch(PDO::FETCH_ASSOC);

    if (!$user || ($user['tipo'] ?? '') !== 'empresa') {
        return Response::html("<h3>403</h3>", 403);
    }

    $perfil = EmpresaProfileService::get($uid);
    $rep    = EmpresaReputationService::get($uid);

    $st = $pdo->prepare("
        SELECT
          (SELECT COUNT(*) FROM vacantes v WHERE v.empresa_user_id = :e1) AS vacantes_total,
          (SELECT COUNT(*) FROM vacantes v WHERE v.empresa_user_id = :e2 AND v.estado='publicada') AS vacantes_publicadas,
          (SELECT COUNT(*) FROM postulaciones p
             INNER JOIN vacantes v ON v.id = p.vacante_id
             WHERE v.empresa_user_id = :e3) AS postulaciones_total,
          (SELECT COUNT(*) FROM feed_boosts fb
             WHERE fb.actor_user_id = :e4 AND fb.actor_tipo='empresa'
               AND fb.estado='activo'
               AND NOW() BETWEEN fb.starts_at AND fb.ends_at) AS boosts_activos
    ");
    $st->execute([
        ':e1' => $uid,
        ':e2' => $uid,
        ':e3' => $uid,
        ':e4' => $uid,
    ]);
    $stats = $st->fetch(PDO::FETCH_ASSOC) ?: [];

    $sidebar = render_template('partials/sidebar_empresa.php', ['active' => 'dash']);

    $html = render_template('layouts/panel.php', [
        'pageTitle' => 'Panel Empresa',
        'topHint'   => 'Gestión y monetización',
        'sidebar'   => $sidebar,
        'content'   => render_template('empresa/dashboard.php', [
            'empresa' => $user,
            'perfil'  => $perfil,
            'rep'     => $rep,
            'stats'   => $stats,
        ]),
    ]);

    return Response::html($html, 200);
});

/* ========= PERFIL EMPRESA ========= */

$this->mapGet('/empresa/perfil', function (Request $req): Response {
    $uid = (int) AuthService::currentUserId();
    if ($uid <= 0) return Response::html("<h3>401</h3>", 401);

    $pdo = DB::pdo();

    $stU = $pdo->prepare("SELECT id, email, tipo, nombre FROM users WHERE id = :id LIMIT 1");
    $stU->execute([':id' => $uid]);
    $user = $stU->fetch(PDO::FETCH_ASSOC);

    if (!$user || ($user['tipo'] ?? '') !== 'empresa') {
        return Response::html("<h3>403</h3>", 403);
    }

    // Si existe tu service, lo seguimos usando para GET (ya lo tenías así).
    // Si no existiera por alguna razón, caemos a consulta directa.
    if (class_exists('EmpresaProfileService') && method_exists('EmpresaProfileService', 'get')) {
        $perfil = EmpresaProfileService::get($uid);
    } else {
        $stP = $pdo->prepare("SELECT * FROM empresa_perfiles WHERE user_id = :uid LIMIT 1");
        $stP->execute([':uid' => $uid]);
        $perfil = $stP->fetch(PDO::FETCH_ASSOC) ?: [];
    }

    $sidebar = render_template('partials/sidebar_empresa.php', ['active' => 'perfil']);
    $csrf    = CsrfToken::issue();

    $ok = null;
    if (($req->query['ok'] ?? '') === '1') $ok = 'Guardado correctamente.';

    $html = render_template('layouts/panel.php', [
        'pageTitle' => 'Perfil Empresa',
        'topHint'   => 'Tu perfil público',
        'sidebar'   => $sidebar,
        'content'   => render_template('empresa/perfil.php', [
            'empresa' => $user,
            'perfil'  => $perfil,
            'csrf'    => $csrf,
            'ok'      => $ok,
            'error'   => null,
        ]),
    ]);

    return Response::html($html, 200);
});

$this->mapPost('/empresa/perfil', function (Request $req): Response {
    // IMPORTANTE: si no tenés asignado 'empresa.profile.manage' te da 403.
    // Cambiamos a un permiso que ya estás usando en vacantes.
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int) AuthService::currentUserId();
        if ($uid <= 0) return Response::html("<h3>401</h3>", 401);

        CsrfToken::verify((string)($req->post['_csrf'] ?? ''));

        $pdo = DB::pdo();

        // Usuario empresa
        $stU = $pdo->prepare("SELECT id, email, tipo, nombre FROM users WHERE id = :id LIMIT 1");
        $stU->execute([':id' => $uid]);
        $user = $stU->fetch(PDO::FETCH_ASSOC);

        if (!$user || ($user['tipo'] ?? '') !== 'empresa') {
            return Response::html("<h3>403</h3>", 403);
        }

        // Inputs
        $nombre         = trim((string)($req->post['nombre'] ?? ''));
        $telefono        = trim((string)($req->post['telefono'] ?? ''));
        $whatsapp        = trim((string)($req->post['whatsapp'] ?? ''));
        $ubicacion       = trim((string)($req->post['ubicacion'] ?? ''));
        $website         = trim((string)($req->post['website_url'] ?? ''));
        $googleBusiness  = trim((string)($req->post['google_business_url'] ?? ''));
        $bio             = trim((string)($req->post['bio'] ?? ''));

        if ($nombre === '' || mb_strlen($nombre) < 2) {
            throw new ValidationException('El nombre es obligatorio.');
        }
        if (mb_strlen($bio) > 500) {
            throw new ValidationException('La descripción no puede superar 500 caracteres.');
        }

        // ===== helper upload inline (evita depender de clases inexistentes) =====
        $processImage = function (array $file, string $publicDir, string $relativeDir, string $prefix): ?string {
            if (empty($file) || !isset($file['tmp_name']) || (int)($file['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
                return null;
            }

            $tmp = (string)$file['tmp_name'];
            if (!is_file($tmp)) return null;

            $info = @getimagesize($tmp);
            if (!$info || empty($info[0]) || empty($info[1])) {
                throw new ValidationException('Imagen inválida.');
            }

            $mime = (string)($info['mime'] ?? '');
            $src = null;

            switch ($mime) {
                case 'image/jpeg': $src = @imagecreatefromjpeg($tmp); break;
                case 'image/png':  $src = @imagecreatefrompng($tmp); break;
                case 'image/webp':
                    $src = function_exists('imagecreatefromwebp') ? @imagecreatefromwebp($tmp) : null;
                    break;
                case 'image/avif':
                    $src = function_exists('imagecreatefromavif') ? @imagecreatefromavif($tmp) : null;
                    break;
                default:
                    throw new ValidationException('Formato no soportado. Usá JPG/PNG/WebP.');
            }

            if (!$src) throw new ValidationException('No se pudo leer la imagen (GD/AVIF/WebP).');

            $destAbsDir = rtrim($publicDir, '/\\') . rtrim($relativeDir, '/\\');
            if (!is_dir($destAbsDir)) @mkdir($destAbsDir, 0775, true);
            if (!is_dir($destAbsDir)) {
                imagedestroy($src);
                throw new ValidationException('No se pudo crear el directorio de uploads.');
            }

            imagepalettetotruecolor($src);
            imagealphablending($src, true);
            imagesavealpha($src, true);

            $id = bin2hex(random_bytes(10));
            $ext = 'webp';
$filename = "{$prefix}_{$id}.{$ext}";
$destAbs  = $destAbsDir . DIRECTORY_SEPARATOR . $filename;

// webp quality 70–82 buen balance
$ok = @imagewebp($src, $destAbs, 78);

            imagedestroy($src);

            if (!$ok || !is_file($destAbs)) {
                throw new ValidationException('No se pudo optimizar/guardar la imagen.');
            }

            return rtrim($relativeDir, '/\\') . '/' . $filename; // path relativo público
        };

        // Perfil actual (para conservar logo/banner si no sube nuevo)
        $stP = $pdo->prepare("SELECT * FROM empresa_perfiles WHERE user_id = :uid LIMIT 1");
        $stP->execute([':uid' => $uid]);
        $perfilActual = $stP->fetch(PDO::FETCH_ASSOC) ?: [];

        $logoPathActual   = (string)($perfilActual['logo_path'] ?? '');
        $bannerPathActual = (string)($perfilActual['banner_path'] ?? '');

        // Normalizar legacy /public/...
        if ($logoPathActual !== '' && str_starts_with($logoPathActual, '/public/')) $logoPathActual = substr($logoPathActual, 7);
        if ($bannerPathActual !== '' && str_starts_with($bannerPathActual, '/public/')) $bannerPathActual = substr($bannerPathActual, 7);

        $logoPath   = $logoPathActual !== '' ? $logoPathActual : null;
        $bannerPath = $bannerPathActual !== '' ? $bannerPathActual : null;

        $publicDir = dirname(__DIR__, 2) . '/public';

        if (!empty($_FILES['logo']) && is_array($_FILES['logo']) && (int)($_FILES['logo']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK) {
            $logoPath = $processImage($_FILES['logo'], $publicDir, '/upload/empresas/logo', 'logo');
        }

        if (!empty($_FILES['banner']) && is_array($_FILES['banner']) && (int)($_FILES['banner']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK) {
            $bannerPath = $processImage($_FILES['banner'], $publicDir, '/upload/empresas/banner', 'banner');
        }

        // Guardar en DB (users + empresa_perfiles)
        $pdo->beginTransaction();
        try {
            // users.nombre
            $stUp = $pdo->prepare("UPDATE users SET nombre = :n WHERE id = :id LIMIT 1");
            $stUp->execute([':n' => $nombre, ':id' => $uid]);

            // empresa_perfiles upsert
            if (!empty($perfilActual)) {
                $st = $pdo->prepare("
                    UPDATE empresa_perfiles
                    SET
                      telefono = :t,
                      whatsapp = :w,
                      ubicacion = :u,
                      bio = :b,
                      website_url = :web,
                      google_business_url = :gb,
                      logo_path = :lp,
                      banner_path = :bp
                    WHERE user_id = :uid
                    LIMIT 1
                ");
                $st->execute([
                    ':t'   => $telefono !== '' ? $telefono : null,
                    ':w'   => $whatsapp !== '' ? $whatsapp : null,
                    ':u'   => $ubicacion !== '' ? $ubicacion : null,
                    ':b'   => $bio !== '' ? $bio : null,
                    ':web' => $website !== '' ? $website : null,
                    ':gb'  => $googleBusiness !== '' ? $googleBusiness : null,
                    ':lp'  => $logoPath,
                    ':bp'  => $bannerPath,
                    ':uid' => $uid,
                ]);
            } else {
                $st = $pdo->prepare("
                    INSERT INTO empresa_perfiles
                      (user_id, telefono, whatsapp, ubicacion, bio, website_url, google_business_url, logo_path, banner_path)
                    VALUES
                      (:uid, :t, :w, :u, :b, :web, :gb, :lp, :bp)
                ");
                $st->execute([
                    ':uid' => $uid,
                    ':t'   => $telefono !== '' ? $telefono : null,
                    ':w'   => $whatsapp !== '' ? $whatsapp : null,
                    ':u'   => $ubicacion !== '' ? $ubicacion : null,
                    ':b'   => $bio !== '' ? $bio : null,
                    ':web' => $website !== '' ? $website : null,
                    ':gb'  => $googleBusiness !== '' ? $googleBusiness : null,
                    ':lp'  => $logoPath,
                    ':bp'  => $bannerPath,
                ]);
            }

            $pdo->commit();
        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }

        return $this->redirect(base_path() . '/empresa/perfil?ok=1', 302);
    });
});

/* ========= EMPRESA BOOSTS (MONETIZACIÓN) ========= */

$this->mapGet('/empresa/boosts', function (Request $req): Response {
    $uid = (int) AuthService::currentUserId();
    if ($uid <= 0) return Response::html("<h3>401</h3>", 401);

    $pdo = DB::pdo();

    $stU = $pdo->prepare("SELECT id, tipo FROM users WHERE id = :id LIMIT 1");
    $stU->execute([':id' => $uid]);
    $user = $stU->fetch(PDO::FETCH_ASSOC);

    if (!$user || ($user['tipo'] ?? '') !== 'empresa') {
        return Response::html("<h3>403</h3>", 403);
    }

    // (opcional) módulo habilitado
    if (class_exists('ModuleService') && method_exists('ModuleService', 'isEnabled')) {
        if (!ModuleService::isEnabled($uid, 'empresa_boosts')) {
            return Response::html("<h3>Módulo no habilitado</h3>", 403);
        }
    }

    $st = $pdo->query("
        SELECT id, nombre
        FROM categorias_laborales
        WHERE estado = 'activo'
        ORDER BY nombre ASC
    ");
    $categorias = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $boosts  = FeedBoostService::listActiveByActor($uid, 'empresa');
    $sidebar = render_template('partials/sidebar_empresa.php', ['active' => 'boosts']);
    $csrf    = CsrfToken::issue();

    $html = render_template('layouts/panel.php', [
        'pageTitle' => 'Boosts Empresa',
        'topHint'   => 'Publicidad por categorías',
        'sidebar'   => $sidebar,
        'content'   => render_template('empresa/boosts/index.php', [
            'categorias' => $categorias,
            'boosts'     => $boosts,
            'csrf'       => $csrf,
            'ok'         => null,
            'error'      => null,
        ]),
    ]);

    return Response::html($html, 200);
});

$this->mapPost('/empresa/boosts/create', function (Request $req): Response {
    $uid = (int) AuthService::currentUserId();
    if ($uid <= 0) return Response::html("<h3>401</h3>", 401);

    $pdo = DB::pdo();

    $stU = $pdo->prepare("SELECT id, tipo FROM users WHERE id = :id LIMIT 1");
    $stU->execute([':id' => $uid]);
    $user = $stU->fetch(PDO::FETCH_ASSOC);

    if (!$user || ($user['tipo'] ?? '') !== 'empresa') {
        return Response::html("<h3>403</h3>", 403);
    }

    // (opcional) módulo habilitado
    if (class_exists('ModuleService') && method_exists('ModuleService', 'isEnabled')) {
        if (!ModuleService::isEnabled($uid, 'empresa_boosts')) {
            return Response::html("<h3>Módulo no habilitado</h3>", 403);
        }
    }

    $post = $req->post ?? $req->body ?? [];
    CsrfToken::verify((string)($post['_csrf'] ?? ''));

    // (opcional) límite por plan
    if (class_exists('PlanLimitService') && method_exists('PlanLimitService', 'getLimit')) {
        $limit = (int) PlanLimitService::getLimit($uid, 'max_boosts_activos');
        if ($limit > 0) {
            $st = $pdo->prepare("
                SELECT COUNT(*) c
                FROM feed_boosts fb
                WHERE fb.actor_user_id = :u
                  AND fb.actor_tipo = 'empresa'
                  AND fb.estado = 'activo'
                  AND NOW() BETWEEN fb.starts_at AND fb.ends_at
            ");
            $st->execute([':u' => $uid]);
            $c = (int)($st->fetch(PDO::FETCH_ASSOC)['c'] ?? 0);
            if ($c >= $limit) {
                return $this->redirect(base_path() . '/empresa/boosts', 302);
            }
        }
    }

    try {
        FeedBoostService::create([
            'actor_user_id' => $uid,
            'actor_tipo'    => 'empresa',
            'scope'         => (string)($post['scope'] ?? 'para_ti'),
            'categoria_id'  => (int)($post['categoria_id'] ?? 0),
            'score_boost'   => (int)($post['score_boost'] ?? 600),
            'days'          => (int)($post['days'] ?? 7),
        ]);
    } catch (Throwable $e) {
        // silencioso, vuelve al GET
    }

    return $this->redirect(base_path() . '/empresa/boosts', 302);
});

$this->mapPost('/empresa/boosts/disable', function (Request $req): Response {
    $uid = (int) AuthService::currentUserId();
    if ($uid <= 0) return Response::html("<h3>401</h3>", 401);

    $pdo = DB::pdo();

    $stU = $pdo->prepare("SELECT id, tipo FROM users WHERE id = :id LIMIT 1");
    $stU->execute([':id' => $uid]);
    $user = $stU->fetch(PDO::FETCH_ASSOC);

    if (!$user || ($user['tipo'] ?? '') !== 'empresa') {
        return Response::html("<h3>403</h3>", 403);
    }

    // (opcional) módulo habilitado
    if (class_exists('ModuleService') && method_exists('ModuleService', 'isEnabled')) {
        if (!ModuleService::isEnabled($uid, 'empresa_boosts')) {
            return Response::html("<h3>Módulo no habilitado</h3>", 403);
        }
    }

    $post = $req->post ?? $req->body ?? [];
    CsrfToken::verify((string)($post['_csrf'] ?? ''));

    $id = (int)($post['id'] ?? 0);
    if ($id > 0) {
        FeedBoostService::disable($id, $uid, 'empresa');
    }

    return $this->redirect(base_path() . '/empresa/boosts', 302);
});

/* ========= RRHH ========= */

$this->mapGet('/rrhh', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.panel.access', function (Request $req): Response {
        $uid  = (int)AuthService::currentUserId();
        $csrf = CsrfToken::issue();

        $panel = render_template('layouts/panel_rrhh.php', [
            'pageTitle' => 'RRHH • Panel',
            'topHint'   => 'Gestión profesional',
            'currentUserId' => $uid,
            'sidebar'   => render_template('partials/sidebar_rrhh.php', ['active' => 'home']),
            'content'   => render_template('rrhh/dashboard.php', [
                'csrf' => $csrf,
            ]),
        ]);

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'RRHH • Panel',
            'content'   => $panel,
        ]);

        return Response::html($html, 200);
    });
});

$this->mapGet('/rrhh/perfil', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.profile.manage', function (Request $req): Response {
        $uid  = (int)AuthService::currentUserId();
        $csrf = CsrfToken::issue();

        $profile = RrhhProfileService::get($uid);

        $panel = render_template('layouts/panel_rrhh.php', [
            'pageTitle' => 'RRHH • Mi perfil',
            'topHint'   => 'Perfil profesional y vínculo a empresa',
            'currentUserId' => $uid,
            'sidebar'   => render_template('partials/sidebar_rrhh.php', ['active' => 'perfil']),
            'content'   => render_template('rrhh/perfil.php', [
                'csrf'    => $csrf,
                'profile' => $profile,
                'error'   => null,
                'ok'      => null,
            ]),
        ]);

        $html = render_template('layouts/app.php', ['pageTitle'=>'RRHH • Mi perfil','content'=>$panel]);
        return Response::html($html, 200);
    });
});

$this->mapPost('/rrhh/perfil', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.profile.manage', function (Request $req): Response {
        $uid  = (int)AuthService::currentUserId();
        $post = $req->post ?? $req->body ?? [];
        CsrfToken::verify((string)($post['_csrf'] ?? ''));

        // Foto opcional (si no querés subir aquí, lo dejamos para después)
        // Para no romper tu proyecto, no obligo ImageUpload: solo texto.

        try {
            RrhhProfileService::update($uid, $post);
            $ok = 'Perfil actualizado.';
            $err = null;
        } catch (Throwable $e) {
            $ok = null;
            $err = 'No se pudo guardar: ' . $e->getMessage();
        }

        $csrf = CsrfToken::issue();
        $profile = RrhhProfileService::get($uid);

        $panel = render_template('layouts/panel_rrhh.php', [
            'pageTitle' => 'RRHH • Mi perfil',
            'topHint'   => 'Perfil profesional y vínculo a empresa',
            'currentUserId' => $uid,
            'sidebar'   => render_template('partials/sidebar_rrhh.php', ['active' => 'perfil']),
            'content'   => render_template('rrhh/perfil.php', [
                'csrf'    => $csrf,
                'profile' => $profile,
                'error'   => $err,
                'ok'      => $ok,
            ]),
        ]);

        $html = render_template('layouts/app.php', ['pageTitle'=>'RRHH • Mi perfil','content'=>$panel]);
        return Response::html($html, 200);
    });
});

$this->mapGet('/rrhh/skills', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.skills.manage', function (Request $req): Response {
        $uid  = (int)AuthService::currentUserId();
        $csrf = CsrfToken::issue();

        $items = RrhhSkillService::listAll($uid);

        $panel = render_template('layouts/panel_rrhh.php', [
            'pageTitle' => 'RRHH • Skills',
            'topHint'   => 'Porcentajes por categorías',
            'currentUserId' => $uid,
            'sidebar'   => render_template('partials/sidebar_rrhh.php', ['active' => 'skills']),
            'content'   => render_template('rrhh/skills.php', [
                'csrf'  => $csrf,
                'items' => $items,
                'error' => null,
                'ok'    => null,
            ]),
        ]);

        $html = render_template('layouts/app.php', ['pageTitle'=>'RRHH • Skills','content'=>$panel]);
        return Response::html($html, 200);
    });
});

$this->mapPost('/rrhh/skills', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.skills.manage', function (Request $req): Response {
        $uid  = (int)AuthService::currentUserId();
        $post = $req->post ?? $req->body ?? [];
        CsrfToken::verify((string)($post['_csrf'] ?? ''));

        $action = (string)($post['action'] ?? 'create');

        try {
            if ($action === 'delete') {
                RrhhSkillService::delete((int)($post['id'] ?? 0), $uid);
            } elseif ($action === 'update') {
                RrhhSkillService::update((int)($post['id'] ?? 0), $uid, $post);
            } else {
                RrhhSkillService::create($uid, $post);
            }
            $ok = 'Cambios guardados.';
            $err = null;
        } catch (Throwable $e) {
            $ok = null;
            $err = 'No se pudo guardar: ' . $e->getMessage();
        }

        $csrf = CsrfToken::issue();
        $items = RrhhSkillService::listAll($uid);

        $panel = render_template('layouts/panel_rrhh.php', [
            'pageTitle' => 'RRHH • Skills',
            'topHint'   => 'Porcentajes por categorías',
            'currentUserId' => $uid,
            'sidebar'   => render_template('partials/sidebar_rrhh.php', ['active' => 'skills']),
            'content'   => render_template('rrhh/skills.php', [
                'csrf'  => $csrf,
                'items' => $items,
                'error' => $err,
                'ok'    => $ok,
            ]),
        ]);

        $html = render_template('layouts/app.php', ['pageTitle'=>'RRHH • Skills','content'=>$panel]);
        return Response::html($html, 200);
    });
});

$this->mapGet('/rrhh/anuncios', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.anuncios.manage', function (Request $req): Response {
        $uid  = (int)AuthService::currentUserId();
        $csrf = CsrfToken::issue();

        $items = RrhhAnuncioService::listMine($uid);

        $panel = render_template('layouts/panel_rrhh.php', [
            'pageTitle' => 'RRHH • Anuncios',
            'topHint'   => 'Servicios y comunicados (no vacantes)',
            'currentUserId' => $uid,
            'sidebar'   => render_template('partials/sidebar_rrhh.php', ['active' => 'anuncios']),
            'content'   => render_template('rrhh/anuncios/index.php', [
                'csrf'  => $csrf,
                'items' => $items,
            ]),
        ]);

        $html = render_template('layouts/app.php', ['pageTitle'=>'RRHH • Anuncios','content'=>$panel]);
        return Response::html($html, 200);
    });
});

$this->mapGet('/rrhh/anuncios/editar', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.anuncios.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $id  = (int)($req->query['id'] ?? 0);
        $csrf = CsrfToken::issue();

        $row = $id > 0 ? RrhhAnuncioService::getMine($id, $uid) : [];

        $panel = render_template('layouts/panel_rrhh.php', [
            'pageTitle' => $id > 0 ? 'RRHH • Editar anuncio' : 'RRHH • Nuevo anuncio',
            'topHint'   => 'Publica anuncios (no vacantes)',
            'currentUserId' => $uid,
            'sidebar'   => render_template('partials/sidebar_rrhh.php', ['active' => 'anuncios']),
            'content'   => render_template('rrhh/anuncios/form.php', [
                'csrf'  => $csrf,
                'row'   => $row,
                'error' => null,
                'ok'    => null,
            ]),
        ]);

        $html = render_template('layouts/app.php', ['pageTitle'=>'RRHH • Anuncios','content'=>$panel]);
        return Response::html($html, 200);
    });
});

$this->mapPost('/rrhh/anuncios/editar', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.anuncios.manage', function (Request $req): Response {
        $uid  = (int)AuthService::currentUserId();
        $post = $req->post ?? $req->body ?? [];
        CsrfToken::verify((string)($post['_csrf'] ?? ''));

        $id = (int)($post['id'] ?? 0);

        try {
            if ((string)($post['action'] ?? '') === 'delete') {
                RrhhAnuncioService::delete($id, $uid);
                return $this->redirect(base_path() . '/rrhh/anuncios', 302);
            }

            if ($id > 0) {
                RrhhAnuncioService::update($id, $uid, $post);
            } else {
                $id = RrhhAnuncioService::create($uid, $post);
            }
            $ok = 'Guardado.';
            $err = null;
        } catch (Throwable $e) {
            $ok = null;
            $err = 'No se pudo guardar: ' . $e->getMessage();
        }

        $csrf = CsrfToken::issue();
        $row = $id > 0 ? RrhhAnuncioService::getMine($id, $uid) : [];

        $panel = render_template('layouts/panel_rrhh.php', [
            'pageTitle' => $id > 0 ? 'RRHH • Editar anuncio' : 'RRHH • Nuevo anuncio',
            'topHint'   => 'Publica anuncios (no vacantes)',
            'currentUserId' => $uid,
            'sidebar'   => render_template('partials/sidebar_rrhh.php', ['active' => 'anuncios']),
            'content'   => render_template('rrhh/anuncios/form.php', [
                'csrf'  => $csrf,
                'row'   => $row,
                'error' => $err,
                'ok'    => $ok,
            ]),
        ]);

        $html = render_template('layouts/app.php', ['pageTitle'=>'RRHH • Anuncios','content'=>$panel]);
        return Response::html($html, 200);
    });
});

$this->mapGet('/rrhh/boosts', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.boost.manage', function (Request $req): Response {
        $uid  = (int)AuthService::currentUserId();
        $csrf = CsrfToken::issue();

        $items = FeedBoostService::listActiveByActor($uid, 'rrhh');
        $cats = CategoriaLaboralService::listForFeed(350);

        $panel = render_template('layouts/panel_rrhh.php', [
            'pageTitle' => 'RRHH • Boost',
            'topHint'   => 'Módulo monetizable',
            'currentUserId' => $uid,
            'sidebar'   => render_template('partials/sidebar_rrhh.php', ['active' => 'boosts']),
            'content'   => render_template('rrhh/boosts.php', [
                'csrf'  => $csrf,
                'items' => $items,
                'cats'  => $cats,
            ]),
        ]);

        $html = render_template('layouts/app.php', ['pageTitle'=>'RRHH • Boost','content'=>$panel]);
        return Response::html($html, 200);
    });
});

$this->mapPost('/rrhh/boosts/create', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.boost.manage', function (Request $req): Response {
        $uid  = (int)AuthService::currentUserId();
        $post = $req->post ?? $req->body ?? [];
        CsrfToken::verify((string)($post['_csrf'] ?? ''));

        // (opcional) módulo habilitado por plan
        if (class_exists('ModuleService') && method_exists('ModuleService', 'isEnabled')) {
            if (!ModuleService::isEnabled($uid, 'rrhh_boosts')) {
                return Response::html("<h3>Módulo no habilitado</h3>", 403);
            }
        }

        try {
            FeedBoostService::create([
                'actor_user_id' => $uid,
                'actor_tipo'    => 'rrhh',
                'scope'         => (string)($post['scope'] ?? 'para_ti'),
                'categoria_id'  => (int)($post['categoria_id'] ?? 0),
                'score_boost'   => (int)($post['score_boost'] ?? 600),
                'days'          => (int)($post['days'] ?? 7),
            ]);
        } catch (Throwable $e) {
            // silencioso
        }

        return $this->redirect(base_path() . '/rrhh/boosts', 302);
    });
});

$this->mapPost('/rrhh/boosts/disable', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.boost.manage', function (Request $req): Response {
        $uid  = (int)AuthService::currentUserId();
        $post = $req->post ?? $req->body ?? [];
        CsrfToken::verify((string)($post['_csrf'] ?? ''));

        $id = (int)($post['id'] ?? 0);
        if ($id > 0) {
            FeedBoostService::disable($id, $uid, 'rrhh');
        }

        return $this->redirect(base_path() . '/rrhh/boosts', 302);
    });
});

$this->mapGet('/rrhh/notificaciones', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.panel.access', function (Request $req): Response {
        $uid  = (int)AuthService::currentUserId();
        $csrf = CsrfToken::issue();

        $items = [];
        if (class_exists('NotificationService') && method_exists('NotificationService', 'listForUser')) {
            $items = NotificationService::listForUser($uid, 80);
        }

        $panel = render_template('layouts/panel_rrhh.php', [
            'pageTitle' => 'RRHH • Notificaciones',
            'topHint'   => 'Alertas del sistema',
            'currentUserId' => $uid,
            'sidebar'   => render_template('partials/sidebar_rrhh.php', ['active' => 'notifs']),
            'content'   => render_template('rrhh/notificaciones.php', [
                'csrf'  => $csrf,
                'items' => $items,
            ]),
        ]);

        $html = render_template('layouts/app.php', ['pageTitle'=>'RRHH • Notificaciones','content'=>$panel]);
        return Response::html($html, 200);
    });
});

$this->mapGet('/rrhh/vacantes', function (Request $req): Response {
    return $this->gateRRHH($req, 'rrhh.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $empresaId = RrhhProfileService::getLinkedEmpresaId($uid);

        $panel = render_template('layouts/panel_rrhh.php', [
            'pageTitle' => 'RRHH • Vacantes',
            'topHint'   => 'Requiere vínculo a empresa',
            'currentUserId' => $uid,
            'sidebar'   => render_template('partials/sidebar_rrhh.php', ['active' => 'vacantes']),
            'content'   => render_template('rrhh/vacantes.php', [
                'empresaId' => $empresaId,
            ]),
        ]);

        $html = render_template('layouts/app.php', ['pageTitle'=>'RRHH • Vacantes','content'=>$panel]);
        return Response::html($html, 200);
    });
});

// Perfil público RRHH
$this->mapGet('/rrhh/ver', function (Request $req): Response {
    $id = (int)($req->query['id'] ?? 0);
    if ($id <= 0) return Response::html("<h3>404</h3>", 404);

    $p = RrhhProfileService::publicById($id);
    if (!$p) return Response::html("<h3>404</h3>", 404);

    // Robustez: si por alguna razón el service no trae skills, las cargamos aquí.
    if (!isset($p['skills']) || !is_array($p['skills'])) {
        if (class_exists('RrhhSkillService') && method_exists('RrhhSkillService', 'listPublicByUserId')) {
            try { $p['skills'] = RrhhSkillService::listPublicByUserId($id); } catch (Throwable $e) { $p['skills'] = []; }
        } else {
            // fallback directo DB si existe tabla rrhh_skills
            $p['skills'] = [];
            if (class_exists('DB')) {
                try {
                    $pdo = DB::pdo();
                    $st = $pdo->prepare("
                        SELECT nombre, categoria, porcentaje
                        FROM rrhh_skills
                        WHERE rrhh_user_id = :uid AND estado = 'activo'
                        ORDER BY orden ASC, id ASC
                    ");
                    $st->execute([':uid' => $id]);
                    $p['skills'] = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
                } catch (Throwable $e) {
                    $p['skills'] = [];
                }
            }
        }
    }

    $html = render_template('layouts/app.php', [
        'pageTitle' => 'RRHH • Perfil',
        'content'   => render_template('rrhh/publico.php', ['p' => $p]),
    ]);

    return Response::html($html, 200);
});

/* ========= POSTULACIONES ========= */

$this->mapGet('/postulante/postulaciones', function (Request $req): Response {
    return $this->gatePostulante($req, function (Request $req): Response {

        $uid = (int)AuthService::currentUserId();

        $items = PostulacionService::listForPostulante($uid);
        $csrf  = CsrfToken::issue();

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'Mis postulaciones',
            'content'   => render_template('postulante/postulaciones/index.php', [
                'items' => $items,
                'error' => null,
                'csrf'  => $csrf,
            ]),
        ]);

        return Response::html($html, 200);
    });
});

/* ========= FEED (PRIVADO) ========= */
$this->mapGet('/feed', function (Request $req): Response {

    $uid  = (int)AuthService::currentUserId();
    $tipo = (string)AuthService::currentUserType();

    $q = trim((string)($req->query['q'] ?? ''));
    $csrf = CsrfToken::issue();

    if ($tipo === 'empresa') {
        $out = FeedService::feedForEmpresa($uid, $q, 30, 0);

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'Feed (Empresa)',
            'content'   => render_template('front/feed.php', [
                'userId' => $uid,
                'tipo'   => $tipo,
                'q'      => $q,
                'items'  => $out['items'],
                'stats'  => $out['stats'],
                'csrf'   => $csrf,
            ]),
        ]);
        return Response::html($html, 200);
    }

    // postulante o admin
    $only = (int)($req->query['only'] ?? 0); // 1 = solo matching

    $catsQ = $req->query['cat'] ?? [];
    if (!is_array($catsQ)) $catsQ = [$catsQ];

    $selectedIds = array_values(array_unique(array_filter(array_map('intval', $catsQ), fn($v) => $v > 0)));

    // si no hay filtro explícito, usar los intereses guardados
    if (empty($selectedIds)) {
        $selectedIds = (array)user_selected_categoria_ids($uid);
        $selectedIds = array_values(array_unique(array_filter(array_map('intval', $selectedIds), fn($v) => $v > 0)));
    }

    $filters = [
        'only_matching' => ($only === 1 ? 1 : 0),
        'categoria_ids' => $selectedIds, // SIEMPRE array
    ];

    $out = FeedService::feedForPostulante($uid, $q, 30, 0, $filters);

    $ideales = [];
    try {
        $ideales = FeedService::idealesParaTi($uid, $selectedIds, 5);
    } catch (Throwable $e) {
        $ideales = [];
    }

    $cats = [];
    try {
        $cats = CategoriaLaboralService::listForFeed(350);
    } catch (Throwable $e) {
        $cats = [];
    }

    $html = render_template('layouts/app.php', [
        'pageTitle' => 'Feed',
        'content'   => render_template('front/feed.php', [
            'userId'   => $uid,
            'tipo'     => $tipo,
            'q'        => $q,
            'only'     => $only,
            'cats'     => $cats,
            'catSel'   => $selectedIds,
            'ideales'  => $ideales,
            'items'    => $out['items'],
            'stats'    => $out['stats'],
            'csrf'     => $csrf,
        ]),
    ]);

    return Response::html($html, 200);
});

/* ========= PERFIL (POSTULANTE) ========= */
$this->mapGet('/perfil', function (Request $req): Response {
    $tipo = (string)AuthService::currentUserType();
    if ($tipo === 'empresa') return $this->redirect(base_path() . '/empresa/perfil', 302);
    return $this->redirect(base_path() . '/postulante/perfil', 302);
});

$this->mapGet('/postulante/perfil', function (Request $req): Response {
    return $this->gatePostulante($req, function (Request $req): Response {

        $uid = (int)AuthService::currentUserId();

        $pdo = DB::pdo();
        $u = $pdo->prepare("SELECT id,email,nombre,tipo FROM users WHERE id=:id LIMIT 1");
        $u->execute(['id' => $uid]);
        $user = (array)$u->fetch(PDO::FETCH_ASSOC);

        $perfil = PostulanteProfileService::getOrCreate($uid);
        $cv     = CvService::latestActiveForUser($uid);
        $csrf   = CsrfToken::issue();

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'Mi perfil',
            'content'   => render_template('postulante/perfil.php', [
                'user'   => $user,
                'perfil' => $perfil,
                'cv'     => $cv,
                'csrf'   => $csrf,
                'error'  => null,
                'ok'     => null,
            ]),
        ]);

        return Response::html($html, 200);
    });
});

$this->mapPost('/postulante/perfil', function (Request $req): Response {
    return $this->gatePostulante($req, function (Request $req): Response {

        $uid = (int)AuthService::currentUserId();

        // Requisito: para construir perfil profesional y habilitar match/boost, debe seleccionar al menos 1 categoría.
        $selectedIds = [];
        try {
            $selectedIds = user_selected_categoria_ids($uid);
        } catch (Throwable $e) {
            $selectedIds = [];
        }

        $data = [
            'nombre'             => trim((string)($req->post['nombre'] ?? '')),
            'telefono'           => trim((string)($req->post['telefono'] ?? '')),
            'ubicacion'          => trim((string)($req->post['ubicacion'] ?? '')),
            'titulo_profesional' => trim((string)($req->post['titulo_profesional'] ?? '')),
            'bio'                => trim((string)($req->post['bio'] ?? '')),
            'linkedin_url'       => trim((string)($req->post['linkedin_url'] ?? '')),
            'portfolio_url'      => trim((string)($req->post['portfolio_url'] ?? '')),
        ];

        if (!$selectedIds) {
            // Renderiza el mismo perfil con error y preserva lo ingresado (sin guardar).
            $pdo = DB::pdo();
            $u = $pdo->prepare("SELECT id,email,nombre,tipo FROM users WHERE id=:id LIMIT 1");
            $u->execute(['id' => $uid]);
            $user = $u->fetch(PDO::FETCH_ASSOC) ?: ['id'=>$uid,'email'=>'','nombre'=>'','tipo'=>'postulante'];

            $perfil = PostulanteProfileService::getOrCreate($uid);

            // sobreescribe con lo que el usuario intentó guardar (UI fiel)
            foreach ($data as $k => $v) {
                $perfil[$k] = $v;
            }

            $cv   = CvService::latestActiveForUser($uid);
            $csrf = CsrfToken::issue();

            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Mi perfil',
                'content'   => render_template('postulante/perfil.php', [
                    'user'   => $user,
                    'perfil' => $perfil,
                    'cv'     => $cv,
                    'csrf'   => $csrf,
                    'error'  => 'Antes de guardar tu perfil profesional, seleccioná al menos 1 categoría (roles/intereses).',
                    'ok'     => null,
                ]),
            ]);

            return Response::html($html, 422);
        }

        PostulanteProfileService::update($uid, $data);

        return $this->redirect(base_path() . '/postulante/panel', 302);
    });
});

// FEED AJAX: reacciones (JSON)  ✅ FIX: columna "reaccion" + CSRF rotativo + conteos reales
$this->mapPost('/feed/react', function (Request $req): Response {
    $uid = (int)AuthService::currentUserId();
    if ($uid <= 0) return Response::json(['ok' => false, 'error' => 'auth'], 401);

    try {
        CsrfToken::verify((string)($req->post['_csrf'] ?? ''));
    } catch (Throwable $e) {
        return Response::json(['ok'=>false,'error'=>'csrf','csrf'=>CsrfToken::issue()], 419);
    }

    $kind   = (string)($req->post['kind'] ?? 'vacante');
    $itemId = (int)($req->post['item_id'] ?? ($req->post['vacante_id'] ?? 0));
    $reaction = (string)($req->post['reaction'] ?? ($req->post['reaccion'] ?? ''));

    $allowed = ['like','recommend','excellent','bad'];
    if ($itemId <= 0) return Response::json(['ok'=>false,'error'=>'item_id','csrf'=>CsrfToken::issue()], 422);
    if (!in_array($reaction, $allowed, true)) return Response::json(['ok'=>false,'error'=>'reaction','csrf'=>CsrfToken::issue()], 422);

    $pdo = DB::pdo();

    // Resolver tabla/columna según kind
    if ($kind === 'rrhh_anuncio') {
        $tbl = 'rrhh_anuncio_reacciones';
        $col = 'anuncio_id';
    } else {
        $kind = 'vacante';
        $tbl = 'vacante_reacciones';
        $col = 'vacante_id';
    }

    $st = $pdo->prepare("SELECT reaccion FROM {$tbl} WHERE {$col}=:id AND user_id=:u LIMIT 1");
    $st->execute(['id'=>$itemId, 'u'=>$uid]);
    $current = (string)($st->fetchColumn() ?: '');

    if ($current !== '' && $current === $reaction) {
        $del = $pdo->prepare("DELETE FROM {$tbl} WHERE {$col}=:id AND user_id=:u");
        $del->execute(['id'=>$itemId, 'u'=>$uid]);
        $my = '';
    } elseif ($current !== '' && $current !== $reaction) {
        $up = $pdo->prepare("UPDATE {$tbl} SET reaccion=:r WHERE {$col}=:id AND user_id=:u");
        $up->execute(['r'=>$reaction, 'id'=>$itemId, 'u'=>$uid]);
        $my = $reaction;
    } else {
        $ins = $pdo->prepare("INSERT INTO {$tbl} ({$col}, user_id, reaccion) VALUES (:id, :u, :r)");
        $ins->execute(['id'=>$itemId, 'u'=>$uid, 'r'=>$reaction]);
        $my = $reaction;
    }

    $cnt = $pdo->prepare("
        SELECT
          SUM(CASE WHEN reaccion='like' THEN 1 ELSE 0 END) AS c_like,
          SUM(CASE WHEN reaccion='recommend' THEN 1 ELSE 0 END) AS c_recommend,
          SUM(CASE WHEN reaccion='excellent' THEN 1 ELSE 0 END) AS c_excellent,
          SUM(CASE WHEN reaccion='bad' THEN 1 ELSE 0 END) AS c_bad
        FROM {$tbl}
        WHERE {$col}=:id
    ");
    $cnt->execute(['id'=>$itemId]);
    $row = (array)$cnt->fetch(PDO::FETCH_ASSOC);

    $counts = [
        'like'      => (int)($row['c_like'] ?? 0),
        'recommend' => (int)($row['c_recommend'] ?? 0),
        'excellent' => (int)($row['c_excellent'] ?? 0),
        'bad'       => (int)($row['c_bad'] ?? 0),
        'share'     => 0,
    ];

    // shares
    try {
        if ($kind === 'rrhh_anuncio') {
            $s = $pdo->prepare("SELECT COUNT(*) FROM rrhh_anuncio_compartidos WHERE anuncio_id=:id");
        } else {
            $s = $pdo->prepare("SELECT COUNT(*) FROM vacante_compartidos WHERE vacante_id=:id");
        }
        $s->execute(['id'=>$itemId]);
        $counts['share'] = (int)$s->fetchColumn();
    } catch (Throwable $e) { }

    return Response::json([
        'ok'          => true,
        'my_reaction' => $my,
        'counts'      => $counts,
        'csrf'        => CsrfToken::issue(),
    ], 200);
});

// FEED AJAX: compartir (JSON) ✅ FIX: CSRF rotativo + conteos + no rompe si no existe tabla
$this->mapPost('/feed/share', function (Request $req): Response {
    $uid = (int)AuthService::currentUserId();
    if ($uid <= 0) return Response::json(['ok' => false, 'error' => 'auth'], 401);

    try {
        CsrfToken::verify((string)($req->post['_csrf'] ?? ''));
    } catch (Throwable $e) {
        return Response::json([
            'ok'   => false,
            'error'=> 'csrf',
            'csrf' => CsrfToken::issue(),
        ], 419);
    }

    $vacanteId = (int)($req->post['vacante_id'] ?? 0);
    if ($vacanteId <= 0) {
        return Response::json(['ok' => false, 'error' => 'vacante_id', 'csrf' => CsrfToken::issue()], 422);
    }

    $pdo = DB::pdo();

    // registra share (si existe tabla). Si no existe, igual devolvemos conteos sin romper.
    try {
        // evita spam duplicado: 1 share por usuario por vacante (ajústalo si querés contar múltiple)
        $chk = $pdo->prepare("SELECT id FROM vacante_compartidos WHERE vacante_id=:v AND user_id=:u LIMIT 1");
        $chk->execute(['v' => $vacanteId, 'u' => $uid]);
        $exists = (int)($chk->fetchColumn() ?: 0);

        if ($exists <= 0) {
            $ins = $pdo->prepare("INSERT INTO vacante_compartidos (vacante_id, user_id) VALUES (:v, :u)");
            $ins->execute(['v' => $vacanteId, 'u' => $uid]);
        }
    } catch (Throwable $e) {
        // si tu sistema usa otra tabla para shares, aquí no reventamos
    }

    // conteos de reacciones
    $cnt = $pdo->prepare("
        SELECT
          SUM(CASE WHEN reaccion='like' THEN 1 ELSE 0 END) AS c_like,
          SUM(CASE WHEN reaccion='recommend' THEN 1 ELSE 0 END) AS c_recommend,
          SUM(CASE WHEN reaccion='excellent' THEN 1 ELSE 0 END) AS c_excellent,
          SUM(CASE WHEN reaccion='bad' THEN 1 ELSE 0 END) AS c_bad
        FROM vacante_reacciones
        WHERE vacante_id=:v
    ");
    $cnt->execute(['v' => $vacanteId]);
    $row = (array)$cnt->fetch(PDO::FETCH_ASSOC);

    $counts = [
        'like'      => (int)($row['c_like'] ?? 0),
        'recommend' => (int)($row['c_recommend'] ?? 0),
        'excellent' => (int)($row['c_excellent'] ?? 0),
        'bad'       => (int)($row['c_bad'] ?? 0),
        'share'     => 0,
    ];

    // conteo shares
    try {
        $s = $pdo->prepare("SELECT COUNT(*) FROM vacante_compartidos WHERE vacante_id=:v");
        $s->execute(['v' => $vacanteId]);
        $counts['share'] = (int)$s->fetchColumn();
    } catch (Throwable $e) {}

    return Response::json([
        'ok'     => true,
        'counts' => $counts,
        'csrf'   => CsrfToken::issue(), // ✅ token nuevo
    ], 200);
});

/* ========= ROLES / CATEGORÍAS (POSTULANTE) ========= */
$this->mapGet('/postulante/roles', function (Request $req): Response {
    return $this->gatePostulante($req, function (Request $req): Response {

        $uid  = (int)AuthService::currentUserId();
        $csrf = CsrfToken::issue();

        $cats = [];
        try {
            // listado para UI (limit alto pero razonable)
            $cats = CategoriaLaboralService::listForFeed(800);
        } catch (Throwable $e) {
            $cats = [];
        }

        // seleccion guardada del usuario
        $selectedIds = [];
        try {
            $selectedIds = user_selected_categoria_ids($uid);
        } catch (Throwable $e) {
            $selectedIds = [];
        }

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'Mis roles / intereses',
            'content'   => render_template('postulante/roles.php', [
                'cats'   => $cats,
                'sel'    => $selectedIds,
                'csrf'   => $csrf,
                'error'  => null,
                'ok'     => null,
            ]),
        ]);

        return Response::html($html, 200);
    });
});

$this->mapPost('/postulante/roles', function (Request $req): Response {
    return $this->gatePostulante($req, function (Request $req): Response {

        $uid = (int)AuthService::currentUserId();

        // CSRF
        $token = (string)($req->post['_csrf'] ?? '');
        CsrfToken::verify($token);

        $roles = $req->post['roles'] ?? [];
        if (!is_array($roles)) $roles = [];

        // normaliza IDs
        $ids = array_values(array_filter(array_map('intval', $roles), fn($v) => $v > 0));
        $ids = array_slice(array_unique($ids), 0, 25); // hard cap (anti abuso)

        // guarda selección
        CategoriaLaboralService::saveUserSelections($uid, $ids);

        // luego el feed usa esto por defecto cuando no mandas ?cat=
        return $this->redirect(base_path() . '/feed', 302);
    });
});

/* ========= CV (POSTULANTE) ========= */
$this->mapPost('/postulante/cv/subir', function (Request $req): Response {
    return $this->gatePostulante($req, function (Request $req): Response {

        $uid = (int)AuthService::currentUserId();
        $file = $_FILES['cv'] ?? null;
        if (!is_array($file)) {
            return Response::html("<h3>Archivo requerido</h3>", 422);
        }

        try {
            CvService::upload($uid, $file, $GLOBALS['APP_CONFIG']);
        } catch (ValidationException $e) {
            $perfil = PostulanteProfileService::getOrCreate($uid);

            $pdo = DB::pdo();
            $u = $pdo->prepare("SELECT id,email,nombre,tipo FROM users WHERE id=:id LIMIT 1");
            $u->execute(['id' => $uid]);
            $user = (array)$u->fetch(PDO::FETCH_ASSOC);

            $cv   = CvService::latestActiveForUser($uid);
            $csrf = CsrfToken::issue();

            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Mi perfil',
                'content'   => render_template('postulante/perfil.php', [
                    'user'   => $user,
                    'perfil' => $perfil,
                    'cv'     => $cv,
                    'csrf'   => $csrf,
                    'error'  => $e->getMessage(),
                    'ok'     => null,
                ]),
            ]);
            return Response::html($html, 422);
        }

        return $this->redirect(base_path() . '/postulante/perfil', 302);
    });
});

$this->mapPost('/postulante/cv/eliminar', function (Request $req): Response {
    return $this->gatePostulante($req, function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        CvService::deleteActive($uid);
        return $this->redirect(base_path() . '/postulante/perfil', 302);
    });
});

/* ========= CV DOWNLOAD (EMPRESA, dueño de la vacante) ========= */
$this->mapGet('/empresa/postulaciones/cv', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.apps.manage', function (Request $req): Response {

        $empresaUid = (int)AuthService::currentUserId();
        $postId = (int)($req->query['postulacion_id'] ?? 0);
        if ($postId <= 0) return Response::html("<h3>Solicitud inválida</h3>", 422);

        $pdo = DB::pdo();

        // Verificar ownership: postulacion -> vacante -> empresa_user_id
        $sql = "SELECT p.postulante_user_id, p.vacante_id
                FROM postulaciones p
                JOIN vacantes v ON v.id = p.vacante_id
                WHERE p.id = :pid AND v.empresa_user_id = :eid
                LIMIT 1";
        $st = $pdo->prepare($sql);
        $st->execute(['pid' => $postId, 'eid' => $empresaUid]);
        $row = $st->fetch(PDO::FETCH_ASSOC);
        if (!$row) return Response::html("<h3>No autorizado</h3>", 403);

        $postulanteId = (int)$row['postulante_user_id'];
        $cv = CvService::latestActiveForUser($postulanteId);
        if (!$cv) return Response::html("<h3>El postulante no tiene CV</h3>", 404);

        return CvService::sendFile(
            (string)$cv['stored_path'],
            (string)$cv['original_name'],
            (string)$cv['mime']
        );
    });
});

/* ========= ADMIN: DASHBOARD ========= */
        $this->mapGet('/admin', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.access', function (Request $req): Response {
                $html = $this->renderAdminLayout(
                    'Dashboard',
                    render_template('admin/dashboard.php', [])
                );
                return Response::html($html, 200);
            });
        });

        /* ========= ADMIN: MÓDULOS ========= */
        $this->mapGet('/admin/modules', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $items = AdminModuleService::listAll();

                $html = $this->renderAdminLayout(
                    'Módulos',
                    render_template('admin/modules/index.php', [
                        'items' => $items,
                        'error' => null,
                    ])
                );
                return Response::html($html, 200);
            });
        });

        $this->mapGet('/admin/modules/create', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $html = $this->renderAdminLayout(
                    'Crear módulo',
                    render_template('admin/modules/create.php', [
                        'error' => null,
                        'old'   => ['code' => '', 'nombre' => '', 'estado' => 'activo'],
                    ])
                );
                return Response::html($html, 200);
            });
        });

        $this->mapPost('/admin/modules/create', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $old = [
                    'code'   => trim((string)($req->post['code'] ?? '')),
                    'nombre' => trim((string)($req->post['nombre'] ?? '')),
                    'estado' => (string)($req->post['estado'] ?? 'activo'),
                ];

                try {
                    AdminModuleService::create($old['code'], $old['nombre'], $old['estado']);
                    return $this->redirect(base_path() . '/admin/modules', 302);

                } catch (ValidationException $e) {
                    $html = $this->renderAdminLayout(
                        'Crear módulo',
                        render_template('admin/modules/create.php', [
                            'error' => $e->getMessage(),
                            'old'   => $old,
                        ])
                    );
                    return Response::html($html, 422);
                }
            });
        });

        $this->mapGet('/admin/modules/edit', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $id = (int)($req->query['id'] ?? 0);
                $row = AdminModuleService::find($id);

                if (!$row) {
                    return Response::html("<h3>No existe</h3>", 404);
                }

                $html = $this->renderAdminLayout(
                    'Editar módulo',
                    render_template('admin/modules/edit.php', [
                        'error' => null,
                        'row'   => $row,
                    ])
                );
                return Response::html($html, 200);
            });
        });

        $this->mapPost('/admin/modules/edit', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $id = (int)($req->post['id'] ?? 0);
                $row = AdminModuleService::find($id);
                if (!$row) return Response::html("<h3>No existe</h3>", 404);

                $new = [
                    'id'     => $id,
                    'code'   => trim((string)($req->post['code'] ?? '')),
                    'nombre' => trim((string)($req->post['nombre'] ?? '')),
                    'estado' => (string)($req->post['estado'] ?? 'activo'),
                ];

                try {
                    AdminModuleService::update($new['id'], $new['code'], $new['nombre'], $new['estado']);
                    return $this->redirect(base_path() . '/admin/modules', 302);

                } catch (ValidationException $e) {
                    $row['code']   = $new['code'];
                    $row['nombre'] = $new['nombre'];
                    $row['estado'] = $new['estado'];

                    $html = $this->renderAdminLayout(
                        'Editar módulo',
                        render_template('admin/modules/edit.php', [
                            'error' => $e->getMessage(),
                            'row'   => $row,
                        ])
                    );
                    return Response::html($html, 422);
                }
            });
        });

        $this->mapPost('/admin/modules/delete', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.modules.manage', function (Request $req): Response {
                $id = (int)($req->post['id'] ?? 0);

                try {
                    AdminModuleService::delete($id);
                    return $this->redirect(base_path() . '/admin/modules', 302);

                } catch (ValidationException $e) {
                    $items = AdminModuleService::listAll();

                    $html = $this->renderAdminLayout(
                        'Módulos',
                        render_template('admin/modules/index.php', [
                            'items' => $items,
                            'error' => $e->getMessage(),
                        ])
                    );
                    return Response::html($html, 422);
                }
            });
        });
        
        /* ========= ADMIN: MODERACIÓN ========= */
$this->mapGet('/admin/moderacion/vacantes', function (Request $req): Response {
    return $this->gateAdmin($req, 'admin.moderation.manage', function (Request $req): Response {

        $q = trim((string)($req->query['q'] ?? ''));
        $estado = trim((string)($req->query['estado'] ?? ''));

        $items = AdminModerationVacanteService::listAll($q, $estado);

        $html = $this->renderAdminLayout(
            'Moderación • Vacantes',
            render_template('admin/moderacion/vacantes/index.php', [
                'items'  => $items,
                'q'      => $q,
                'estado' => $estado,
                'error'  => null,
                'csrf'   => CsrfToken::issue(),
            ])
        );

        return Response::html($html, 200);
    });
});

$this->mapGet('/admin/moderacion/vacantes/ver', function (Request $req): Response {
    return $this->gateAdmin($req, 'admin.moderation.manage', function (Request $req): Response {

        $id = (int)($req->query['id'] ?? 0);
        $row = AdminModerationVacanteService::find($id);
        if (!$row) return Response::html("<h3>No existe</h3>", 404);

        $posts = AdminModerationVacanteService::listPostulaciones($id);

        $html = $this->renderAdminLayout(
            'Moderación • Ver vacante',
            render_template('admin/moderacion/vacantes/ver.php', [
                'row'   => $row,
                'posts' => $posts,
                'error' => null,
                'csrf'  => CsrfToken::issue(),
            ])
        );

        return Response::html($html, 200);
    });
});

$this->mapPost('/admin/moderacion/vacantes/estado', function (Request $req): Response {
    return $this->gateAdmin($req, 'admin.moderation.manage', function (Request $req): Response {

        $id     = (int)($req->post['vacante_id'] ?? 0);
        $estado = (string)($req->post['estado'] ?? '');

        $before = AdminModerationVacanteService::find($id);
        $beforeEstado = (string)($before['estado'] ?? '');

        AdminModerationVacanteService::setEstado($id, $estado);

        $actorId = (int)AuthService::currentUserId();
        AuditLogService::write(
            $actorId,
            'admin.vacante.estado.set',
            'vacantes',
            $id,
            [
                'from' => $beforeEstado,
                'to'   => $estado,
            ]
        );

        return $this->redirect(base_path() . '/admin/moderacion/vacantes/ver?id=' . $id, 302);
    });
});

        /* ========= AUDITORÍA ========= */
        
        $this->mapGet('/admin/auditoria', function (Request $req): Response {
    return $this->gateAdmin($req, 'admin.audit.view', function (Request $req): Response {

        $q      = trim((string)($req->query['q'] ?? ''));
        $action = trim((string)($req->query['action'] ?? ''));
        $entity = trim((string)($req->query['entity'] ?? ''));

        $items = AuditLogService::list([
            'q' => $q,
            'action' => $action,
            'entity' => $entity,
        ]);

        $html = $this->renderAdminLayout(
            'Auditoría',
            render_template('admin/auditoria/index.php', [
                'items'  => $items,
                'q'      => $q,
                'action' => $action,
                'entity' => $entity,
            ])
        );

        return Response::html($html, 200);
    });
});



 /* ========= POSTULACIONES (POSTULANTE) ========= */
$this->mapPost('/vacantes/postular', function (Request $req): Response {
    // require auth ya está como middleware global
    return $this->gatePostulante($req, function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();

        $vacanteId = (int)($req->post['vacante_id'] ?? 0);
        $mensaje   = isset($req->post['mensaje']) ? (string)$req->post['mensaje'] : null;

        try {
            PostulacionService::create($vacanteId, $uid, $mensaje);

            // ===== NOTIFICACIÓN A EMPRESA: nueva postulación =====
            $pdo = DB::pdo();
            $st = $pdo->prepare("SELECT empresa_user_id, titulo FROM vacantes WHERE id = :id LIMIT 1");
            $st->execute(['id' => $vacanteId]);
            $v = $st->fetch(\PDO::FETCH_ASSOC);

            if ($v && !empty($v['empresa_user_id'])) {
                $empresaId = (int)$v['empresa_user_id'];
                $tituloVac = (string)($v['titulo'] ?? 'Vacante');

                NotificationService::create(
                    $empresaId,
                    'empresa.postulacion.nueva',
                    'Nueva postulación recibida',
                    "Se recibió una postulación para: {$tituloVac}",
                    base_path() . '/empresa/postulaciones?vacante_id=' . (int)$vacanteId
                );
            }
            // ===== FIN NOTIFICACIÓN =====

            return $this->redirect(base_path() . '/vacantes/ver?id=' . $vacanteId, 302);

        } catch (ValidationException $e) {
            $viewerId = (int)AuthService::currentUserId(); $row = VacantePublicService::findPublicada($vacanteId, $viewerId);
            if (!$row) return Response::html("<h3>No existe</h3>", 404);

            $csrf = CsrfToken::issue();
            $canApply = true;

            $html = render_template('layouts/front.php', [
                'pageTitle' => 'Vacante',
                'content'   => render_template('front/vacantes/ver.php', [
                    'row'      => $row,
                    'error'    => $e->getMessage(),
                    'ok'       => null,
                    'csrf'     => $csrf,
                    'canApply' => $canApply,
                ]),
            ]);
            return Response::html($html, 422);
        }
    });
});


                
        /* ========= VACANTES (PÚBLICO) ========= */
        $this->mapGet('/vacantes', function (Request $req): Response {
            $items = VacantePublicService::listPublicadas();
        
            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Vacantes',
                'content'   => render_template('front/vacantes/index.php', [
                    'items' => $items,
                ]),
            ]);
            return Response::html($html, 200);
        });
        
        $this->mapGet('/vacantes/ver', function (Request $req): Response {
            $id = (int)($req->query['id'] ?? 0);
            $viewerId = (int)AuthService::currentUserId(); $row = VacantePublicService::findPublicada($id, $viewerId);
            if (!$row) {
                return Response::html("<h3>No existe</h3>", 404);
            }
        
            $csrf = CsrfToken::issue();
            $canApply = in_array(AuthService::currentUserType(), ['postulante', 'admin'], true);
        
            $html = render_template('layouts/app.php', [
                'pageTitle' => 'Vacante',
                'content'   => render_template('front/vacantes/ver.php', [
                    'row'      => $row,
                    'error'    => null,
                    'ok'       => null,
                    'csrf'     => $csrf,
                    'canApply' => $canApply,
                ]),
            ]);
            return Response::html($html, 200);
        });

        /* ========= ADMIN: PLANES ========= */
        $this->mapGet('/admin/plans', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.plans.manage', function (Request $req): Response {
                $items = AdminPlanService::listAll();
                $csrf  = CsrfToken::issue();

                $html = $this->renderAdminLayout(
                    'Planes',
                    render_template('admin/plans/index.php', [
                        'items' => $items,
                        'error' => null,
                        'csrf'  => $csrf,
                    ])
                );
                return Response::html($html, 200);
            });
        });

        $this->mapGet('/admin/plans/create', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.plans.manage', function (Request $req): Response {
                $csrf = CsrfToken::issue();

                $html = $this->renderAdminLayout(
                    'Crear plan',
                    render_template('admin/plans/create.php', [
                        'error' => null,
                        'old'   => [
                            'code'          => '',
                            'nombre'        => '',
                            'descripcion'   => '',
                            'precio_mensual'=> '0.00',
                            'precio_anual'  => '0.00',
                            'estado'        => 'activo',
                        ],
                        'csrf'  => $csrf,
                    ])
                );
                return Response::html($html, 200);
            });
        });

        $this->mapPost('/admin/plans/create', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.plans.manage', function (Request $req): Response {
                $old = [
                    'code'           => trim((string)($req->post['code'] ?? '')),
                    'nombre'         => trim((string)($req->post['nombre'] ?? '')),
                    'descripcion'    => (string)($req->post['descripcion'] ?? ''),
                    'precio_mensual' => (string)($req->post['precio_mensual'] ?? '0.00'),
                    'precio_anual'   => (string)($req->post['precio_anual'] ?? '0.00'),
                    'estado'         => (string)($req->post['estado'] ?? 'activo'),
                ];

                try {
                    AdminPlanService::create(
                        $old['code'],
                        $old['nombre'],
                        $old['descripcion'],
                        $old['precio_mensual'],
                        $old['precio_anual'],
                        $old['estado']
                    );

                    return $this->redirect(base_path() . '/admin/plans', 302);

                } catch (ValidationException $e) {
                    $csrf = CsrfToken::issue();

                    $html = $this->renderAdminLayout(
                        'Crear plan',
                        render_template('admin/plans/create.php', [
                            'error' => $e->getMessage(),
                            'old'   => $old,
                            'csrf'  => $csrf,
                        ])
                    );
                    return Response::html($html, 422);
                }
            });
        });

        $this->mapGet('/admin/plans/edit', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.plans.manage', function (Request $req): Response {
                $id  = (int)($req->query['id'] ?? 0);
                $row = AdminPlanService::find($id);

                if (!$row) {
                    return Response::html("<h3>No existe</h3>", 404);
                }

                $csrf = CsrfToken::issue();

                $html = $this->renderAdminLayout(
                    'Editar plan',
                    render_template('admin/plans/edit.php', [
                        'error' => null,
                        'row'   => $row,
                        'csrf'  => $csrf,
                    ])
                );
                return Response::html($html, 200);
            });
        });

        $this->mapPost('/admin/plans/edit', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.plans.manage', function (Request $req): Response {
                $id  = (int)($req->post['id'] ?? 0);
                $row = AdminPlanService::find($id);

                if (!$row) {
                    return Response::html("<h3>No existe</h3>", 404);
                }

                $new = [
                    'id'            => $id,
                    'code'          => trim((string)($req->post['code'] ?? '')),
                    'nombre'        => trim((string)($req->post['nombre'] ?? '')),
                    'descripcion'   => (string)($req->post['descripcion'] ?? ''),
                    'precio_mensual'=> (string)($req->post['precio_mensual'] ?? '0.00'),
                    'precio_anual'  => (string)($req->post['precio_anual'] ?? '0.00'),
                    'estado'        => (string)($req->post['estado'] ?? 'activo'),
                ];

                try {
                    AdminPlanService::update(
                        $new['id'],
                        $new['code'],
                        $new['nombre'],
                        $new['descripcion'],
                        $new['precio_mensual'],
                        $new['precio_anual'],
                        $new['estado']
                    );

                    return $this->redirect(base_path() . '/admin/plans', 302);

                } catch (ValidationException $e) {
                    $row['code']          = $new['code'];
                    $row['nombre']        = $new['nombre'];
                    $row['descripcion']   = $new['descripcion'];
                    $row['precio_mensual']= $new['precio_mensual'];
                    $row['precio_anual']  = $new['precio_anual'];
                    $row['estado']        = $new['estado'];

                    $csrf = CsrfToken::issue();

                    $html = $this->renderAdminLayout(
                        'Editar plan',
                        render_template('admin/plans/edit.php', [
                            'error' => $e->getMessage(),
                            'row'   => $row,
                            'csrf'  => $csrf,
                        ])
                    );
                    return Response::html($html, 422);
                }
            });
        });

        $this->mapPost('/admin/plans/delete', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.plans.manage', function (Request $req): Response {
                $id = (int)($req->post['id'] ?? 0);

                try {
                    AdminPlanService::delete($id);
                    return $this->redirect(base_path() . '/admin/plans', 302);

                } catch (ValidationException $e) {
                    $items = AdminPlanService::listAll();
                    $csrf  = CsrfToken::issue();

                    $html = $this->renderAdminLayout(
                        'Planes',
                        render_template('admin/plans/index.php', [
                            'items' => $items,
                            'error' => $e->getMessage(),
                            'csrf'  => $csrf,
                        ])
                    );
                    return Response::html($html, 422);
                }
            });
        });

        /* ========= ADMIN: PLAN → MÓDULOS ========= */
        $this->mapGet('/admin/plans/modules', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.plans.manage', function (Request $req): Response {
                $planId = (int)($req->query['plan_id'] ?? 0);
                $plan   = AdminPlanService::find($planId);

                if (!$plan) {
                    return Response::html("<h3>Plan no existe</h3>", 404);
                }

                $modules = AdminPlanModuleService::listModulesForPlan($planId);
                $csrf    = CsrfToken::issue();

                $html = $this->renderAdminLayout(
                    'Plan → Módulos',
                    render_template('admin/plans/modules.php', [
                        'plan'    => $plan,
                        'modules' => $modules,
                        'error'   => null,
                        'csrf'    => $csrf,
                    ])
                );

                return Response::html($html, 200);
            });
        });

        $this->mapPost('/admin/plans/modules/save', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.plans.manage', function (Request $req): Response {
                $planId  = (int)($req->post['plan_id'] ?? 0);
                $plan    = AdminPlanService::find($planId);

                if (!$plan) {
                    return Response::html("<h3>Plan no existe</h3>", 404);
                }

                $modules = $req->post['modules'] ?? [];
                if (!is_array($modules)) $modules = [];

                try {
                    AdminPlanModuleService::savePlanModules($planId, $modules);
                    return $this->redirect(base_path() . '/admin/plans/modules?plan_id=' . $planId, 302);

                } catch (ValidationException $e) {
                    $modulesList = AdminPlanModuleService::listModulesForPlan($planId);
                    $csrf        = CsrfToken::issue();

                    $html = $this->renderAdminLayout(
                        'Plan → Módulos',
                        render_template('admin/plans/modules.php', [
                            'plan'    => $plan,
                            'modules' => $modulesList,
                            'error'   => $e->getMessage(),
                            'csrf'    => $csrf,
                        ])
                    );

                    return Response::html($html, 422);
                }
            });
        });

        /* ========= ADMIN: ROLES ========= */
        $this->mapGet('/admin/roles', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.rbac.manage', function (Request $req): Response {
                $items = AdminRoleService::listAll();
                $csrf  = CsrfToken::issue();

                $html = $this->renderAdminLayout(
                    'Roles',
                    render_template('admin/roles/index.php', [
                        'items' => $items,
                        'error' => null,
                        'csrf'  => $csrf,
                    ])
                );
                return Response::html($html, 200);
            });
        });

        $this->mapGet('/admin/roles/create', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.rbac.manage', function (Request $req): Response {
                $csrf = CsrfToken::issue();

                $html = $this->renderAdminLayout(
                    'Crear rol',
                    render_template('admin/roles/create.php', [
                        'error' => null,
                        'old'   => ['code' => '', 'nombre' => '', 'descripcion' => '', 'estado' => 'activo'],
                        'csrf'  => $csrf,
                    ])
                );
                return Response::html($html, 200);
            });
        });

        $this->mapPost('/admin/roles/create', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.rbac.manage', function (Request $req): Response {
                $old = [
                    'code'        => trim((string)($req->post['code'] ?? '')),
                    'nombre'      => trim((string)($req->post['nombre'] ?? '')),
                    'descripcion' => trim((string)($req->post['descripcion'] ?? '')),
                    'estado'      => (string)($req->post['estado'] ?? 'activo'),
                ];

                try {
                    AdminRoleService::create($old['code'], $old['nombre'], $old['descripcion'], $old['estado']);
                    return $this->redirect(base_path() . '/admin/roles', 302);

                } catch (ValidationException $e) {
                    $csrf = CsrfToken::issue();

                    $html = $this->renderAdminLayout(
                        'Crear rol',
                        render_template('admin/roles/create.php', [
                            'error' => $e->getMessage(),
                            'old'   => $old,
                            'csrf'  => $csrf,
                        ])
                    );
                    return Response::html($html, 422);
                }
            });
        });

        $this->mapGet('/admin/roles/edit', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.rbac.manage', function (Request $req): Response {
                $id  = (int)($req->query['id'] ?? 0);
                $row = AdminRoleService::find($id);

                if (!$row) {
                    return Response::html("<h3>No existe</h3>", 404);
                }

                $csrf = CsrfToken::issue();

                $html = $this->renderAdminLayout(
                    'Editar rol',
                    render_template('admin/roles/edit.php', [
                        'error' => null,
                        'row'   => $row,
                        'csrf'  => $csrf,
                    ])
                );
                return Response::html($html, 200);
            });
        });

        $this->mapPost('/admin/roles/edit', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.rbac.manage', function (Request $req): Response {
                $id  = (int)($req->post['id'] ?? 0);
                $row = AdminRoleService::find($id);
                if (!$row) return Response::html("<h3>No existe</h3>", 404);

                $new = [
                    'id'          => $id,
                    'code'        => trim((string)($req->post['code'] ?? '')),
                    'nombre'      => trim((string)($req->post['nombre'] ?? '')),
                    'descripcion' => trim((string)($req->post['descripcion'] ?? '')),
                    'estado'      => (string)($req->post['estado'] ?? 'activo'),
                ];

                try {
                    AdminRoleService::update($new['id'], $new['code'], $new['nombre'], $new['descripcion'], $new['estado']);
                    return $this->redirect(base_path() . '/admin/roles', 302);

                } catch (ValidationException $e) {
                    $row['code']        = $new['code'];
                    $row['nombre']      = $new['nombre'];
                    $row['descripcion'] = $new['descripcion'];
                    $row['estado']      = $new['estado'];

                    $csrf = CsrfToken::issue();

                    $html = $this->renderAdminLayout(
                        'Editar rol',
                        render_template('admin/roles/edit.php', [
                            'error' => $e->getMessage(),
                            'row'   => $row,
                            'csrf'  => $csrf,
                        ])
                    );
                    return Response::html($html, 422);
                }
            });
        });

        $this->mapPost('/admin/roles/delete', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.rbac.manage', function (Request $req): Response {
                $id = (int)($req->post['id'] ?? 0);

                try {
                    AdminRoleService::delete($id);
                    return $this->redirect(base_path() . '/admin/roles', 302);

                } catch (ValidationException $e) {
                    $items = AdminRoleService::listAll();
                    $csrf  = CsrfToken::issue();

                    $html = $this->renderAdminLayout(
                        'Roles',
                        render_template('admin/roles/index.php', [
                            'items' => $items,
                            'error' => $e->getMessage(),
                            'csrf'  => $csrf,
                        ])
                    );
                    return Response::html($html, 422);
                }
            });
        });

        /* ========= ADMIN: RBAC (alias) ========= */
        $this->mapGet('/admin/rbac', function (Request $req): Response {
            return $this->redirect(base_path() . '/admin/roles', 302);
        });

        /* ========= ADMIN: ROL → PERMISOS ========= */
        $this->mapGet('/admin/roles/permissions', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.rbac.manage', function (Request $req): Response {
                $roleId = (int)($req->query['role_id'] ?? 0);
                $role   = AdminRoleService::find($roleId);

                if (!$role) {
                    return Response::html("<h3>Rol no existe</h3>", 404);
                }

                $items = AdminRolePermissionService::listPermissionsForRole($roleId);
                $csrf  = CsrfToken::issue();

                $html = $this->renderAdminLayout(
                    'Rol → Permisos',
                    render_template('admin/roles/permissions.php', [
                        'role'  => $role,
                        'items' => $items,
                        'error' => null,
                        'csrf'  => $csrf,
                    ])
                );

                return Response::html($html, 200);
            });
        });

        $this->mapPost('/admin/roles/permissions/save', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.rbac.manage', function (Request $req): Response {
                $roleId = (int)($req->post['role_id'] ?? 0);
                $role   = AdminRoleService::find($roleId);

                if (!$role) {
                    return Response::html("<h3>Rol no existe</h3>", 404);
                }

                $perms = $req->post['permissions'] ?? [];
                if (!is_array($perms)) $perms = [];

                try {
                    AdminRolePermissionService::saveRolePermissions($roleId, $perms);
                    return $this->redirect(base_path() . '/admin/roles/permissions?role_id=' . $roleId, 302);

                } catch (ValidationException $e) {
                    $items = AdminRolePermissionService::listPermissionsForRole($roleId);
                    $csrf  = CsrfToken::issue();

                    $html = $this->renderAdminLayout(
                        'Rol → Permisos',
                        render_template('admin/roles/permissions.php', [
                            'role'  => $role,
                            'items' => $items,
                            'error' => $e->getMessage(),
                            'csrf'  => $csrf,
                        ])
                    );

                    return Response::html($html, 422);
                }
            });
        });

        /* ========= ADMIN: USUARIOS (LIST) ========= */
        $this->mapGet('/admin/users', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.rbac.manage', function (Request $req): Response {
                $q = trim((string)($req->query['q'] ?? ''));

                $items = AdminUserRoleService::listUsers($q);

                $html = $this->renderAdminLayout(
                    'Usuarios',
                    render_template('admin/users/index.php', [
                        'items' => $items,
                        'q'     => $q,
                        'error' => null,
                    ])
                );

                return Response::html($html, 200);
            });
        });

        /* ========= ADMIN: USUARIO → ROLES ========= */
        $this->mapGet('/admin/users/roles', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.rbac.manage', function (Request $req): Response {
                $userId = (int)($req->query['user_id'] ?? 0);
                $user   = AdminUserRoleService::findUser($userId);

                if (!$user) {
                    return Response::html("<h3>Usuario no existe</h3>", 404);
                }

                $roles = AdminUserRoleService::listRolesForUser($userId);
                $csrf  = CsrfToken::issue();

                $html = $this->renderAdminLayout(
                    'Usuario → Roles',
                    render_template('admin/users/roles.php', [
                        'user'  => $user,
                        'roles' => $roles,
                        'error' => null,
                        'csrf'  => $csrf,
                    ])
                );

                return Response::html($html, 200);
            });
        });

        $this->mapPost('/admin/users/roles/save', function (Request $req): Response {
            return $this->gateAdmin($req, 'admin.rbac.manage', function (Request $req): Response {
                $userId = (int)($req->post['user_id'] ?? 0);
                $user   = AdminUserRoleService::findUser($userId);

                if (!$user) {
                    return Response::html("<h3>Usuario no existe</h3>", 404);
                }

                $roles = $req->post['roles'] ?? [];
                if (!is_array($roles)) $roles = [];

                try {
                    AdminUserRoleService::saveUserRoles($userId, $roles);
                    return $this->redirect(base_path() . '/admin/users/roles?user_id=' . $userId, 302);

                } catch (ValidationException $e) {
                    $rolesList = AdminUserRoleService::listRolesForUser($userId);
                    $csrf      = CsrfToken::issue();

                    $html = $this->renderAdminLayout(
                        'Usuario → Roles',
                        render_template('admin/users/roles.php', [
                            'user'  => $user,
                            'roles' => $rolesList,
                            'error' => $e->getMessage(),
                            'csrf'  => $csrf,
                        ])
                    );

                    return Response::html($html, 422);
                }
            });
        });
        
        /* ========= NOTIFICACIONES (EMPRESA) ========= */
$this->mapGet('/empresa/notificaciones', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $items = NotificationService::listForUser($uid, 100);

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'Notificaciones',
            'content' => render_template('empresa/notificaciones/index.php', [
                'items' => $items,
                'csrf'  => CsrfToken::issue(),
            ]),
        ]);
        return Response::html($html, 200);
    });
});

$this->mapPost('/empresa/notificaciones/read', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $id  = (int)($req->post['id'] ?? 0);
        if ($id > 0) NotificationService::markRead($uid, $id);
        return $this->redirect(base_path().'/empresa/notificaciones', 302);
    });
});

$this->mapPost('/empresa/notificaciones/read_all', function (Request $req): Response {
    return $this->gateEmpresa($req, 'empresa.jobs.manage', function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        NotificationService::markAllRead($uid);
        return $this->redirect(base_path().'/empresa/notificaciones', 302);
    });
});

/* ========= NOTIFICACIONES (POSTULANTE) ========= */
$this->mapGet('/postulante/notificaciones', function (Request $req): Response {
    return $this->gatePostulante($req, function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $items = NotificationService::listForUser($uid, 100);

        $html = render_template('layouts/app.php', [
            'pageTitle' => 'Notificaciones',
            'content' => render_template('postulante/notificaciones/index.php', [
                'items' => $items,
                'csrf'  => CsrfToken::issue(),
            ]),
        ]);
        return Response::html($html, 200);
    });
});

$this->mapPost('/postulante/notificaciones/read', function (Request $req): Response {
    return $this->gatePostulante($req, function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        $id  = (int)($req->post['id'] ?? 0);
        if ($id > 0) NotificationService::markRead($uid, $id);
        return $this->redirect(base_path().'/postulante/notificaciones', 302);
    });
});

$this->mapPost('/postulante/notificaciones/read_all', function (Request $req): Response {
    return $this->gatePostulante($req, function (Request $req): Response {
        $uid = (int)AuthService::currentUserId();
        NotificationService::markAllRead($uid);
        return $this->redirect(base_path().'/postulante/notificaciones', 302);
    });
});

/* ========= EMPRESAS (PÚBLICO) ========= */
$this->mapGet('/empresas', function (Request $req): Response {
    $page = (int)($req->query['p'] ?? 1);
    if ($page < 1) $page = 1;

    $limit = 24;
    $offset = ($page - 1) * $limit;

    $items = EmpresaPublicService::listEmpresasPublicas($limit, $offset);

    $html = render_template('layouts/app.php', [
        'pageTitle' => 'Empresas',
        'content'   => render_template('front/empresas/index.php', [
            'items' => $items ?: [],
            'page'  => $page,
        ]),
    ]);

    return Response::html($html, 200);
});


        
        /* ========= EMPRESAS (PÚBLICO) ========= */
        $this->mapGet('/empresas/ver', function (Request $req): Response {
            $id = (int)($req->query['id'] ?? 0);
            if ($id <= 0) {
                return Response::html("<h3>No existe</h3>", 404);
            }
        
            $empresa = EmpresaPublicService::findEmpresa($id);
            if (!$empresa) {
                return Response::html("<h3>No existe</h3>", 404);
            }
        
            // PERFIL PÚBLICO (logo, banner, bio, etc.)
            // Usa el servicio real del proyecto (no inventamos otro)
            $perfil = [];
            if (class_exists('EmpresaProfileService') && method_exists('EmpresaProfileService', 'get')) {
                $perfil = EmpresaProfileService::get($id);
                if (!is_array($perfil)) $perfil = [];
            }
        
            $vacantes = EmpresaPublicService::listVacantesPublicadas($id);
            $totalPub = is_array($vacantes) ? count($vacantes) : 0;
        
            $html = render_template('layouts/app.php', [
                'pageTitle' => (string)($empresa['nombre'] ?? 'Empresa'),
                'content'   => render_template('front/empresas/ver.php', [
                    'empresa'  => $empresa,
                    'perfil'   => $perfil,
                    'vacantes' => $vacantes ?: [],
                    'total'    => $totalPub,
                ]),
            ]);
        
            return Response::html($html, 200);
        });
     
        /* ========= LOGOUT ========= */
        $this->mapGet('/logout', function (Request $req): Response {
            AuthService::logout();
            return $this->redirect(base_path() . '/login', 302);
        });

                // Rutas del Panel de Postulante (modular)
        require_once __DIR__ . '/routes/postulante_panel.php';

/* ========= 404 ========= */
        $this->router->setNotFoundHandler(function (Request $req): Response {
            $accept = strtolower((string)($_SERVER['HTTP_ACCEPT'] ?? ''));
            $wantsHtml = $accept === '' || str_contains($accept, 'text/html') || str_contains($accept, '*/*');

            if ($wantsHtml) {
                $html = "<!doctype html><html><head><meta charset='utf-8'>"
                    . "<meta name='viewport' content='width=device-width,initial-scale=1'>"
                    . "<title>404</title></head><body>"
                    . "<h2>404</h2><p>Ruta no encontrada.</p></body></html>";
                return Response::html($html, 404);
            }

            return Response::json(['ok' => false, 'error' => 'Ruta no encontrada'], 404);
        });
    }
}

/* ===============================
 * APP BOOTSTRAP
 * =============================== */
function app(): App
{
    $config = config();
    $GLOBALS['APP_CONFIG'] = $config;

    if (session_status() !== PHP_SESSION_ACTIVE) {
        session_set_cookie_params([
            'lifetime' => 0,
            'path'     => '/',
            'secure'   => (bool)($config['security']['cookies_secure'] ?? true),
            'httponly' => true,
            'samesite' => 'Lax',
        ]);
        session_name($config['security']['session_name'] ?? 'APPSESSID');
        session_start();
    }

    return new App($config);
}
