<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/DB/DB.php';

final class EmpresaProfileService
{
    public static function get(int $empresaUserId): array
    {
        $pdo = DB::pdo();

        // Asegurar que exista fila
        $pdo->prepare("INSERT IGNORE INTO empresa_perfiles (user_id) VALUES (:uid)")
            ->execute([':uid' => $empresaUserId]);

        $st = $pdo->prepare("
            SELECT
              user_id,
              telefono,
              whatsapp,
              ubicacion,
              bio,
              website_url,
              google_business_url,
              logo_path,
              banner_path
            FROM empresa_perfiles
            WHERE user_id = :uid
            LIMIT 1
        ");
        $st->execute([':uid' => $empresaUserId]);

        $row = $st->fetch(\PDO::FETCH_ASSOC) ?: [];

        return [
            'user_id' => $empresaUserId,
            'telefono' => $row['telefono'] ?? null,
            'whatsapp' => $row['whatsapp'] ?? null,
            'ubicacion' => $row['ubicacion'] ?? null,
            'bio' => $row['bio'] ?? null,
            'website_url' => $row['website_url'] ?? null,
            'google_business_url' => $row['google_business_url'] ?? null,
            'logo_path' => $row['logo_path'] ?? null,
            'banner_path' => $row['banner_path'] ?? null,
        ];
    }

    public static function update(int $empresaUserId, array $in): void
    {
        $pdo = DB::pdo();

        $bio = self::t((string)($in['bio'] ?? ''));
        if ($bio !== null && mb_strlen($bio) > 500) {
            $bio = mb_substr($bio, 0, 500);
        }

        $data = [
            ':uid' => $empresaUserId,
            ':telefono' => self::s((string)($in['telefono'] ?? ''), 40),
            ':whatsapp' => self::s((string)($in['whatsapp'] ?? ''), 40),
            ':ubicacion' => self::s((string)($in['ubicacion'] ?? ''), 120),
            ':bio' => $bio,
            ':website_url' => self::url((string)($in['website_url'] ?? '')),
            ':google_business_url' => self::url((string)($in['google_business_url'] ?? '')),
            ':logo_path' => self::path((string)($in['logo_path'] ?? '')),
            ':banner_path' => self::path((string)($in['banner_path'] ?? '')),
        ];

        $sql = "
            INSERT INTO empresa_perfiles
              (user_id, telefono, whatsapp, ubicacion, bio, website_url, google_business_url, logo_path, banner_path)
            VALUES
              (:uid, :telefono, :whatsapp, :ubicacion, :bio, :website_url, :google_business_url, :logo_path, :banner_path)
            ON DUPLICATE KEY UPDATE
              telefono = VALUES(telefono),
              whatsapp = VALUES(whatsapp),
              ubicacion = VALUES(ubicacion),
              bio = VALUES(bio),
              website_url = VALUES(website_url),
              google_business_url = VALUES(google_business_url),
              logo_path = VALUES(logo_path),
              banner_path = VALUES(banner_path)
        ";

        $pdo->prepare($sql)->execute($data);
    }

    // ===================== HELPERS =====================

    private static function s(string $v, int $max): ?string
    {
        $v = trim($v);
        if ($v === '') return null;
        if (mb_strlen($v) > $max) $v = mb_substr($v, 0, $max);
        return $v;
    }

    private static function t(string $v): ?string
    {
        $v = trim($v);
        return $v === '' ? null : $v;
    }

    private static function url(string $v): ?string
    {
        $v = trim($v);
        if ($v === '') return null;
        if (!preg_match('~^https?://~i', $v)) return null;
        return $v;
    }

    /**
     * Normaliza paths:
     * - acepta: /upload/..., /public/upload/..., upload/...
     * - retorna: /upload/... o /public/upload/... (se guarda tal cual venga, pero validado)
     *
     * Nota: no inventa prefijos; eso lo resuelve el template (tu caso: requiere /public).
     */
    private static function path(string $p): ?string
    {
        $p = trim($p);
        if ($p === '') return null;

        $p = str_replace('\\', '/', $p);
        if (!str_starts_with($p, '/')) $p = '/' . $p;

        // sólo permitimos rutas hacia upload
        if (!str_contains($p, '/upload/')) return null;

        return $p;
    }
}
