<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/DB/DB.php';

final class EmpresaReputationService
{
    /**
     * Calcula reputación global de la empresa usando reacciones en sus publicaciones (vacantes).
     * P = like + recommend + excellent
     * N = bad
     * score_0a10 = (P / (P+N)) * 10
     */
    public static function get(int $empresaUserId): array
    {
        $pdo = DB::pdo();

        $st = $pdo->prepare("
            SELECT
              SUM(CASE WHEN r.reaccion='like' THEN 1 ELSE 0 END) AS c_like,
              SUM(CASE WHEN r.reaccion='recommend' THEN 1 ELSE 0 END) AS c_recommend,
              SUM(CASE WHEN r.reaccion='excellent' THEN 1 ELSE 0 END) AS c_excellent,
              SUM(CASE WHEN r.reaccion='bad' THEN 1 ELSE 0 END) AS c_bad
            FROM vacantes v
            LEFT JOIN vacante_reacciones r ON r.vacante_id = v.id
            WHERE v.empresa_user_id = :eid
        ");
        $st->execute([':eid' => $empresaUserId]);
        $row = $st->fetch(\PDO::FETCH_ASSOC) ?: [];

        $like = (int)($row['c_like'] ?? 0);
        $rec  = (int)($row['c_recommend'] ?? 0);
        $exc  = (int)($row['c_excellent'] ?? 0);
        $bad  = (int)($row['c_bad'] ?? 0);

        $P = $like + $rec + $exc;
        $N = $bad;
        $T = $P + $N;

        $pct = $T > 0 ? ($P / $T) * 100.0 : 0.0;
        $score = $pct / 10.0; // 0..10

        $label = self::label($score);

        return [
            'like' => $like,
            'recommend' => $rec,
            'excellent' => $exc,
            'bad' => $bad,
            'P' => $P,
            'N' => $N,
            'T' => $T,
            'pct' => round($pct, 1),
            'score' => round($score, 2),
            'label' => $label['label'],
            'hint'  => $label['hint'],
        ];
    }

    private static function label(float $score): array
    {
        // Escala operativa (coherente con lo que pediste)
        if ($score < 4.0)  return ['label' => 'Alerta',        'hint' => 'Muchos reportes negativos'];
        if ($score < 5.0)  return ['label' => 'Cuidado',       'hint' => 'Revisá antes de confiar'];
        if ($score < 6.0)  return ['label' => 'Regular',       'hint' => 'Con precaución'];
        if ($score < 7.0)  return ['label' => 'Aceptable',     'hint' => 'Historial estable'];
        if ($score < 8.0)  return ['label' => 'Confiable',     'hint' => 'Buena experiencia general'];
        if ($score < 9.0)  return ['label' => 'Muy confiable', 'hint' => 'Alta recomendación'];
        return                ['label' => 'Excelente',      'hint' => 'Consistencia sobresaliente'];
    }
}
