<?php
declare(strict_types=1);

final class EmpresaVacanteService
{
    public static function listByEmpresa(int $empresaUserId): array
    {
        $pdo = DB::pdo();

        $sql = "
          SELECT
            id,
            titulo,
            ubicacion,
            modalidad,
            tipo_empleo,
            salario_min,
            salario_max,
            moneda,
            estado,
            imagen_path,
            publicada_en,
            cerrada_en,
            creado_en
          FROM vacantes
          WHERE empresa_user_id = :eid
          ORDER BY id DESC
        ";

        $st = $pdo->prepare($sql);
        $st->execute([':eid' => $empresaUserId]);
        return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    public static function findOwned(int $id, int $empresaUserId): ?array
    {
        $pdo = DB::pdo();

        $sql = "
          SELECT
            id,
            empresa_user_id,
            titulo,
            descripcion,
            ubicacion,
            modalidad,
            tipo_empleo,
            salario_min,
            salario_max,
            moneda,
            estado,
            imagen_path,
            publicada_en,
            cerrada_en,
            creado_en,
            actualizado_en
          FROM vacantes
          WHERE id = :id AND empresa_user_id = :eid
          LIMIT 1
        ";

        $st = $pdo->prepare($sql);
        $st->execute([':id' => $id, ':eid' => $empresaUserId]);

        $row = $st->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    /** @return int[] */
    public static function getCategoriaIds(int $vacanteId): array
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("SELECT categoria_id FROM vacante_categorias WHERE vacante_id = :v");
        $st->execute([':v' => $vacanteId]);
        $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $ids = [];
        foreach ($rows as $r) $ids[] = (int)($r['categoria_id'] ?? 0);

        $ids = array_values(array_unique(array_filter($ids, fn($x) => $x > 0)));
        return $ids;
    }

    /**
     * @param array $data incluye:
     *  - categoria_ids (int[])
     *  - imagen_path (string|null)
     */
    public static function create(int $empresaUserId, array $data): int
    {
        $v = self::validate($data);

        $pdo = DB::pdo();
        $pdo->beginTransaction();

        try {
            $sql = "
              INSERT INTO vacantes
              (empresa_user_id, titulo, descripcion, ubicacion, modalidad, tipo_empleo, salario_min, salario_max, moneda, estado, imagen_path)
              VALUES
              (:eid, :t, :d, :u, :m, :te, :smin, :smax, :mon, 'borrador', :img)
            ";

            $st = $pdo->prepare($sql);
            $st->execute([
                ':eid'  => $empresaUserId,
                ':t'    => $v['titulo'],
                ':d'    => $v['descripcion'],
                ':u'    => $v['ubicacion'],
                ':m'    => $v['modalidad'],
                ':te'   => $v['tipo_empleo'],
                ':smin' => $v['salario_min'],
                ':smax' => $v['salario_max'],
                ':mon'  => $v['moneda'],
                ':img'  => $v['imagen_path'],
            ]);

            $vacanteId = (int)$pdo->lastInsertId();

            self::saveCategorias($pdo, $vacanteId, $v['categoria_ids']);

            $pdo->commit();
            return $vacanteId;

        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }

    public static function update(int $id, int $empresaUserId, array $data): void
    {
        $row = self::findOwned($id, $empresaUserId);
        if (!$row) throw new ValidationException('La vacante no existe.');

        if (($row['estado'] ?? '') === 'cerrada') {
            throw new ValidationException('No se puede editar una vacante cerrada.');
        }

        $v = self::validate($data);

        $pdo = DB::pdo();
        $pdo->beginTransaction();

        try {
            $sql = "
              UPDATE vacantes
              SET
                titulo = :t,
                descripcion = :d,
                ubicacion = :u,
                modalidad = :m,
                tipo_empleo = :te,
                salario_min = :smin,
                salario_max = :smax,
                moneda = :mon,
                imagen_path = :img
              WHERE id = :id AND empresa_user_id = :eid
            ";

            $st = $pdo->prepare($sql);
            $st->execute([
                ':id'   => $id,
                ':eid'  => $empresaUserId,
                ':t'    => $v['titulo'],
                ':d'    => $v['descripcion'],
                ':u'    => $v['ubicacion'],
                ':m'    => $v['modalidad'],
                ':te'   => $v['tipo_empleo'],
                ':smin' => $v['salario_min'],
                ':smax' => $v['salario_max'],
                ':mon'  => $v['moneda'],
                ':img'  => $v['imagen_path'],
            ]);

            self::saveCategorias($pdo, $id, $v['categoria_ids']);

            $pdo->commit();
        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }

    public static function publish(int $id, int $empresaUserId): void
    {
        $row = self::findOwned($id, $empresaUserId);
        if (!$row) throw new ValidationException('La vacante no existe.');
        if (($row['estado'] ?? '') === 'cerrada') throw new ValidationException('No se puede publicar una vacante cerrada.');

        PlanLimitService::assertCanPublishVacante($empresaUserId);

        $pdo = DB::pdo();
        $st = $pdo->prepare("
          UPDATE vacantes
          SET estado = 'publicada',
              publicada_en = COALESCE(publicada_en, NOW()),
              cerrada_en = NULL
          WHERE id = :id AND empresa_user_id = :eid
        ");
        $st->execute([':id' => $id, ':eid' => $empresaUserId]);

        VacanteSlugService::ensureSlugForVacante($id, (string)($row['titulo'] ?? ''), null);
    }

    public static function close(int $id, int $empresaUserId): void
    {
        $row = self::findOwned($id, $empresaUserId);
        if (!$row) throw new ValidationException('La vacante no existe.');

        $pdo = DB::pdo();
        $st = $pdo->prepare("
          UPDATE vacantes
          SET estado = 'cerrada',
              cerrada_en = COALESCE(cerrada_en, NOW())
          WHERE id = :id AND empresa_user_id = :eid
        ");
        $st->execute([':id' => $id, ':eid' => $empresaUserId]);
    }

    public static function delete(int $id, int $empresaUserId): void
    {
        $row = self::findOwned($id, $empresaUserId);
        if (!$row) throw new ValidationException('La vacante no existe.');

        $pdo = DB::pdo();
        $pdo->beginTransaction();
        try {
            $pdo->prepare("DELETE FROM vacante_categorias WHERE vacante_id = :v")->execute([':v' => $id]);
            $pdo->prepare("DELETE FROM vacantes WHERE id = :id AND empresa_user_id = :eid")->execute([':id' => $id, ':eid' => $empresaUserId]);
            $pdo->commit();
        } catch (Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }

    // ===================== INTERNALS =====================

    /**
     * Guarda categorías del pivot de forma robusta:
     * - usa placeholders posicionales (?) para evitar HY093 por conteo/duplicados.
     */
    private static function saveCategorias(PDO $pdo, int $vacanteId, array $categoriaIds): void
    {
        // Normalizar por seguridad
        $categoriaIds = array_values(array_unique(array_filter(array_map('intval', (array)$categoriaIds), fn($v) => $v > 0)));
        $categoriaIds = array_slice($categoriaIds, 0, 25);

        $pdo->prepare("DELETE FROM vacante_categorias WHERE vacante_id = ?")->execute([$vacanteId]);

        if (empty($categoriaIds)) return;

        $placeholders = [];
        $params = [];

        foreach ($categoriaIds as $cid) {
            $placeholders[] = "(?, ?)";
            $params[] = $vacanteId;
            $params[] = (int)$cid;
        }

        $sql = "INSERT INTO vacante_categorias (vacante_id, categoria_id) VALUES " . implode(',', $placeholders);
        $pdo->prepare($sql)->execute($params);
    }

    private static function validate(array $data): array
    {
        $titulo = trim((string)($data['titulo'] ?? ''));
        $descripcion = trim((string)($data['descripcion'] ?? ''));

        $ubicacion = trim((string)($data['ubicacion'] ?? ''));
        $ubicacion = $ubicacion === '' ? null : $ubicacion;

        $modalidad = (string)($data['modalidad'] ?? 'presencial');
        $tipoEmpleo = (string)($data['tipo_empleo'] ?? 'tiempo_completo');

        $moneda = strtoupper(trim((string)($data['moneda'] ?? 'CRC')));
        if ($moneda === '') $moneda = 'CRC';

        $salMinRaw = trim((string)($data['salario_min'] ?? ''));
        $salMaxRaw = trim((string)($data['salario_max'] ?? ''));

        $salarioMin = $salMinRaw === '' ? null : (float)$salMinRaw;
        $salarioMax = $salMaxRaw === '' ? null : (float)$salMaxRaw;

        $categoriaIds = $data['categoria_ids'] ?? [];
        $categoriaIds = array_values(array_unique(array_filter(array_map('intval', (array)$categoriaIds), fn($v) => $v > 0)));
        $categoriaIds = array_slice($categoriaIds, 0, 25);

        $imagenPath = $data['imagen_path'] ?? null;
        $imagenPath = is_string($imagenPath) && trim($imagenPath) !== '' ? trim($imagenPath) : null;

        if ($titulo === '' || mb_strlen($titulo) < 5) {
            throw new ValidationException('El título es obligatorio (mínimo 5 caracteres).');
        }
        if ($descripcion === '' || mb_strlen($descripcion) < 20) {
            throw new ValidationException('La descripción es obligatoria (mínimo 20 caracteres).');
        }
        if (empty($categoriaIds)) {
            throw new ValidationException('Seleccioná al menos 1 categoría laboral.');
        }

        $modalidades = ['presencial','remoto','hibrido'];
        if (!in_array($modalidad, $modalidades, true)) throw new ValidationException('Modalidad inválida.');

        $tipos = ['tiempo_completo','medio_tiempo','temporal','freelance','practica'];
        if (!in_array($tipoEmpleo, $tipos, true)) throw new ValidationException('Tipo de empleo inválido.');

        if ($salarioMin !== null && $salarioMin < 0) throw new ValidationException('Salario mínimo inválido.');
        if ($salarioMax !== null && $salarioMax < 0) throw new ValidationException('Salario máximo inválido.');
        if ($salarioMin !== null && $salarioMax !== null && $salarioMax < $salarioMin) {
            throw new ValidationException('El salario máximo no puede ser menor que el salario mínimo.');
        }

        return [
            'titulo' => $titulo,
            'descripcion' => $descripcion,
            'ubicacion' => $ubicacion,
            'modalidad' => $modalidad,
            'tipo_empleo' => $tipoEmpleo,
            'salario_min' => $salarioMin,
            'salario_max' => $salarioMax,
            'moneda' => $moneda,
            'categoria_ids' => $categoriaIds,
            'imagen_path' => $imagenPath,
        ];
    }
}
