<?php
declare(strict_types=1);

final class Request
{
    public function __construct(
        public readonly string $method,
        public readonly string $path,
        public readonly array $query,
        public readonly array $post,
        public readonly array $files,
        public readonly array $headers,
        public readonly string $ip,
        public readonly string $userAgent,
        public readonly string $rawBody
    ) {}

    public static function fromGlobals(): self
    {
        $method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');
        $uri    = $_SERVER['REQUEST_URI'] ?? '/';
        $path   = parse_url($uri, PHP_URL_PATH) ?: '/';
        
        // Hard-normalize cuando el sitio se visita con prefijo /public (evita 404 por base mal detectada)
        if ($path === '/public') {
            $path = '/';
        } elseif (str_starts_with($path, '/public/')) {
            $path = substr($path, 7); // remove "/public"
            if ($path === '') $path = '/';
        }

        // Detectar base path según SCRIPT_NAME (ej: /public/index.php => base=/public)
        $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
        $base = rtrim(str_replace('\\', '/', dirname($scriptName)), '/');
        // Normalizar: si base está vacío, asumir '/'
        if ($base === '') {
            $base = '';
        }

        // Si la app corre dentro de /public, convertir /public/health -> /health
        if ($base !== '' && $base !== '/' && str_starts_with($path, $base . '/')) {
            $path = substr($path, strlen($base));
            if ($path === '') {
                $path = '/';
            }
        }

        // Caso típico: base=/public, request=/public => home
        if ($base !== '' && $base !== '/' && $path === $base) {
            $path = '/';
        }

        $headers = [];
        foreach ($_SERVER as $k => $v) {
            if (str_starts_with($k, 'HTTP_')) {
                $name = str_replace('_', '-', strtolower(substr($k, 5)));
                $headers[$name] = (string)$v;
            }
        }

        $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $raw = file_get_contents('php://input') ?: '';

        return new self(
            $method,
            $path,
            $_GET ?? [],
            $_POST ?? [],
            $_FILES ?? [],
            $headers,
            $ip,
            $ua,
            $raw
        );
    }

    public function header(string $name): ?string
    {
        $key = strtolower($name);
        return $this->headers[$key] ?? null;
    }

    public function isPostLike(): bool
    {
        return in_array($this->method, ['POST', 'PUT', 'PATCH', 'DELETE'], true);
    }
}
