<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/DB/DB.php';

final class FeedBoostService
{
    /**
     * Crea un boost para el actor (empresa/postulante/profesional/rrhh futuro),
     * enfocado por categoria_id (o NULL para todas) y scope (global o para_ti).
     */
    public static function create(array $in): int
    {
        $pdo = DB::pdo();

        $actorUserId = (int)($in['actor_user_id'] ?? 0);
        $actorTipo   = (string)($in['actor_tipo'] ?? '');
        $scope       = (string)($in['scope'] ?? 'para_ti');
        $categoriaId = (int)($in['categoria_id'] ?? 0);
        $scoreBoost  = (int)($in['score_boost'] ?? 500);
        $days        = (int)($in['days'] ?? 7);

        if ($actorUserId <= 0) throw new \RuntimeException('actor_user_id inválido');
        if (!in_array($actorTipo, ['empresa','postulante','profesional'], true)) {
            throw new \RuntimeException('actor_tipo inválido');
        }
        if (!in_array($scope, ['para_ti','global'], true)) {
            throw new \RuntimeException('scope inválido');
        }

        $categoria = $categoriaId > 0 ? $categoriaId : null;
        $days = max(1, min(30, $days));
        $scoreBoost = max(0, min(5000, $scoreBoost));

        $st = $pdo->prepare("
            INSERT INTO feed_boosts
              (actor_user_id, actor_tipo, scope, categoria_id, score_boost, starts_at, ends_at, estado)
            VALUES
              (:uid, :tipo, :scope, :cid, :sb, NOW(), DATE_ADD(NOW(), INTERVAL :days DAY), 'activo')
        ");
        $st->bindValue(':uid', $actorUserId, \PDO::PARAM_INT);
        $st->bindValue(':tipo', $actorTipo);
        $st->bindValue(':scope', $scope);
        if ($categoria === null) {
            $st->bindValue(':cid', null, \PDO::PARAM_NULL);
        } else {
            $st->bindValue(':cid', $categoria, \PDO::PARAM_INT);
        }
        $st->bindValue(':sb', $scoreBoost, \PDO::PARAM_INT);
        $st->bindValue(':days', $days, \PDO::PARAM_INT);

        $st->execute();
        return (int)$pdo->lastInsertId();
    }

    public static function listActiveByActor(int $actorUserId, string $actorTipo): array
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT fb.*, c.nombre AS categoria_nombre
            FROM feed_boosts fb
            LEFT JOIN categorias_laborales c ON c.id = fb.categoria_id
            WHERE fb.actor_user_id = :uid
              AND fb.actor_tipo = :tipo
            ORDER BY fb.estado ASC, fb.ends_at DESC, fb.id DESC
        ");
        $st->execute([':uid' => $actorUserId, ':tipo' => $actorTipo]);
        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    public static function disable(int $boostId, int $actorUserId, string $actorTipo): void
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("
            UPDATE feed_boosts
            SET estado='inactivo'
            WHERE id = :id
              AND actor_user_id = :uid
              AND actor_tipo = :tipo
            LIMIT 1
        ");
        $st->execute([':id'=>$boostId, ':uid'=>$actorUserId, ':tipo'=>$actorTipo]);
    }
}
