<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/DB/DB.php';

final class RrhhAnuncioService
{
    public static function listMine(int $rrhhUserId): array
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT *
            FROM rrhh_anuncios
            WHERE rrhh_user_id=:uid
            ORDER BY estado='publicado' DESC, creado_en DESC, id DESC
        ");
        $st->execute([':uid'=>$rrhhUserId]);
        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    public static function getMine(int $id, int $rrhhUserId): array
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("SELECT * FROM rrhh_anuncios WHERE id=:id AND rrhh_user_id=:uid LIMIT 1");
        $st->execute([':id'=>$id, ':uid'=>$rrhhUserId]);
        return $st->fetch(\PDO::FETCH_ASSOC) ?: [];
    }

    public static function create(int $rrhhUserId, array $in): int
    {
        $pdo = DB::pdo();
        $titulo = trim((string)($in['titulo'] ?? ''));
        $desc   = trim((string)($in['descripcion'] ?? ''));
        $tipo   = (string)($in['tipo'] ?? 'servicio');
        $estado = (string)($in['estado'] ?? 'borrador');
        if ($titulo === '' || $desc === '') throw new \RuntimeException('Título y descripción son requeridos');

        if (!in_array($tipo, ['servicio','oferta','comunicado','otro'], true)) $tipo = 'servicio';
        if (!in_array($estado, ['borrador','publicado','archivado'], true)) $estado = 'borrador';

        $slug = self::slugify($in['slug'] ?? $titulo);

        $pub = null;
        if ($estado === 'publicado') $pub = date('Y-m-d H:i:s');

        $st = $pdo->prepare("
            INSERT INTO rrhh_anuncios
              (rrhh_user_id, titulo, slug, descripcion, tipo, imagen_path, estado, publicado_en, creado_en)
            VALUES
              (:uid,:t,:s,:d,:tipo,:img,:est,:pub,NOW())
        ");
        $st->bindValue(':uid', $rrhhUserId, \PDO::PARAM_INT);
        $st->bindValue(':t', $titulo);
        $st->bindValue(':s', $slug);
        $st->bindValue(':d', $desc);
        $st->bindValue(':tipo', $tipo);
        $img = trim((string)($in['imagen_path'] ?? ''));
        $st->bindValue(':img', $img !== '' ? $img : null, $img !== '' ? \PDO::PARAM_STR : \PDO::PARAM_NULL);
        $st->bindValue(':est', $estado);
        $st->bindValue(':pub', $pub, $pub === null ? \PDO::PARAM_NULL : \PDO::PARAM_STR);
        $st->execute();

        $id = (int)$pdo->lastInsertId();
        self::ensureUniqueSlug($id);

        return $id;
    }

    public static function update(int $id, int $rrhhUserId, array $in): void
    {
        $pdo = DB::pdo();

        $row = self::getMine($id, $rrhhUserId);
        if (!$row) return;

        $titulo = trim((string)($in['titulo'] ?? $row['titulo'] ?? ''));
        $desc   = trim((string)($in['descripcion'] ?? $row['descripcion'] ?? ''));
        $tipo   = (string)($in['tipo'] ?? $row['tipo'] ?? 'servicio');
        $estado = (string)($in['estado'] ?? $row['estado'] ?? 'borrador');
        if (!in_array($tipo, ['servicio','oferta','comunicado','otro'], true)) $tipo = 'servicio';
        if (!in_array($estado, ['borrador','publicado','archivado'], true)) $estado = 'borrador';

        $slug = trim((string)($in['slug'] ?? ''));
        $slug = $slug !== '' ? self::slugify($slug) : self::slugify($titulo);

        $pub = $row['publicado_en'] ?? null;
        if ($estado === 'publicado' && !$pub) $pub = date('Y-m-d H:i:s');
        if ($estado !== 'publicado') $pub = null;

        $img = trim((string)($in['imagen_path'] ?? ''));
        if ($img === '') $img = (string)($row['imagen_path'] ?? '');

        $st = $pdo->prepare("
            UPDATE rrhh_anuncios
            SET titulo=:t, slug=:s, descripcion=:d, tipo=:tipo, imagen_path=:img, estado=:est, publicado_en=:pub
            WHERE id=:id AND rrhh_user_id=:uid
            LIMIT 1
        ");
        $st->bindValue(':t', $titulo);
        $st->bindValue(':s', $slug);
        $st->bindValue(':d', $desc);
        $st->bindValue(':tipo', $tipo);
        $st->bindValue(':img', $img !== '' ? $img : null, $img !== '' ? \PDO::PARAM_STR : \PDO::PARAM_NULL);
        $st->bindValue(':est', $estado);
        $st->bindValue(':pub', $pub, $pub === null ? \PDO::PARAM_NULL : \PDO::PARAM_STR);
        $st->bindValue(':id', $id, \PDO::PARAM_INT);
        $st->bindValue(':uid', $rrhhUserId, \PDO::PARAM_INT);
        $st->execute();

        self::ensureUniqueSlug($id);
    }

    public static function delete(int $id, int $rrhhUserId): void
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("DELETE FROM rrhh_anuncios WHERE id=:id AND rrhh_user_id=:uid LIMIT 1");
        $st->execute([':id'=>$id, ':uid'=>$rrhhUserId]);
    }

    private static function ensureUniqueSlug(int $id): void
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("SELECT id, slug FROM rrhh_anuncios WHERE id=:id LIMIT 1");
        $st->execute([':id'=>$id]);
        $r = $st->fetch(\PDO::FETCH_ASSOC);
        if (!$r) return;
        $slug = (string)($r['slug'] ?? '');
        if ($slug === '') return;

        $st2 = $pdo->prepare("SELECT id FROM rrhh_anuncios WHERE slug=:s AND id<>:id LIMIT 1");
        $st2->execute([':s'=>$slug, ':id'=>$id]);
        if ($st2->fetch(\PDO::FETCH_ASSOC)) {
            $slug = $slug . '-' . $id;
            $u = $pdo->prepare("UPDATE rrhh_anuncios SET slug=:s WHERE id=:id LIMIT 1");
            $u->execute([':s'=>$slug, ':id'=>$id]);
        }
    }

    private static function slugify(string $s): string
    {
        $s = mb_strtolower(trim($s), 'UTF-8');
        $s = preg_replace('~[^\pL\d]+~u', '-', $s) ?? $s;
        $s = trim($s, '-');
        $s = preg_replace('~[^-\w]+~', '', $s) ?? $s;
        $s = preg_replace('~-+~', '-', $s) ?? $s;
        return $s !== '' ? $s : 'anuncio';
    }
}
