<?php
declare(strict_types=1);

require_once dirname(__DIR__) . '/DB/DB.php';

final class RrhhSkillService
{
    public static function listAll(int $rrhhUserId): array
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT * FROM rrhh_skills
            WHERE rrhh_user_id=:uid
            ORDER BY estado='activo' DESC, orden ASC, id DESC
        ");
        $st->execute([':uid'=>$rrhhUserId]);
        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    public static function listActiveGrouped(int $rrhhUserId): array
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT categoria, label, porcentaje, orden
            FROM rrhh_skills
            WHERE rrhh_user_id=:uid AND estado='activo'
            ORDER BY categoria ASC, orden ASC, id ASC
        ");
        $st->execute([':uid'=>$rrhhUserId]);
        $rows = $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];

        $out = [];
        foreach ($rows as $r) {
            $cat = (string)$r['categoria'];
            if (!isset($out[$cat])) $out[$cat] = [];
            $out[$cat][] = $r;
        }
        return $out;
    }

    public static function create(int $rrhhUserId, array $in): int
    {
        $pdo = DB::pdo();
        $cat = trim((string)($in['categoria'] ?? 'General'));
        $lab = trim((string)($in['label'] ?? ''));
        $pct = (int)($in['porcentaje'] ?? 0);
        $ord = (int)($in['orden'] ?? 0);
        $est = (string)($in['estado'] ?? 'activo');

        if ($lab === '') throw new \RuntimeException('Label requerido');
        $pct = max(0, min(100, $pct));
        if (!in_array($est, ['activo','inactivo'], true)) $est = 'activo';

        $st = $pdo->prepare("
            INSERT INTO rrhh_skills (rrhh_user_id, categoria, label, porcentaje, orden, estado, creado_en)
            VALUES (:uid,:cat,:lab,:pct,:ord,:est,NOW())
        ");
        $st->execute([
            ':uid'=>$rrhhUserId, ':cat'=>$cat, ':lab'=>$lab, ':pct'=>$pct, ':ord'=>$ord, ':est'=>$est
        ]);
        return (int)$pdo->lastInsertId();
    }

    public static function update(int $id, int $rrhhUserId, array $in): void
    {
        $pdo = DB::pdo();
        $cat = trim((string)($in['categoria'] ?? 'General'));
        $lab = trim((string)($in['label'] ?? ''));
        $pct = max(0, min(100, (int)($in['porcentaje'] ?? 0)));
        $ord = (int)($in['orden'] ?? 0);
        $est = (string)($in['estado'] ?? 'activo');
        if (!in_array($est, ['activo','inactivo'], true)) $est = 'activo';

        $st = $pdo->prepare("
            UPDATE rrhh_skills
            SET categoria=:cat, label=:lab, porcentaje=:pct, orden=:ord, estado=:est
            WHERE id=:id AND rrhh_user_id=:uid
            LIMIT 1
        ");
        $st->execute([
            ':cat'=>$cat, ':lab'=>$lab, ':pct'=>$pct, ':ord'=>$ord, ':est'=>$est,
            ':id'=>$id, ':uid'=>$rrhhUserId
        ]);
    }

    public static function delete(int $id, int $rrhhUserId): void
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("DELETE FROM rrhh_skills WHERE id=:id AND rrhh_user_id=:uid LIMIT 1");
        $st->execute([':id'=>$id, ':uid'=>$rrhhUserId]);
    }

    public static function get(int $id, int $rrhhUserId): array
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("SELECT * FROM rrhh_skills WHERE id=:id AND rrhh_user_id=:uid LIMIT 1");
        $st->execute([':id'=>$id, ':uid'=>$rrhhUserId]);
        return $st->fetch(\PDO::FETCH_ASSOC) ?: [];
    }
}
