<?php
declare(strict_types=1);

final class AuditLogService
{
    public static function write(
        ?int $actorUserId,
        string $action,
        ?string $entity = null,
        ?int $entityId = null,
        array $meta = []
    ): void {
        $pdo = DB::pdo();

        $ip = self::detectIp();
        $ua = isset($_SERVER['HTTP_USER_AGENT']) ? substr((string)$_SERVER['HTTP_USER_AGENT'], 0, 255) : null;

        $metaJson = null;
        if (!empty($meta)) {
            $metaJson = json_encode($meta, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        }

        $st = $pdo->prepare("
            INSERT INTO audit_logs (actor_user_id, action, entity, entity_id, meta_json, ip, user_agent)
            VALUES (:actor, :action, :entity, :eid, :meta, :ip, :ua)
        ");

        $st->execute([
            'actor'  => $actorUserId,
            'action' => $action,
            'entity' => $entity,
            'eid'    => $entityId,
            'meta'   => $metaJson,
            'ip'     => $ip,
            'ua'     => $ua,
        ]);
    }

    public static function list(array $filters = []): array
    {
        $q      = trim((string)($filters['q'] ?? ''));
        $action = trim((string)($filters['action'] ?? ''));
        $entity = trim((string)($filters['entity'] ?? ''));

        $pdo = DB::pdo();

        $sql = "
          SELECT
            a.*,
            u.email AS actor_email,
            u.nombre AS actor_nombre
          FROM audit_logs a
          LEFT JOIN users u ON u.id = a.actor_user_id
          WHERE 1=1
        ";
        $params = [];

        if ($action !== '') {
            $sql .= " AND a.action = :action";
            $params['action'] = $action;
        }
        if ($entity !== '') {
            $sql .= " AND a.entity = :entity";
            $params['entity'] = $entity;
        }
        if ($q !== '') {
            $sql .= " AND (
              a.action LIKE :q
              OR a.entity LIKE :q
              OR CAST(a.entity_id AS CHAR) LIKE :q
              OR a.ip LIKE :q
              OR u.email LIKE :q
              OR u.nombre LIKE :q
            )";
            $params['q'] = '%' . $q . '%';
        }

        $sql .= " ORDER BY a.id DESC LIMIT 500";

        $st = $pdo->prepare($sql);
        $st->execute($params);

        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    private static function detectIp(): ?string
    {
        $keys = ['HTTP_CF_CONNECTING_IP','HTTP_X_FORWARDED_FOR','REMOTE_ADDR'];
        foreach ($keys as $k) {
            if (!empty($_SERVER[$k])) {
                $v = (string)$_SERVER[$k];
                // si viene lista (XFF), tomar el primero
                $v = trim(explode(',', $v)[0]);
                return $v !== '' ? substr($v, 0, 64) : null;
            }
        }
        return null;
    }
}
