<?php
declare(strict_types=1);

final class EmpresaProfileUpload
{
    public static function processImage(array $file, string $publicDir, string $relativeDir, string $prefix): ?string
    {
        if (empty($file) || !isset($file['tmp_name']) || (int)($file['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
            return null;
        }

        $tmp = (string)$file['tmp_name'];
        if (!is_file($tmp)) return null;

        $info = @getimagesize($tmp);
        if (!$info || empty($info[0]) || empty($info[1])) {
            throw new ValidationException('Imagen inválida.');
        }

        $mime = (string)($info['mime'] ?? '');
        $src = null;

        switch ($mime) {
            case 'image/jpeg': $src = @imagecreatefromjpeg($tmp); break;
            case 'image/png':  $src = @imagecreatefrompng($tmp); break;
            case 'image/webp': $src = function_exists('imagecreatefromwebp') ? @imagecreatefromwebp($tmp) : null; break;
            case 'image/avif': $src = function_exists('imagecreatefromavif') ? @imagecreatefromavif($tmp) : null; break;
            default:
                throw new ValidationException('Formato no soportado. Usá JPG/PNG/WebP.');
        }

        if (!$src) throw new ValidationException('No se pudo leer la imagen.');

        $destAbsDir = rtrim($publicDir, '/\\') . rtrim($relativeDir, '/\\');
        if (!is_dir($destAbsDir)) @mkdir($destAbsDir, 0775, true);
        if (!is_dir($destAbsDir)) throw new ValidationException('No se pudo crear el directorio de uploads.');

        imagepalettetotruecolor($src);
        imagealphablending($src, true);
        imagesavealpha($src, true);

        $id = bin2hex(random_bytes(10));
        $useAvif = function_exists('imageavif');
        $ext = $useAvif ? 'avif' : 'webp';

        $filename = "{$prefix}_{$id}.{$ext}";
        $destAbs  = $destAbsDir . DIRECTORY_SEPARATOR . $filename;

        $ok = $useAvif ? @imageavif($src, $destAbs, 55) : @imagewebp($src, $destAbs, 78);
        imagedestroy($src);

        if (!$ok || !is_file($destAbs)) {
            throw new ValidationException('No se pudo guardar la imagen.');
        }

        return rtrim($relativeDir, '/\\') . '/' . $filename;
    }
}
