<?php
declare(strict_types=1);

final class ImageUpload
{
    /**
     * Procesa un upload de imagen y devuelve path relativo para guardar en DB.
     * - Intenta AVIF (imageavif) si existe
     * - Si no, WebP (imagewebp)
     */
    // Nota: en este proyecto los uploads viven en /public/upload/vacantes (sin "s").
    // Guardamos en DB un path público relativo (ej: /upload/vacantes/xxx.avif) para usar base_path() + imagen_path.
    public static function processVacanteImage(array $file, string $publicDir, string $relativeDir = '/upload/vacantes'): ?string
    {
        if (empty($file) || !isset($file['tmp_name']) || (int)($file['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
            return null;
        }

        $tmp = (string)$file['tmp_name'];
        if (!is_file($tmp)) return null;

        $info = @getimagesize($tmp);
        if (!$info || empty($info[0]) || empty($info[1])) {
            throw new ValidationException('Imagen inválida.');
        }

        $mime = (string)($info['mime'] ?? '');
        $src = null;

        switch ($mime) {
            case 'image/jpeg': $src = @imagecreatefromjpeg($tmp); break;
            case 'image/png':  $src = @imagecreatefrompng($tmp); break;
            case 'image/webp': $src = function_exists('imagecreatefromwebp') ? @imagecreatefromwebp($tmp) : null; break;
            case 'image/avif':
                // GD no siempre puede leer AVIF como source en todas las builds.
                // Si no puede, rechazamos para evitar guardar basura.
                $src = function_exists('imagecreatefromavif') ? @imagecreatefromavif($tmp) : null;
                break;
            default:
                throw new ValidationException('Formato no soportado. Usá JPG/PNG/WebP.');
        }

        if (!$src) throw new ValidationException('No se pudo leer la imagen (GD/AVIF/WebP).');

        $destAbsDir = rtrim($publicDir, '/\\') . rtrim($relativeDir, '/\\');
        if (!is_dir($destAbsDir)) {
            @mkdir($destAbsDir, 0775, true);
        }
        if (!is_dir($destAbsDir)) {
            throw new ValidationException('No se pudo crear el directorio de uploads.');
        }

        // Mantener tamaño original (no limitar), pero optimizar:
        // - si es PNG con alpha, preserva alpha.
        imagepalettetotruecolor($src);
        imagealphablending($src, true);
        imagesavealpha($src, true);

        $id = bin2hex(random_bytes(10));
        $useAvif = function_exists('imageavif');
        $ext = $useAvif ? 'avif' : 'webp';

        $filename = "vacante_{$id}.{$ext}";
        $destAbs = $destAbsDir . DIRECTORY_SEPARATOR . $filename;

        if ($useAvif) {
            // calidad 45–60 suele ser buen balance. Subí si querés más calidad.
            $ok = @imageavif($src, $destAbs, 55);
        } else {
            // webp quality 70–82 suele ser buen balance
            $ok = @imagewebp($src, $destAbs, 78);
        }

        imagedestroy($src);

        if (!$ok || !is_file($destAbs)) {
            throw new ValidationException('No se pudo optimizar/guardar la imagen.');
        }

        // path relativo público
        $relativePath = rtrim($relativeDir, '/\\') . '/' . $filename;
        return $relativePath;
    }
}
