<?php
declare(strict_types=1);

final class NotificationService
{
    public static function create(int $userId, string $type, string $title, ?string $body = null, ?string $url = null): void
    {
        $pdo = DB::pdo();

        $st = $pdo->prepare("
            INSERT INTO notifications (user_id, type, title, body, url)
            VALUES (:uid, :type, :title, :body, :url)
        ");
        $st->execute([
            'uid'   => $userId,
            'type'  => $type,
            'title' => mb_substr($title, 0, 160),
            'body'  => $body,
            'url'   => $url !== null ? mb_substr($url, 0, 255) : null,
        ]);
    }

    public static function listForUser(int $userId, int $limit = 50): array
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("
            SELECT *
            FROM notifications
            WHERE user_id = :uid
            ORDER BY id DESC
            LIMIT :lim
        ");
        $st->bindValue(':uid', $userId, \PDO::PARAM_INT);
        $st->bindValue(':lim', $limit, \PDO::PARAM_INT);
        $st->execute();
        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    public static function markRead(int $userId, int $id): void
    {
        $pdo = DB::pdo();
        $st = $pdo->prepare("
            UPDATE notifications
            SET is_read = 1, read_at = COALESCE(read_at, NOW())
            WHERE id = :id AND user_id = :uid
        ");
        $st->execute(['id' => $id, 'uid' => $userId]);
    }

    public static function markAllRead(int $userId): void
    {
        $pdo = DB::pdo();
        $pdo->prepare("
            UPDATE notifications
            SET is_read = 1, read_at = COALESCE(read_at, NOW())
            WHERE user_id = :uid AND is_read = 0
        ")->execute(['uid' => $userId]);
    }
}
