<?php
declare(strict_types=1);

final class CategoriaLaboralService
{
    /**
     * Listado para UI / selectores.
     * Columnas reales (dump): id, parent_id, codigo, nombre, descripcion, keywords, orden, estado, creado_en
     */
    public static function listActivas(int $limit = 500): array
    {
        $pdo = DB::pdo();
        $limit = max(1, min(2000, (int)$limit));

        $st = $pdo->prepare("
            SELECT id, parent_id, codigo, nombre, descripcion, keywords, orden, estado
            FROM categorias_laborales
            WHERE estado = 'activo'
            ORDER BY orden ASC, id ASC
            LIMIT :lim
        ");
        $st->bindValue(':lim', $limit, \PDO::PARAM_INT);
        $st->execute();
        return $st->fetchAll(\PDO::FETCH_ASSOC) ?: [];
    }

    /**
     * Alias usado en app.php (feed/roles).
     * Mantener firma estable para evitar 500 por método inexistente.
     */
    public static function listForFeed(int $limit = 500): array
    {
        return self::listActivas($limit);
    }

    /**
     * IDs seleccionados por el usuario (tabla: user_categorias).
     */
    public static function userSelectedIds(int $userId, int $limit = 200): array
    {
        $pdo = DB::pdo();
        $userId = (int)$userId;
        if ($userId <= 0) return [];

        $limit = max(1, min(1000, (int)$limit));

        $st = $pdo->prepare("
            SELECT categoria_id
            FROM user_categorias
            WHERE user_id = :uid
            ORDER BY categoria_id ASC
            LIMIT :lim
        ");
        $st->bindValue(':uid', $userId, \PDO::PARAM_INT);
        $st->bindValue(':lim', $limit, \PDO::PARAM_INT);
        $st->execute();

        $ids = $st->fetchAll(\PDO::FETCH_COLUMN) ?: [];
        $out = [];
        foreach ($ids as $v) {
            $n = (int)$v;
            if ($n > 0) $out[] = $n;
        }
        return array_values(array_unique($out));
    }

    /**
     * Guarda selección del usuario (tabla: user_categorias) reemplazando lo anterior.
     * Alias usado en app.php.
     */
    public static function saveUserSelections(int $userId, array $categoriaIds): void
    {
        $pdo = DB::pdo();
        $userId = (int)$userId;
        if ($userId <= 0) throw new ValidationException('Usuario inválido.');

        $ids = array_values(array_filter(array_map('intval', $categoriaIds), fn($v) => $v > 0));
        $ids = array_values(array_unique($ids));
        $ids = array_slice($ids, 0, 25); // hard cap anti abuso

        $pdo->beginTransaction();
        try {
            $pdo->prepare("DELETE FROM user_categorias WHERE user_id = :uid")->execute(['uid' => $userId]);

            if ($ids) {
                $st = $pdo->prepare("INSERT INTO user_categorias (user_id, categoria_id) VALUES (:uid, :cid)");
                foreach ($ids as $cid) {
                    $st->execute(['uid' => $userId, 'cid' => (int)$cid]);
                }
            }

            $pdo->commit();
        } catch (\Throwable $e) {
            $pdo->rollBack();
            throw $e;
        }
    }
}
