<?php declare(strict_types=1);
/** @var string|null $error */
/** @var array $row */
/** @var string $csrf */
/** @var array $categorias */
/** @var array $oldCategorias */

$bp = base_path();
$categorias = $categorias ?? [];
$oldCategorias = $oldCategorias ?? [];

if (!function_exists('h')) {
  function h(string $s): string { return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
}

$imgPath = (string)($row['imagen_path'] ?? '');
// Normalización: en DB puede venir "/public/upload/..." o "/upload/...".
// El docroot ya es /public, así que removemos el prefijo /public si existe.
if ($imgPath !== '' && str_starts_with($imgPath, '/public/')) {
  $imgPath = substr($imgPath, 7); // quita "/public"
}
$imgUrl = $imgPath !== '' ? ($bp . $imgPath) : '';
?>

<style>
  .editor-wrap{ border:1px solid #e8eefc; border-radius:12px; overflow:hidden; background:#fff; }
  .editor-toolbar{ display:flex; flex-wrap:wrap; gap:6px; padding:10px; border-bottom:1px solid #e8eefc; background:#f7f9ff; }
  .editor-toolbar button{ border:1px solid #e8eefc; background:#fff; border-radius:10px; padding:6px 10px; font-size:14px; }
  .editor-toolbar button:hover{ background:#f1f5ff; }
  .editor-area{ min-height:220px; padding:12px; outline:none; }
  .editor-area:empty:before{ content: attr(data-placeholder); color:#8896b1; }
  .chip{ display:inline-flex; align-items:center; gap:8px; padding:6px 10px; border:1px solid #e8eefc; border-radius:999px; background:#fff; font-size:13px; }
  .chip b{ font-weight:800; }
  .cat-box{ border:1px solid #e8eefc; border-radius:12px; background:#fff; }
  .cat-top{ padding:10px; border-bottom:1px solid #e8eefc; background:#f7f9ff; border-radius:12px 12px 0 0; }
  .cat-list{ max-height:260px; overflow:auto; padding:10px; }
  .cat-item{ display:flex; align-items:center; gap:10px; padding:8px 10px; border-radius:10px; }
  .cat-item:hover{ background:#f6f8ff; }
  .cat-item input{ width:18px; height:18px; }
  .img-preview{ width:100%; max-width:420px; aspect-ratio:1080/1180; border:1px dashed #cfd8ee; border-radius:14px; display:flex; align-items:center; justify-content:center; overflow:hidden; background:#fbfcff; }
  .img-preview img{ width:100%; height:100%; object-fit:cover; }
</style>

<div class="container py-4">
  <div class="d-flex align-items-center justify-content-between mb-3">
    <div>
      <h3 class="mb-0">Editar vacante</h3>
      <div class="small text-muted">ID #<?= (int)$row['id'] ?> — Estado: <?= h((string)$row['estado']) ?></div>
    </div>
    <a class="btn btn-outline-secondary" href="<?= h($bp . '/empresa/vacantes') ?>">Volver</a>
  </div>

  <?php if (!empty($error)): ?>
    <div class="alert alert-danger"><?= h((string)$error) ?></div>
  <?php endif; ?>

  <div class="card">
    <div class="card-body">

      <form method="post"
            action="<?= h($bp . '/empresa/vacantes/edit') ?>"
            enctype="multipart/form-data"
            class="row g-3">

        <input type="hidden" name="_csrf" value="<?= h((string)$csrf) ?>">
        <input type="hidden" name="id" value="<?= (int)$row['id'] ?>">

        <div class="col-12">
          <label class="form-label">Título</label>
          <input class="form-control" name="titulo" required value="<?= h((string)$row['titulo']) ?>">
        </div>

        <!-- Categorías buscables -->
        <div class="col-12">
          <label class="form-label">Categorías laborales</label>

          <div class="cat-box">
            <div class="cat-top">
              <div class="row g-2 align-items-center">
                <div class="col-12 col-lg-6">
                  <input id="catSearch" class="form-control" placeholder="Buscar categoría... (ej: Contador, Finanzas, IT)">
                </div>
                <div class="col-12 col-lg-6">
                  <div class="d-flex flex-wrap gap-2" id="catChips"></div>
                </div>
              </div>
              <div class="form-text mt-2">
                Seleccioná 1 o más categorías. Esto alimenta el matching y visibilidad del feed.
              </div>
            </div>

            <div class="cat-list" id="catList">
              <?php foreach ($categorias as $c): ?>
                <?php
                  $cid = (int)($c['id'] ?? 0);
                  $cn  = (string)($c['nombre'] ?? '');
                  $sel = in_array($cid, $oldCategorias, true);
                ?>
                <label class="cat-item" data-name="<?= h(mb_strtolower($cn)) ?>">
                  <input type="checkbox" name="categoria_ids[]" value="<?= $cid ?>" <?= $sel ? 'checked' : '' ?>>
                  <div><?= h($cn) ?></div>
                </label>
              <?php endforeach; ?>
            </div>
          </div>
        </div>

        <!-- Imagen -->
        <div class="col-12">
          <label class="form-label">Imagen (opcional)</label>

          <div class="row g-3 align-items-start">
            <div class="col-12 col-lg-5">
              <div class="img-preview" id="imgPreviewBox">
                <div class="text-muted small text-center px-3" id="imgPreviewEmpty" style="<?= $imgUrl ? 'display:none;' : '' ?>">
                  Recomendado: <strong>1080 x 1180</strong><br>
                  (se mostrará ajustada en el feed)
                </div>

                <?php if ($imgUrl): ?>
                  <img id="imgCurrent" src="<?= h($imgUrl) ?>" alt="Imagen actual">
                <?php else: ?>
                  <img id="imgCurrent" src="" alt="" style="display:none;">
                <?php endif; ?>

                <img id="imgPreview" src="" alt="" style="display:none;">
              </div>

              <div class="form-text mt-2">
                Al guardar, el backend optimiza a <strong>AVIF</strong> (si tu PHP lo soporta) o <strong>WebP</strong>.
              </div>
            </div>

            <div class="col-12 col-lg-7">
              <input class="form-control" type="file" name="imagen" accept="image/*" id="imgInput">
              <div class="form-text">
                Subí una nueva para reemplazar. Si no subís, se mantiene la actual.
              </div>
            </div>
          </div>
        </div>

        <!-- Editor -->
        <div class="col-12">
          <label class="form-label">Descripción</label>

          <div class="editor-wrap">
            <div class="editor-toolbar">
              <button type="button" data-cmd="bold" title="Negrita"><strong>B</strong></button>
              <button type="button" data-cmd="italic" title="Cursiva"><em>I</em></button>
              <button type="button" data-cmd="underline" title="Subrayado"><u>U</u></button>
              <span class="mx-1"></span>
              <button type="button" data-cmd="insertUnorderedList" title="Lista">• Lista</button>
              <button type="button" data-cmd="insertOrderedList" title="Numerada">1. Lista</button>
              <span class="mx-1"></span>
              <button type="button" id="btnLink" title="Enlace">🔗 Link</button>
              <button type="button" id="btnClear" title="Limpiar formato">🧽 Limpiar</button>
              <div class="ms-auto text-muted small d-flex align-items-center gap-2">
                <span class="chip"><span>Chars:</span><b id="charCount">0</b></span>
              </div>
            </div>

            <div id="editor" class="editor-area" contenteditable="true"
                 data-placeholder="Describe funciones, requisitos, beneficios..."><?= (string)($row['descripcion'] ?? '') ?></div>
          </div>

          <textarea class="form-control mt-2" name="descripcion" id="descripcion"
                    rows="6" required style="display:none;"><?= h((string)($row['descripcion'] ?? '')) ?></textarea>

          <div class="form-text">Mínimo recomendado: 20 caracteres.</div>
        </div>

        <div class="col-md-6">
          <label class="form-label">Ubicación</label>
          <input class="form-control" name="ubicacion" value="<?= h((string)($row['ubicacion'] ?? '')) ?>">
        </div>

        <div class="col-md-3">
          <label class="form-label">Modalidad</label>
          <?php $m = (string)($row['modalidad'] ?? 'presencial'); ?>
          <select class="form-select" name="modalidad">
            <option value="presencial" <?= $m==='presencial'?'selected':'' ?>>Presencial</option>
            <option value="remoto" <?= $m==='remoto'?'selected':'' ?>>Remoto</option>
            <option value="hibrido" <?= $m==='hibrido'?'selected':'' ?>>Híbrido</option>
          </select>
        </div>

        <div class="col-md-3">
          <label class="form-label">Tipo empleo</label>
          <?php $t = (string)($row['tipo_empleo'] ?? 'tiempo_completo'); ?>
          <select class="form-select" name="tipo_empleo">
            <option value="tiempo_completo" <?= $t==='tiempo_completo'?'selected':'' ?>>Tiempo completo</option>
            <option value="medio_tiempo" <?= $t==='medio_tiempo'?'selected':'' ?>>Medio tiempo</option>
            <option value="temporal" <?= $t==='temporal'?'selected':'' ?>>Temporal</option>
            <option value="freelance" <?= $t==='freelance'?'selected':'' ?>>Freelance</option>
            <option value="practica" <?= $t==='practica'?'selected':'' ?>>Práctica</option>
          </select>
        </div>

        <div class="col-md-3">
          <label class="form-label">Salario mín.</label>
          <input class="form-control" name="salario_min" inputmode="decimal"
                 value="<?= h((string)($row['salario_min'] ?? '')) ?>">
        </div>

        <div class="col-md-3">
          <label class="form-label">Salario máx.</label>
          <input class="form-control" name="salario_max" inputmode="decimal"
                 value="<?= h((string)($row['salario_max'] ?? '')) ?>">
        </div>

        <div class="col-md-2">
          <label class="form-label">Moneda</label>
          <?php $mon = strtoupper((string)($row['moneda'] ?? 'CRC')); ?>
          <select class="form-select" name="moneda">
            <option value="CRC" <?= $mon==='CRC'?'selected':'' ?>>CRC</option>
            <option value="USD" <?= $mon==='USD'?'selected':'' ?>>USD</option>
          </select>
        </div>

        <div class="col-12 d-flex gap-2">
          <button class="btn btn-primary" type="submit">Guardar cambios</button>
          <a class="btn btn-light" href="<?= h($bp . '/empresa/vacantes') ?>">Cancelar</a>
        </div>
      </form>

      <div class="mt-3 d-flex gap-2 flex-wrap">
        <?php if (($row['estado'] ?? '') !== 'publicada' && ($row['estado'] ?? '') !== 'cerrada'): ?>
          <form method="post" action="<?= h($bp . '/empresa/vacantes/publish') ?>" class="d-inline">
            <input type="hidden" name="_csrf" value="<?= h((string)$csrf) ?>">
            <input type="hidden" name="id" value="<?= (int)$row['id'] ?>">
            <button class="btn btn-success" type="submit">Publicar</button>
          </form>
        <?php endif; ?>

        <?php if (($row['estado'] ?? '') !== 'cerrada'): ?>
          <form method="post" action="<?= h($bp . '/empresa/vacantes/close') ?>" class="d-inline"
                onsubmit="return confirm('¿Cerrar esta vacante?');">
            <input type="hidden" name="_csrf" value="<?= h((string)$csrf) ?>">
            <input type="hidden" name="id" value="<?= (int)$row['id'] ?>">
            <button class="btn btn-outline-secondary" type="submit">Cerrar</button>
          </form>
        <?php endif; ?>
      </div>

    </div>
  </div>
</div>

<script>
(function(){
  // ===== Categorías: buscador + chips =====
  const search = document.getElementById('catSearch');
  const list = document.getElementById('catList');
  const chips = document.getElementById('catChips');

  function rebuildChips(){
    chips.innerHTML = '';
    const checked = list.querySelectorAll('input[type="checkbox"]:checked');
    checked.forEach(cb=>{
      const label = cb.closest('.cat-item');
      const txt = label ? label.innerText.trim() : cb.value;
      const chip = document.createElement('span');
      chip.className = 'chip';
      chip.innerHTML = '<b>✓</b>' + txt;
      chips.appendChild(chip);
    });
  }

  list.addEventListener('change', rebuildChips);
  rebuildChips();

  search.addEventListener('input', ()=>{
    const q = (search.value || '').trim().toLowerCase();
    list.querySelectorAll('.cat-item').forEach(row=>{
      const name = row.getAttribute('data-name') || '';
      row.style.display = (q === '' || name.includes(q)) ? '' : 'none';
    });
  });

  // ===== Editor =====
  const editor = document.getElementById('editor');
  const textarea = document.getElementById('descripcion');
  const charCount = document.getElementById('charCount');

  function syncToTextarea(){
    textarea.value = editor.innerHTML.trim();
    charCount.textContent = (editor.innerText || '').trim().length.toString();
  }

  document.querySelectorAll('.editor-toolbar button[data-cmd]').forEach(btn=>{
    btn.addEventListener('click', ()=>{
      const cmd = btn.getAttribute('data-cmd');
      document.execCommand(cmd, false, null);
      editor.focus();
      syncToTextarea();
    });
  });

  document.getElementById('btnLink').addEventListener('click', ()=>{
    const url = prompt('URL (https://...)');
    if (!url) return;
    document.execCommand('createLink', false, url);
    editor.focus();
    syncToTextarea();
  });

  document.getElementById('btnClear').addEventListener('click', ()=>{
    document.execCommand('removeFormat', false, null);
    editor.querySelectorAll('script,style').forEach(n=>n.remove());
    editor.focus();
    syncToTextarea();
  });

  editor.addEventListener('input', syncToTextarea);
  syncToTextarea();

  // ===== Imagen preview =====
  const imgInput = document.getElementById('imgInput');
  const imgPreview = document.getElementById('imgPreview');
  const imgCurrent = document.getElementById('imgCurrent');
  const empty = document.getElementById('imgPreviewEmpty');

  imgInput.addEventListener('change', ()=>{
    const f = imgInput.files && imgInput.files[0];
    if (!f) {
      imgPreview.style.display = 'none';
      imgPreview.src = '';
      return;
    }
    const url = URL.createObjectURL(f);
    imgPreview.src = url;
    imgPreview.style.display = 'block';
    if (imgCurrent) imgCurrent.style.display = 'none';
    if (empty) empty.style.display = 'none';
  });
})();
</script>
