<?php
/** @var array $empresa */
/** @var array $perfil */
/** @var array $vacantes */
/** @var int   $total */

$bp = base_path();

$empresaId = (int)($empresa['id'] ?? 0);
$nombre    = (string)($empresa['nombre'] ?? ($empresa['name'] ?? ''));
$email     = (string)($empresa['email'] ?? '');

$telefono  = (string)($perfil['telefono'] ?? '');
$whatsapp  = (string)($perfil['whatsapp'] ?? '');
$ubic      = (string)($perfil['ubicacion'] ?? '');
$bio       = (string)($perfil['bio'] ?? '');

$web            = (string)($perfil['website_url'] ?? '');
$googleBusiness = (string)($perfil['google_business_url'] ?? '');

/**
 * OJO: tu sistema ya migró a paths en DB (no urls):
 * - logo_path / banner_path
 * pero puede quedar legacy en logo_url/banner_url.
 */
$logoRaw   = (string)($perfil['logo_path']   ?? ($perfil['logo_url'] ?? ''));
$bannerRaw = (string)($perfil['banner_path'] ?? ($perfil['banner_url'] ?? ''));

/**
 * Normaliza a PATH público correcto en TU hosting:
 * - Si viene "/upload/..." => "/public/upload/..."
 * - Si viene "/public/upload/..." => se deja igual
 * - Evita "/public/public/..."
 */
function normalize_public_asset_path(string $p): string {
    $p = trim($p);
    if ($p === '') return '';

    // si viene URL completa, la dejamos (por si aún tenés datos legacy en logo_url/banner_url)
    if (preg_match('~^https?://~i', $p)) {
        // quitar doble /public/public si existiera en la URL
        $p = str_replace('/public/public/', '/public/', $p);
        return $p;
    }

    // asegurar slash inicial
    if (!str_starts_with($p, '/')) $p = '/' . $p;

    // normalizar dobles
    $p = str_replace('//', '/', $p);
    $p = str_replace('/public/public/', '/public/', $p);

    // caso correcto ya servido por tu hosting
    if (str_starts_with($p, '/public/upload/')) return $p;

    // caso DB guarda /upload/... pero tu hosting sirve /public/upload/...
    if (str_starts_with($p, '/upload/')) return '/public' . $p;

    // si ya venía /public/ pero no /public/upload/ igual lo dejamos
    if (str_starts_with($p, '/public/')) return $p;

    // fallback: forzar /public/ + path
    return '/public' . $p;
}

/**
 * Convierte el PATH normalizado en URL final.
 * Si ya era URL (https://...), devuelve tal cual.
 */
function asset_url(string $bp, string $raw): string {
    $raw = trim($raw);
    if ($raw === '') return '';
    if (preg_match('~^https?://~i', $raw)) return $raw;

    $p = normalize_public_asset_path($raw);

    // evitar /public/public en el armado final
    $url = rtrim($bp, '/') . $p;
    $url = str_replace('/public/public/', '/public/', $url);
    $url = str_replace('//', '/', $url);

    // reponer esquema en caso de que quedara "https:/" (no debería aquí)
    return $url;
}

$logoUrl   = asset_url($bp, $logoRaw);
$bannerUrl = asset_url($bp, $bannerRaw);

// links acciones
$webOk   = (preg_match('~^https?://~i', $web) ? $web : '');
$gbOk    = (preg_match('~^https?://~i', $googleBusiness) ? $googleBusiness : '');

$telDigits = preg_replace('~[^\d+]~', '', $telefono);
$waDigits  = preg_replace('~[^\d]~', '', $whatsapp);
if ($waDigits !== '' && !str_starts_with($waDigits, '506') && strlen($waDigits) === 8) {
    // CR típico: si te pasan 8 dígitos, asumimos +506
    $waDigits = '506' . $waDigits;
}
$waLink = $waDigits !== '' ? ('https://wa.me/' . $waDigits) : '';
?>
<style>
  .empresa-hero{border-radius:18px;overflow:hidden;border:1px solid #eef2ff;background:#fff}
  .empresa-banner{height:240px;background:#0b1220;position:relative}
  .empresa-banner img{width:100%;height:100%;object-fit:cover;display:block}
  .empresa-banner::after{
    content:"";position:absolute;inset:0;
    background:linear-gradient(180deg, rgba(0,0,0,.15), rgba(0,0,0,.55));
  }
  .empresa-head{padding:18px}
  .empresa-top{display:flex;gap:16px;align-items:flex-start;flex-wrap:wrap}
  .empresa-logo{
    width:110px;height:110px;border-radius:18px;overflow:hidden;
    border:1px solid #e8eefc;background:#fff;flex:0 0 auto;
    display:flex;align-items:center;justify-content:center
  }
  .empresa-logo img{width:100%;height:100%;object-fit:contain;display:block}
  .empresa-title{flex:1 1 320px}
  .empresa-title h1{font-size:28px;margin:0}
  .empresa-meta{color:#6b7a99;font-size:13px;margin-top:6px}
  .chips{display:flex;gap:8px;flex-wrap:wrap;margin-top:10px}
  .chip{border:1px solid #e8eefc;background:#f7f9ff;border-radius:999px;padding:6px 10px;font-size:13px}
  .actions{display:flex;gap:10px;flex-wrap:wrap;margin-top:14px}
  .actions a{border-radius:12px}
  .bio{padding:16px 18px;border-top:1px solid #eef2ff}
  .bio .label{color:#6b7a99;font-size:12px;margin-bottom:6px}
</style>

<div class="container py-4">

  <!-- Header Empresa -->
  <div class="empresa-hero mb-4">

    <?php if ($bannerUrl !== ''): ?>
      <div class="empresa-banner">
        <img
          src="<?= e($bannerUrl) ?>"
          alt="Banner empresa"
          loading="lazy"
          onerror="this.style.display='none';"
        >
      </div>
    <?php endif; ?>

    <div class="empresa-head">
      <div class="empresa-top">
        <div class="empresa-logo">
          <?php if ($logoUrl !== ''): ?>
            <img
              src="<?= e($logoUrl) ?>"
              alt="Logo empresa"
              loading="lazy"
              onerror="this.style.display='none';"
            >
          <?php else: ?>
            <span class="text-muted small">Sin logo</span>
          <?php endif; ?>
        </div>

        <div class="empresa-title">
          <h1><?= e($nombre !== '' ? $nombre : ('Empresa #' . $empresaId)) ?></h1>

          <div class="empresa-meta">
            <?= e($email) ?>
            <?php if ($ubic !== ''): ?> • <?= e($ubic) ?><?php endif; ?>
          </div>

          <div class="chips">
            <span class="chip">Vacantes publicadas: <strong><?= (int)$total ?></strong></span>
            <?php if ($webOk !== ''): ?><span class="chip">Website</span><?php endif; ?>
            <?php if ($gbOk !== ''): ?><span class="chip">Google Business</span><?php endif; ?>
          </div>

          <div class="actions">
            <?php if ($webOk !== ''): ?>
              <a class="btn btn-sm btn-outline-primary" target="_blank" rel="noopener" href="<?= e($webOk) ?>">Website</a>
            <?php endif; ?>

            <?php if ($gbOk !== ''): ?>
              <a class="btn btn-sm btn-outline-secondary" target="_blank" rel="noopener" href="<?= e($gbOk) ?>">Google Business</a>
            <?php endif; ?>

            <?php if ($waLink !== ''): ?>
              <a class="btn btn-sm btn-success" target="_blank" rel="noopener" href="<?= e($waLink) ?>">WhatsApp</a>
            <?php endif; ?>

            <?php if ($telDigits !== ''): ?>
              <a class="btn btn-sm btn-outline-dark" href="<?= e('tel:' . $telDigits) ?>">Llamar</a>
            <?php endif; ?>
          </div>
        </div>
      </div>
    </div>

    <?php if (trim($bio) !== ''): ?>
      <div class="bio">
        <div class="label">Sobre la empresa</div>
        <div><?= nl2br(e($bio)) ?></div>
      </div>
    <?php endif; ?>
  </div>

  <!-- Vacantes -->
  <div class="card">
    <div class="card-body">
      <div class="fw-semibold mb-3">Vacantes publicadas</div>

      <?php if (empty($vacantes)): ?>
        <div class="alert alert-light border mb-0">
          Esta empresa no tiene vacantes publicadas actualmente.
        </div>
      <?php else: ?>
        <div class="row g-3">
          <?php foreach ($vacantes as $it): ?>
            <?php
              $id = (int)($it['id'] ?? 0);
              $mon  = (string)($it['moneda'] ?? 'CRC');
              $smin = $it['salario_min'] ?? null;
              $smax = $it['salario_max'] ?? null;

              $sal = '—';
              if ($smin !== null || $smax !== null) {
                $a = $smin !== null ? number_format((float)$smin, 2) : '—';
                $b = $smax !== null ? number_format((float)$smax, 2) : '—';
                $sal = "{$mon} {$a} - {$b}";
              }
            ?>
            <div class="col-12 col-md-6 col-lg-4">
              <div class="card h-100">
                <div class="card-body">
                  <div class="fw-semibold mb-1"><?= e((string)($it['titulo'] ?? '')) ?></div>
                  <div class="small text-muted mb-2">
                    <?= e((string)($it['ubicacion'] ?? '—')) ?> •
                    <?= e((string)($it['modalidad'] ?? '—')) ?> •
                    <?= e((string)($it['tipo_empleo'] ?? '—')) ?>
                  </div>
                  <div class="small">Salario: <?= e($sal) ?></div>
                </div>
                <div class="card-footer bg-white border-0 pt-0">
                  <a class="btn btn-sm btn-outline-primary w-100"
                     href="<?= e(url_path('vacantes/ver') . '?id=' . $id) ?>">
                    Ver detalle
                  </a>
                </div>
              </div>
            </div>
          <?php endforeach; ?>
        </div>
      <?php endif; ?>
    </div>
  </div>

</div>
