<?php
/** @var int $userId */
/** @var string $tipo */
/** @var string $q */
/** @var int $only */
/** @var array $cats */
/** @var array $catSel */
/** @var array $items */
/** @var array $stats */
/** @var string $csrf */
/** @var array $ideales */

$bp = base_path();

// ===== Helpers (assets, tiempo, HTML seguro) =====
if (!function_exists('feed_asset_url')) {
  /**
   * En este proyecto base_path() ya incluye "/public".
   * DB guarda "/upload/..." y la URL final debe ser "$bp/upload/...".
   * Si llega "/public/upload/...", se recorta a "/upload/..." para evitar "/public/public/...".
   */
  function feed_asset_url(string $bp, ?string $path): string {
    $p = trim((string)$path);
    if ($p === '') return '';
    $p = str_replace('\\', '/', $p);

    if (preg_match('~^https?://~i', $p)) return $p;

    // eliminar duplicados típicos
    $p = str_replace('/public/public/', '/public/', $p);

    // si DB guardó /public/upload/... => lo pasamos a /upload/...
    if (str_starts_with($p, '/public/upload/')) {
      $p = substr($p, 7); // quita "/public"
    }

    // si DB guardó public/upload/... => /upload/...
    if (str_starts_with($p, 'public/upload/')) {
      $p = '/' . substr($p, 7);
    }

    // si DB guardó upload/... => /upload/...
    if (str_starts_with($p, 'upload/')) {
      $p = '/' . $p;
    }

    // si DB guardó /upload/... => OK
    if (!str_starts_with($p, '/')) $p = '/' . $p;

    // base_path ya trae /public
    return $bp . $p;
  }
}

if (!function_exists('feed_time_ago')) {
  function feed_time_ago(?string $dt): string {
    if (!$dt) return '—';
    try { $d = new DateTime($dt); } catch (Throwable $e) { return '—'; }
    $now = new DateTime('now');
    $sec = $now->getTimestamp() - $d->getTimestamp();
    if ($sec < 0) $sec = 0;

    if ($sec < 60) return 'hace 1 min';
    if ($sec < 3600) return 'hace ' . (int)floor($sec/60) . ' min';
    if ($sec < 86400) return 'hace ' . (int)floor($sec/3600) . ' h';

    $days = (int)floor($sec/86400);
    if ($days < 7) return 'hace ' . $days . ' día' . ($days === 1 ? '' : 's');

    $weeks = (int)floor($days/7);
    if ($weeks < 4) return 'hace ' . $weeks . ' semana' . ($weeks === 1 ? '' : 's');

    $months = (int)floor($days/30);
    if ($months < 12) return 'hace ' . $months . ' mes' . ($months === 1 ? '' : 'es');

    $years = (int)floor($days/365);
    return 'hace ' . $years . ' año' . ($years === 1 ? '' : 's');
  }
}

if (!function_exists('feed_safe_html_excerpt')) {
  function feed_safe_html_excerpt(string $html, int $maxChars = 420): string {
    $html = trim($html);
    if ($html === '') return '';

    // remover cosas peligrosas
    $html = preg_replace('~<(script|iframe|object|embed|style)\b[^>]*>.*?</\1>~is', '', $html) ?? '';

    $allowed = '<p><br><b><strong><i><em><ul><ol><li><a><blockquote><code><pre>';
    $clean = strip_tags($html, $allowed);

    // remover on* y style
    $clean = preg_replace('~\son\w+\s*=\s*(".*?"|\'.*?\'|[^\s>]+)~i', '', $clean) ?? '';
    $clean = preg_replace('~\sstyle\s*=\s*(".*?"|\'.*?\')~i', '', $clean) ?? '';

    // sanitizar href
    $clean = preg_replace_callback(
      '~<a\s+([^>]*?)href\s*=\s*("|\')(.*?)\2([^>]*)>~i',
      function($m){
        $before = $m[1] ?? '';
        $q      = $m[2] ?? '"';
        $href   = trim((string)($m[3] ?? ''));
        $after  = $m[4] ?? '';
    
        if (preg_match('~^(javascript:|data:)~i', $href)) $href = '#';
        if ($href !== '#' && !preg_match('~^(https?://|/)~i', $href)) $href = '#';
    
        $hrefEsc = htmlspecialchars($href, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
        return '<a ' . $before . 'href=' . $q . $hrefEsc . $q . $after . ' target="_blank" rel="noopener noreferrer">';
      },
      $clean
    );
    $textOnly = trim(preg_replace('~\s+~', ' ', strip_tags($clean)));
    if (mb_strlen($textOnly, 'UTF-8') <= $maxChars) return $clean;

    $tr = mb_strimwidth($textOnly, 0, $maxChars, '…', 'UTF-8');
    return '<p>' . htmlspecialchars($tr, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . '</p>';
  }
}

// defaults
$only   = isset($only) ? (int)$only : 0;
$cats   = isset($cats) && is_array($cats) ? $cats : [];
$catSel = isset($catSel) && is_array($catSel) ? $catSel : [];
$ideales = isset($ideales) && is_array($ideales) ? $ideales : [];

$catSelIds = array_values(array_filter(array_map('intval', (array)$catSel), fn($v) => $v > 0));
$catSelMap = array_flip($catSelIds);

$fmtMoney = function ($mon, $smin, $smax): string {
  $mon = (string)($mon ?: 'CRC');
  if ($smin === null && $smax === null) return '—';
  $a = $smin !== null ? number_format((float)$smin, 2) : '—';
  $b = $smax !== null ? number_format((float)$smax, 2) : '—';
  return "{$mon} {$a} - {$b}";
};

$badgeEstadoPost = function (?string $estado): array {
  $estado = $estado ?? '';
  return match ($estado) {
    'enviada'      => ['secondary', 'Enviada'],
    'vista'        => ['info', 'Vista'],
    'preseleccion' => ['success', 'Preselección'],
    'rechazada'    => ['danger', 'Rechazada'],
    default        => ['light', '—'],
  };
};
?>

<div class="container feed-page">

  <div class="row g-3 mb-3" id="buscar">
    <div class="col-12">
      <div class="feed-hero">
        <div class="feed-hero-inner">
          <div class="d-flex justify-content-between align-items-start gap-3 flex-wrap">
            <div>
              <h1>Red Laboral</h1>
              <p>Feed unificado: vacantes + “para ti” por roles. Posts RRHH/oradores se agregan luego en el mismo feed.</p>
            </div>

            <div class="text-end">
              <div class="fw-semibold text-white">Tu sesión</div>
              <div style="font-size:12px;color:rgba(255,255,255,.75)">Usuario ID: <?= (int)$userId ?></div>
              <div style="font-size:12px;color:rgba(255,255,255,.75)">Tipo: <?= htmlspecialchars($tipo) ?></div>
            </div>
          </div>

          <div class="feed-search-wrap">
            <form method="get" action="<?= htmlspecialchars($bp . '/feed') ?>" class="feed-search">
              <input class="form-control"
                     name="q"
                     value="<?= htmlspecialchars($q) ?>"
                     placeholder="<?= $tipo==='empresa' ? 'Buscar en tus vacantes...' : 'Buscar vacantes (título/ubicación/descripcion)...' ?>"
                     aria-label="Buscar">

              <button class="btn btn-primary" type="submit">Buscar</button>

              <?php if ($q !== '' || !empty($catSelIds) || ($only === 1)): ?>
                <a class="btn btn-outline-light" href="<?= htmlspecialchars($bp . '/feed') ?>">Limpiar</a>
              <?php endif; ?>

              <?php
                foreach ($catSelIds as $cid) echo '<input type="hidden" name="cat[]" value="'.(int)$cid.'">';
                if ($only === 1) echo '<input type="hidden" name="only" value="1">';
              ?>
            </form>
          </div>

        </div>
      </div>
    </div>
  </div>

  <div class="row g-3">
    <!-- Left -->
    <div class="col-12 col-lg-3">
      <div class="app-card p-3">
        <div class="fw-semibold mb-2">Acciones rápidas</div>

        <?php if ($tipo === 'empresa'): ?>
          <div class="d-grid gap-2">
            <a class="btn btn-outline-primary btn-sm" href="<?= htmlspecialchars($bp . '/empresa/vacantes') ?>">Mis vacantes</a>
            <a class="btn btn-primary btn-sm" href="<?= htmlspecialchars($bp . '/empresa/vacantes/create') ?>">Nueva vacante</a>
          </div>
          <hr>
          <div class="small app-muted">Este feed muestra tus vacantes con contador de postulaciones.</div>
        <?php else: ?>
          <div class="d-grid gap-2">
            <a class="btn btn-outline-primary btn-sm" href="<?= htmlspecialchars($bp . '/postulante/perfil') ?>">Mi perfil</a>
            <a class="btn btn-outline-primary btn-sm" href="<?= htmlspecialchars($bp . '/postulante/postulaciones') ?>">Mis postulaciones</a>
          </div>

          <hr>

          <div class="fw-semibold mb-2">Roles (Para ti)</div>

          <?php if (empty($cats)): ?>
            <div class="small text-muted">No hay roles cargados (migración pendiente).</div>
          <?php else: ?>
            <form method="get" action="<?= htmlspecialchars($bp . '/feed') ?>" class="d-grid gap-2">
              <input type="hidden" name="q" value="<?= htmlspecialchars($q) ?>">

              <div class="border rounded-3 p-2" style="border-color:#e8eefc; max-height: 240px; overflow:auto;">
                <?php foreach ($cats as $c): ?>
                  <?php
                    $cid = (int)($c['id'] ?? 0);
                    if ($cid <= 0) continue;
                    $name = (string)($c['nombre'] ?? '');
                  ?>
                  <label class="d-flex gap-2 align-items-start mb-1">
                    <input class="form-check-input mt-1" type="checkbox" name="cat[]" value="<?= $cid ?>"
                           <?= isset($catSelMap[$cid]) ? 'checked' : '' ?>>
                    <span class="small"><?= htmlspecialchars($name) ?></span>
                  </label>
                <?php endforeach; ?>
              </div>

              <label class="d-flex gap-2 align-items-start mt-2">
                <input class="form-check-input mt-1" type="checkbox" name="only" value="1" <?= ($only === 1) ? 'checked' : '' ?>>
                <div>
                  <div class="small fw-semibold">Ver solo vacantes que hacen match</div>
                  <div class="small text-muted">Usa tus roles seleccionados para filtrar el feed.</div>
                </div>
              </label>

              <button class="btn btn-outline-primary btn-sm" type="submit">Aplicar filtro</button>
            </form>
          <?php endif; ?>
        <?php endif; ?>
      </div>
    </div>

    <!-- Center -->
    <div class="col-12 col-lg-6">
      <div class="feed-center-scroll">

        <div class="app-card p-3 mb-3">
          <div class="fw-semibold mb-2"><?= $tipo !== 'empresa' ? 'Vacantes recientes' : 'Tus vacantes' ?></div>
          <div class="small app-muted">
            <?= $tipo !== 'empresa' ? 'Reaccioná o compartí. Postular es texto y queda registrado.' : 'Panel operativo empresa.' ?>
          </div>
        </div>

        <?php if (empty($items)): ?>
          <div class="app-card p-3">
            <div class="text-muted">No hay resultados para mostrar.</div>
          </div>
        <?php else: ?>
          <?php foreach ($items as $it): ?>
            <?php $kind = (string)($it['kind'] ?? 'vacante'); ?>

            <?php if ($kind === 'rrhh_anuncio'): ?>
              <?= render_template('partials/rrhh_anuncio_card.php', [
                'it'   => $it,
                'tipo' => $tipo,
                'bp'   => $bp,
                'csrf' => $csrf,
              ]) ?>
            <?php else: ?>
              <?= render_template('partials/vacante_card.php', [
                'it'   => $it,
                'tipo' => $tipo,
                'bp'   => $bp,
                'csrf' => $csrf,
              ]) ?>
            <?php endif; ?>

          <?php endforeach; ?>
        <?php endif; ?>

      </div><!-- /feed-center-scroll -->
    </div>

    <!-- Right -->
    <div class="col-12 col-lg-3" id="alertas">

      <!-- Resumen -->
      <div class="app-card p-3 mb-3">
        <div class="fw-semibold mb-2">Resumen</div>

        <?php if ($tipo === 'empresa'): ?>
          <div class="small app-muted">Vacantes: <b><?= (int)($stats['vacantes_total'] ?? 0) ?></b></div>
          <div class="small app-muted">Publicadas: <b><?= (int)($stats['vacantes_publicadas'] ?? 0) ?></b></div>
          <div class="small app-muted">Postulaciones recibidas: <b><?= (int)($stats['postulaciones_total'] ?? 0) ?></b></div>
        <?php else: ?>
          <div class="small app-muted">Postulaciones: <b><?= (int)($stats['total'] ?? 0) ?></b></div>
          <div class="small app-muted">Enviadas: <b><?= (int)($stats['enviada'] ?? 0) ?></b></div>
          <div class="small app-muted">Vistas: <b><?= (int)($stats['vista'] ?? 0) ?></b></div>
          <div class="small app-muted">Preselección: <b><?= (int)($stats['preseleccion'] ?? 0) ?></b></div>
          <div class="small app-muted">Rechazadas: <b><?= (int)($stats['rechazada'] ?? 0) ?></b></div>
        <?php endif; // <-- ESTE endif faltaba y te rompía el archivo ?>
      </div>

      <?php
        // Ideales vienen del controller (app.php): $ideales
        $sidebarItems = $ideales ?? [];
        if (!is_array($sidebarItems)) $sidebarItems = [];
      ?>

      <?php if ($tipo !== 'empresa' && !empty($sidebarItems)): ?>
        <div class="card mb-3">
          <div class="card-body">
            <div class="fw-semibold mb-2">Ideales para ti</div>

            <!-- DESKTOP: 3 hacia abajo -->
            <div class="d-none d-md-block">
              <?php foreach (array_slice($sidebarItems, 0, 3) as $itMini): ?>
                <?= render_template('partials/vacante_card_mini.php', [
                  'it' => $itMini,
                  'bp' => $bp,
                ]) ?>
              <?php endforeach; ?>
            </div>

            <!-- MOBILE: 5 en carrusel infinito -->
            <div class="d-md-none">
              <div class="carousel-infinite">
                <div class="carousel-track">
                  <?php foreach (array_slice($sidebarItems, 0, 5) as $itMini): ?>
                    <div class="carousel-item">
                      <?= render_template('partials/vacante_card_mini.php', ['it'=>$itMini,'bp'=>$bp]) ?>
                    </div>
                  <?php endforeach; ?>

                  <!-- duplicado para loop infinito -->
                  <?php foreach (array_slice($sidebarItems, 0, 5) as $itMini): ?>
                    <div class="carousel-item">
                      <?= render_template('partials/vacante_card_mini.php', ['it'=>$itMini,'bp'=>$bp]) ?>
                    </div>
                  <?php endforeach; ?>
                </div>
              </div>
            </div>

          </div>
        </div>
      <?php endif; ?>

      <div class="app-card p-3">
        <div class="fw-semibold mb-2">Nota</div>
        <div class="small app-muted">
          Reacciones son privadas por usuario; solo ves conteos. “Mala experiencia” sirve como señal de riesgo sin exponer identidad.
        </div>
      </div>

    </div><!-- /Right -->

  </div><!-- /row -->
</div><!-- /container -->

<script>
(function(){
  const bp = <?= json_encode($bp) ?>;
  let csrf = <?= json_encode($csrf) ?>;

  function setActiveButtons(bar, myReaction){
    const buttons = bar.querySelectorAll('button[data-action="react"]');
    buttons.forEach(btn => {
      const r = btn.getAttribute('data-reaction');
      if (r === myReaction) {
        btn.classList.remove('btn-outline-secondary');
        btn.classList.add('btn-primary');
      } else {
        btn.classList.add('btn-outline-secondary');
        btn.classList.remove('btn-primary');
      }
    });
  }

  function updateCounts(bar, counts){
    for (const k in counts){
      const el = bar.querySelector('[data-count="'+k+'"]');
      if (el) el.textContent = String(counts[k] ?? 0);
    }
  }

  async function post(url, data){
    const form = new URLSearchParams();
    Object.keys(data).forEach(k => form.append(k, data[k]));
    const res = await fetch(url, {
      method: 'POST',
      headers: {'Content-Type':'application/x-www-form-urlencoded; charset=UTF-8'},
      body: form.toString(),
      credentials: 'same-origin'
    });

    const ct = res.headers.get('content-type') || '';
    if (!ct.includes('application/json')) {
      const txt = await res.text();
      throw new Error('Respuesta no JSON: ' + txt.slice(0, 180));
    }
    return res.json();
  }

  function getItemMeta(bar){
    // Nuevo estándar:
    // data-kind="vacante|rrhh_anuncio" data-item-id="123"
    const kind = (bar.getAttribute('data-kind') || '').trim();
    const itemId = (bar.getAttribute('data-item-id') || '').trim();

    if (kind && itemId) {
      return { kind, item_id: itemId };
    }

    // Compat viejo: solo vacante_id
    const vacanteId = (bar.getAttribute('data-vacante-id') || '').trim();
    if (vacanteId) {
      return { kind: 'vacante', item_id: vacanteId };
    }

    return { kind: '', item_id: '' };
  }

  document.addEventListener('click', async (ev) => {
    const btn = ev.target.closest('button[data-action]');
    if (!btn) return;

    const bar = btn.closest('[data-social-bar="1"]');
    if (!bar) return;

    const meta = getItemMeta(bar);
    if (!meta.kind || !meta.item_id) return;

    const action = btn.getAttribute('data-action');
    btn.disabled = true;

    try {
      if (action === 'react') {
        const reaction = btn.getAttribute('data-reaction');
        const out = await post(bp + '/feed/react', {
          _csrf: csrf,
          kind: meta.kind,
          item_id: meta.item_id,
          reaction: reaction
        });

        if (out && out.csrf) csrf = out.csrf;
        if (out && out.ok) {
          updateCounts(bar, out.counts || {});
          setActiveButtons(bar, out.my_reaction || '');
        }
      }

      if (action === 'share') {
        const out = await post(bp + '/feed/share', {
          _csrf: csrf,
          kind: meta.kind,
          item_id: meta.item_id
        });

        if (out && out.csrf) csrf = out.csrf;
        if (out && out.ok) {
          updateCounts(bar, out.counts || {});
        }
      }
    } catch (e) {
      console.error(e);
    } finally {
      btn.disabled = false;
    }
  });
})();
</script>

