<?php
/** @var array $row */
/** @var string|null $error */
/** @var string|null $ok */
/** @var string $csrf */
/** @var bool $canApply */

$bp = base_path();

/**
 * Helpers (compatibles con feed.php) para NO romper path de imágenes.
 */
if (!function_exists('feed_asset_url')) {
  function feed_asset_url(string $bp, ?string $path): string {
    $p = trim((string)$path);
    if ($p === '') return '';
    $p = str_replace('\\', '/', $p);

    if (preg_match('~^https?://~i', $p)) return $p;

    $p = str_replace('/public/public/', '/public/', $p);

    if (str_starts_with($p, '/public/upload/')) {
      $p = substr($p, 7);
    }
    if (str_starts_with($p, 'public/upload/')) {
      $p = '/' . substr($p, 7);
    }
    if (str_starts_with($p, 'upload/')) {
      $p = '/' . $p;
    }
    if (!str_starts_with($p, '/')) $p = '/' . $p;

    return $bp . $p;
  }
}

if (!function_exists('safe_rich_html')) {
  function safe_rich_html(string $html, string $bp): string {
    $html = trim($html);
    if ($html === '') return '';

    $html = preg_replace('~<(script|iframe|object|embed|style)\b[^>]*>.*?</\1>~is', '', $html) ?? '';

    $allowed = '<p><br><b><strong><i><em><u><h1><h2><h3><h4><h5><h6><ul><ol><li><a><blockquote><code><pre><hr><img>';
    $clean = strip_tags($html, $allowed);

    $clean = preg_replace('~\son\w+\s*=\s*(".*?"|\'.*?\'|[^\s>]+)~i', '', $clean) ?? '';
    $clean = preg_replace('~\sstyle\s*=\s*(".*?"|\'.*?\')~i', '', $clean) ?? '';

    $clean = preg_replace_callback('~<a\s+([^>]*?)href\s*=\s*("|\')(.*?)\2([^>]*)>~i', function($m){
      $before = $m[1] ?? '';
      $q = $m[2] ?? '"';
      $href = trim((string)($m[3] ?? ''));
      $after = $m[4] ?? '';

      if ($href === '') $href = '#';
      if (preg_match('~^(javascript:|data:)~i', $href)) $href = '#';
      if ($href !== '#' && !preg_match('~^(https?://|/)~i', $href)) $href = '#';

      return '<a ' . $before . 'href=' . $q . htmlspecialchars($href, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . $q
        . $after . ' target="_blank" rel="noopener noreferrer">';
    }, $clean);

    $clean = preg_replace_callback('~<img\b([^>]*?)>~i', function($m) use ($bp){
      $attrs = $m[1] ?? '';
      $src = '';
      $alt = '';
      $title = '';

      if (preg_match('~\bsrc\s*=\s*("|\')(.*?)\1~i', $attrs, $mm)) $src = trim((string)($mm[2] ?? ''));
      if (preg_match('~\balt\s*=\s*("|\')(.*?)\1~i', $attrs, $mm)) $alt = (string)($mm[2] ?? '');
      if (preg_match('~\btitle\s*=\s*("|\')(.*?)\1~i', $attrs, $mm)) $title = (string)($mm[2] ?? '');

      if ($src === '' || preg_match('~^(javascript:|data:)~i', $src)) return '';

      if (!preg_match('~^https?://~i', $src)) {
        $src = feed_asset_url($bp, $src);
      }

      $altEsc = htmlspecialchars($alt, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
      $titleEsc = htmlspecialchars($title, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
      $srcEsc = htmlspecialchars($src, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');

      return '<img src="' . $srcEsc . '" alt="' . $altEsc . '" title="' . $titleEsc . '" loading="lazy" style="max-width:100%;height:auto;border-radius:14px;border:1px solid #e8eefc;background:#fff;">';
    }, $clean);

    return $clean;
  }
}

/** Iconos inline (misma vibra que vacante_card) */
$icon = function(string $name): string {
  return match ($name) {
    'bolt' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M13 2L3 14h7l-1 8 10-12h-7l1-8z"/></svg>',
    'star' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M12 2l2.5 5 5.5.8-4 3.9.9 5.5L12 15.8 7.1 17.2l.9-5.5-4-3.9 5.5-.8L12 2z"/></svg>',
    'check' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M20 6L9 17l-5-5"/></svg>',
    'shield' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M12 2l8 4v6c0 5-3.3 9.3-8 10-4.7-.7-8-5-8-10V6l8-4z"/></svg>',
    'clock' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M12 8v5l3 2"/><path d="M12 22a10 10 0 1 0 0-20 10 10 0 0 0 0 20z"/></svg>',
    'pin' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M12 21s7-4.5 7-11a7 7 0 0 0-14 0c0 6.5 7 11 7 11z"/><path d="M12 10a2 2 0 1 0 0-4 2 2 0 0 0 0 4z"/></svg>',
    'layers' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M12 2l9 5-9 5-9-5 9-5z"/><path d="M3 12l9 5 9-5"/><path d="M3 17l9 5 9-5"/></svg>',
    'briefcase' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M10 6V5a2 2 0 0 1 2-2h0a2 2 0 0 1 2 2v1"/><path d="M4 7h16v12a2 2 0 0 1-2 2H6a2 2 0 0 1-2-2V7z"/></svg>',
    default => '',
  };
};

/** ---------- Datos base ---------- */
$mon  = (string)($row['moneda'] ?? 'CRC');
$smin = $row['salario_min'] ?? null;
$smax = $row['salario_max'] ?? null;

$sal = '—';
if ($smin !== null || $smax !== null) {
  $a = $smin !== null ? number_format((float)$smin, 2) : '—';
  $b = $smax !== null ? number_format((float)$smax, 2) : '—';
  $sal = "{$mon} {$a} - {$b}";
}

$vacanteId = (int)($row['id'] ?? 0);
$slug = (string)($row['slug'] ?? '');

$publicUrl = $bp . '/vacantes/ver?id=' . $vacanteId;
if ($slug !== '') $publicUrl = $bp . '/vacante?slug=' . rawurlencode($slug);

// Empresa
$empresaId = (int)($row['empresa_user_id'] ?? 0);
$empresaNombre = trim((string)($row['empresa_nombre'] ?? ''));
if ($empresaNombre === '') $empresaNombre = (string)($row['empresa_email'] ?? 'Empresa');

$empresaLogoUrl = feed_asset_url($bp, (string)($row['empresa_logo_path'] ?? ''));
$empresaVerificada = (int)($row['empresa_verificada'] ?? 0) === 1;

$empresaRating = (float)($row['empresa_rating'] ?? 0.0);
if ($empresaRating < 0) $empresaRating = 0.0;
if ($empresaRating > 5) $empresaRating = 5.0;

$isIdeal = (int)($row['is_ideal'] ?? 0) === 1;
$boostMax = (int)($row['boost_max'] ?? 0);

$catsVac = trim((string)($row['categorias_vacante'] ?? ''));
$catsMatch = trim((string)($row['categorias_match'] ?? ''));
$matchCount = (int)($row['match_count'] ?? 0);

$vacImgUrl = feed_asset_url($bp, (string)($row['imagen_path'] ?? ''));

$pubDtRaw = (string)($row['publicada_en'] ?? $row['creado_en'] ?? '');
$pubLabel = '—';
if ($pubDtRaw !== '') {
  try { $pubLabel = (new DateTime($pubDtRaw))->format('d/m/Y H:i'); } catch (Throwable $e) {}
}

// JSON-LD JobPosting (SEO)
$titulo = (string)($row['titulo'] ?? 'Vacante');
$desc   = (string)($row['descripcion'] ?? '');
$descPlain = trim(preg_replace('/\s+/', ' ', strip_tags($desc)));

$ubic = (string)($row['ubicacion'] ?? '');
$tipoEmpleo = (string)($row['tipo_empleo'] ?? '');
$modalidad  = (string)($row['modalidad'] ?? '');
$datePosted = (string)($row['publicada_en'] ?? '');
$datePostedIso = $datePosted !== '' ? substr($datePosted, 0, 10) : null;

$jsonLd = [
  '@context' => 'https://schema.org',
  '@type' => 'JobPosting',
  'title' => $titulo,
  'description' => $descPlain !== '' ? $descPlain : $titulo,
  'datePosted' => $datePostedIso,
  'employmentType' => $tipoEmpleo !== '' ? $tipoEmpleo : null,
  'hiringOrganization' => [
    '@type' => 'Organization',
    'name'  => $empresaNombre,
  ],
  'jobLocation' => $ubic !== '' ? [
    '@type' => 'Place',
    'address' => [
      '@type' => 'PostalAddress',
      'addressLocality' => $ubic,
      'addressCountry' => 'CR',
    ],
  ] : null,
  'applicantLocationRequirements' => $modalidad !== '' ? $modalidad : null,
  'identifier' => [
    '@type' => 'PropertyValue',
    'name'  => 'job.fasicat.com',
    'value' => (string)$vacanteId,
  ],
  'url' => $publicUrl,
];

$clean = function ($arr) use (&$clean) {
  if (!is_array($arr)) return $arr;
  foreach ($arr as $k => $v) {
    if (is_array($v)) {
      $arr[$k] = $clean($v);
      if ($arr[$k] === []) unset($arr[$k]);
    } elseif ($v === null || $v === '') {
      unset($arr[$k]);
    }
  }
  return $arr;
};
$jsonLd = $clean($jsonLd);

$descHtml = safe_rich_html((string)($row['descripcion'] ?? ''), $bp);

// Chips (mismo criterio que card)
$insignia = 'Empresa';
$insVariant = 'neutral';
$insIcon = 'shield';
if ($isIdeal) { $insignia = 'Patrocinada'; $insVariant = 'warn'; $insIcon = 'bolt'; }
elseif ($empresaRating >= 4.5) { $insignia = 'Top'; $insVariant = 'success'; $insIcon = 'star'; }

// UI tokens
$empresaUrl = $empresaId > 0 ? (url_path('empresas/ver') . '?id=' . $empresaId) : '';
$backUrl = url_path('vacantes');
?>

<link rel="canonical" href="<?= htmlspecialchars($publicUrl) ?>">
<script type="application/ld+json">
<?= json_encode($jsonLd, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) ?>
</script>

<style>
  .vac-ico svg{width:16px;height:16px;fill:none;stroke:currentColor;stroke-width:2;stroke-linecap:round;stroke-linejoin:round;}
  .vac-chip{display:inline-flex;align-items:center;gap:8px;padding:6px 10px;border-radius:999px;border:1px solid #e8eefc;background:#fff;font-size:12px;line-height:1;color:#0b1220;}
  .vac-chip--neutral{background:#f7f9ff;}
  .vac-chip--success{background:#eefaf3;border-color:#cfeedd;color:#0f6d3a;}
  .vac-chip--info{background:#eef4ff;border-color:#d8e6ff;color:#245bff;}
  .vac-chip--warn{background:#fff6df;border-color:#ffe6a3;color:#7a5b00;}
  .vac-meta{color:#5b6b86;}
  .vac-title{font-weight:800;font-size:1.25rem;line-height:1.15;}
  .vac-sub{color:#5b6b86;}
</style>

<div class="container py-4" style="max-width: 1080px;">

  <div class="d-flex align-items-start justify-content-between gap-3 mb-3 flex-wrap">
    <div class="d-flex gap-3 align-items-start">
      <?php if ($empresaLogoUrl !== ''): ?>
        <img src="<?= htmlspecialchars($empresaLogoUrl) ?>"
             alt="Logo empresa"
             style="width:56px;height:56px;object-fit:cover;border-radius:14px;border:1px solid #e8eefc;background:#fff"
             loading="lazy">
      <?php else: ?>
        <div style="width:56px;height:56px;border-radius:14px;border:1px solid #e8eefc;background:#f7f9ff"></div>
      <?php endif; ?>

      <div>
        <div class="d-flex flex-wrap align-items-center gap-2 mb-1">
          <span class="vac-title"><?= htmlspecialchars($titulo) ?></span>
        </div>

        <div class="small vac-sub d-flex flex-wrap gap-2 align-items-center">
          <span class="me-1">
            <span class="vac-ico" style="color:#5b6b86;"><?= $icon('pin') ?></span>
            <?= htmlspecialchars($ubic !== '' ? $ubic : '—') ?>
          </span>
          <span class="me-1">
            <span class="vac-ico" style="color:#5b6b86;"><?= $icon('layers') ?></span>
            <?= htmlspecialchars($modalidad !== '' ? $modalidad : '—') ?>
          </span>
          <span class="me-1">
            <span class="vac-ico" style="color:#5b6b86;"><?= $icon('briefcase') ?></span>
            <?= htmlspecialchars($tipoEmpleo !== '' ? $tipoEmpleo : '—') ?>
          </span>
          <span class="me-1">
            <span class="vac-ico" style="color:#5b6b86;"><?= $icon('clock') ?></span>
            <?= htmlspecialchars($pubLabel) ?>
          </span>
        </div>

        <div class="mt-2 d-flex flex-wrap gap-2 align-items-center">
          <span class="vac-chip vac-chip--<?= htmlspecialchars($insVariant) ?>">
            <span class="vac-ico"><?= $icon($insIcon) ?></span>
            <span><?= htmlspecialchars($insignia) ?></span>
          </span>

          <span class="vac-chip vac-chip--neutral" title="Puntuación de empresa">
            <span class="vac-ico"><?= $icon('star') ?></span>
            <span><?= htmlspecialchars(number_format($empresaRating, 1)) ?>/5</span>
          </span>

          <?php if ($empresaVerificada): ?>
            <span class="vac-chip vac-chip--info">
              <span class="vac-ico"><?= $icon('check') ?></span>
              <span>Verificada</span>
            </span>
          <?php else: ?>
            <span class="vac-chip vac-chip--neutral">
              <span class="vac-ico"><?= $icon('shield') ?></span>
              <span>Básica</span>
            </span>
          <?php endif; ?>

          <?php if ($matchCount > 0): ?>
            <span class="vac-chip vac-chip--info" title="Categorías que hacen match contigo">
              <span class="vac-ico"><?= $icon('check') ?></span>
              <span>Match: <?= (int)$matchCount ?></span>
            </span>
          <?php endif; ?>

          <?php if ($empresaNombre !== ''): ?>
            <?php if ($empresaUrl !== ''): ?>
              <a class="vac-chip vac-chip--neutral text-decoration-none" href="<?= htmlspecialchars($empresaUrl) ?>">
                <span><?= htmlspecialchars($empresaNombre) ?></span>
              </a>
            <?php else: ?>
              <span class="vac-chip vac-chip--neutral"><?= htmlspecialchars($empresaNombre) ?></span>
            <?php endif; ?>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <a class="btn btn-outline-secondary btn-sm" href="<?= htmlspecialchars($backUrl) ?>">Volver</a>
  </div>

  <?php if (!empty($error)): ?>
    <div class="alert alert-danger"><?= htmlspecialchars((string)$error) ?></div>
  <?php endif; ?>

  <?php if (!empty($ok)): ?>
    <div class="alert alert-success"><?= htmlspecialchars((string)$ok) ?></div>
  <?php endif; ?>

  <div class="row g-3">

    <!-- MAIN -->
    <div class="col-12 col-lg-8">

      <?php if ($vacImgUrl !== ''): ?>
        <div class="app-card p-2 mb-3">
          <div style="width:100%; aspect-ratio: 1080 / 1180; overflow:hidden; border-radius:16px; border:1px solid #e8eefc; background:#fff;">
            <img src="<?= htmlspecialchars($vacImgUrl) ?>"
                 alt="Imagen vacante"
                 loading="lazy"
                 style="width:100%;height:100%;object-fit:cover;">
          </div>
        </div>
      <?php endif; ?>

      <div class="app-card p-4">
        <div class="d-flex flex-wrap gap-2 justify-content-between align-items-center mb-3">
          <div class="fw-semibold">Descripción</div>

          <?php if ($catsVac !== '' || $catsMatch !== ''): ?>
            <div class="d-flex flex-wrap gap-2">
              <?php if ($catsVac !== ''): ?>
                <span class="vac-chip vac-chip--neutral" title="Categorías de la vacante">
                  <span><?= htmlspecialchars($catsVac) ?></span>
                </span>
              <?php endif; ?>

              <?php if ($catsMatch !== ''): ?>
                <span class="vac-chip vac-chip--success" title="Categorías que hacen match contigo">
                  <span class="vac-ico"><?= $icon('check') ?></span>
                  <span>Match: <?= htmlspecialchars($catsMatch) ?></span>
                </span>
              <?php endif; ?>
            </div>
          <?php endif; ?>
        </div>

        <?php if ($descHtml !== ''): ?>
          <div class="small" style="white-space:normal; line-height:1.5;">
            <?= $descHtml ?>
          </div>
        <?php else: ?>
          <div class="text-muted">Sin descripción.</div>
        <?php endif; ?>
      </div>

    </div>

    <!-- SIDE -->
    <div class="col-12 col-lg-4">

      <div class="app-card p-4 mb-3">
        <div class="small text-muted mb-1">Salario</div>
        <div class="fw-semibold mb-3"><?= htmlspecialchars($sal) ?></div>

        <div class="small text-muted mb-1">Compartir</div>
        <div class="d-flex gap-2 flex-wrap mb-3">
          <input class="form-control form-control-sm" value="<?= htmlspecialchars($publicUrl) ?>" readonly onclick="this.select()">
          <a class="btn btn-outline-secondary btn-sm" target="_blank" rel="noopener noreferrer"
             href="<?= htmlspecialchars('https://wa.me/?text=' . rawurlencode($publicUrl)) ?>">WhatsApp</a>
        </div>

        <hr>

        <?php if ($canApply): ?>
          <div class="fw-semibold mb-2">Postular</div>

          <form method="post" action="<?= htmlspecialchars(url_path('vacantes/postular')) ?>" class="d-grid gap-2">
            <input type="hidden" name="_csrf" value="<?= htmlspecialchars((string)$csrf) ?>">
            <input type="hidden" name="vacante_id" value="<?= $vacanteId ?>">

            <label class="form-label mb-1">Mensaje (opcional)</label>
            <textarea class="form-control" name="mensaje" rows="4" placeholder="Cuento brevemente por qué aplico..."></textarea>

            <button class="btn btn-primary" type="submit">Postular</button>
            <div class="small text-muted">Queda registrada tu postulación.</div>
          </form>
        <?php else: ?>
          <div class="alert alert-light border mb-0">
            Para postular debes iniciar sesión como <b>postulante</b>.
          </div>
        <?php endif; ?>
      </div>

      <div class="app-card p-4">
        <div class="fw-semibold mb-2">Señales de confianza</div>
        <ul class="list-unstyled small mb-0">
          <li class="d-flex justify-content-between py-1">
            <span>Empresa verificada</span>
            <span class="badge <?= $empresaVerificada ? 'text-bg-success' : 'text-bg-light' ?>" style="<?= $empresaVerificada ? '' : 'border:1px solid #e8eefc;' ?>">
              <?= $empresaVerificada ? 'Sí' : 'No' ?>
            </span>
          </li>
          <li class="d-flex justify-content-between py-1">
            <span>Puntuación</span>
            <span class="badge text-bg-dark">★ <?= number_format($empresaRating, 1) ?>/5</span>
          </li>
          <li class="d-flex justify-content-between py-1">
            <span>Match</span>
            <span class="badge <?= ($matchCount > 0) ? 'text-bg-primary' : 'text-bg-light' ?>" style="<?= ($matchCount > 0) ? '' : 'border:1px solid #e8eefc;' ?>">
              <?= ($matchCount > 0) ? ('Sí (' . (int)$matchCount . ')') : 'No' ?>
            </span>
          </li>
          <?php if ($isIdeal || $boostMax > 0): ?>
            <li class="d-flex justify-content-between py-1">
              <span>Promoción</span>
              <span class="badge text-bg-warning text-dark">
                <?= $isIdeal ? 'Patrocinada' : 'Boost' ?>
              </span>
            </li>
          <?php endif; ?>
        </ul>
      </div>

    </div>
  </div>
</div>
