<?php declare(strict_types=1); ?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title><?= htmlspecialchars($pageTitle ?? 'Admin', ENT_QUOTES, 'UTF-8') ?></title>

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">

  <style>
    :root { --sbw: 280px; }
    .admin-shell { min-height: 100vh; }
    .sidebar { width: var(--sbw); }
    .sidebar a.nav-link { border-radius: .6rem; color: #1f2937; }
    .sidebar a.nav-link:hover { background: #f3f4f6; }
    .sidebar a.nav-link.active { background: #0d6efd; color: #fff; }
    .sidebar .section-title {
      font-size: .75rem;
      text-transform: uppercase;
      letter-spacing: .04em;
      color: #6b7280;
      margin: .75rem 0 .35rem;
    }
    .content-wrap { width: calc(100% - var(--sbw)); }
    @media (max-width: 991.98px) {
      .sidebar { width: 100%; position: sticky; top: 0; z-index: 1020; }
      .content-wrap { width: 100%; }
    }
  </style>
</head>

<body class="bg-light">
<?php
  /**
   * Layout Admin - Versión compatible:
   * 1) Preferencia: $menu (dinámico) pasado desde backend (AdminMenuBuilder::buildForUser()).
   * 2) Fallback: Menú hardcodeado con $can() + $hasModule() si $menu no existe.
   */

  // Usuario (si existe helper)
  $user = null;
  if (function_exists('auth_user')) {
      $user = auth_user();
  }

  // base_path (si existe)
  $bp = function_exists('base_path') ? (string)base_path() : '';
  $bpNorm = rtrim($bp, '/');

  // Path actual para "active"
  $uri  = $_SERVER['REQUEST_URI'] ?? '';
  $path = parse_url($uri, PHP_URL_PATH) ?: '';

  // Helpers defensivos para permisos/módulos (fallback)
  $can = function (string $perm): bool {
      if (function_exists('can')) return (bool)can($perm);
      if (function_exists('has_permission')) return (bool)has_permission($perm);
      if (function_exists('user_can')) return (bool)user_can($perm);
      return true; // no romper UI si helper no existe
  };

  $hasModule = function (string $moduleCode): bool {
      if (function_exists('has_module')) return (bool)has_module($moduleCode);
      if (function_exists('user_has_module')) return (bool)user_has_module($moduleCode);
      if (function_exists('module_enabled')) return (bool)module_enabled($moduleCode);
      return true;
  };

  // URL builder seguro
  $href = function (string $route) use ($bpNorm): string {
      $r = '/' . ltrim($route, '/');
      return htmlspecialchars($bpNorm . $r, ENT_QUOTES, 'UTF-8');
  };

  // Active robusto: compara el path actual vs prefix con base_path aplicado
  $isActive = function (string $routePrefix) use ($path, $bpNorm): bool {
      $pref = '/' . ltrim($routePrefix, '/');
      $full = $bpNorm . $pref;
      return $full !== '' && str_starts_with($path, $full);
  };

  // Nombres usuario
  $uName  = is_array($user) ? (string)($user['name'] ?? $user['nombre'] ?? $user['email'] ?? 'Usuario') : 'Usuario';
  $uEmail = is_array($user) ? (string)($user['email'] ?? '') : '';

  // Menú dinámico si el controlador lo pasa
  // Esperado: $menu = [ ['label'=>..., 'icon'=>..., 'route'=>..., ...], ... ]
  $menuItems = [];
  if (isset($menu) && is_array($menu)) {
      foreach ($menu as $it) {
          if (!is_array($it)) continue;
          if (empty($it['label']) || empty($it['route'])) continue;
          $menuItems[] = $it;
      }
  }

  // Fallback menu si no hay $menu
  if (empty($menuItems)) {
      $routeDashboard = '/admin';
      $routeModules   = '/admin/modules';
      $routePlans     = '/admin/plans';
      $routeRoles     = '/admin/roles';
      $routeUsers     = '/admin/users';

      $dashboardActive = $isActive($routeDashboard)
          && !$isActive($routeModules)
          && !$isActive($routePlans)
          && !$isActive($routeRoles)
          && !$isActive($routeUsers);

      $menuItems[] = [
          'section' => 'General',
          'items' => [
              [
                  'label' => 'Dashboard',
                  'icon'  => 'bi-speedometer2',
                  'route' => $routeDashboard,
                  'active' => $dashboardActive,
              ],
          ]
      ];

      $adminGroup = [];

      if ($can('admin.modules.manage') && $hasModule('admin_panel')) {
          $adminGroup[] = [
              'label' => 'Módulos',
              'icon'  => 'bi-grid-1x2',
              'route' => $routeModules,
              'active' => $isActive($routeModules),
          ];
      }

      if ($can('admin.plans.manage') && $hasModule('admin_panel')) {
          $adminGroup[] = [
              'label' => 'Planes',
              'icon'  => 'bi-box',
              'route' => $routePlans,
              'active' => $isActive($routePlans),
          ];
      }

      if ($can('admin.rbac.manage') && $hasModule('admin_panel')) {
          $adminGroup[] = [
              'label' => 'Roles',
              'icon'  => 'bi-shield-lock',
              'route' => $routeRoles,
              'active' => $isActive($routeRoles),
          ];
          $adminGroup[] = [
              'label' => 'Usuarios',
              'icon'  => 'bi-people',
              'route' => $routeUsers,
              'active' => $isActive($routeUsers),
          ];
      }

      $menuItems[] = [
          'section' => 'Administración',
          'items' => $adminGroup,
      ];
  } else {
      /**
       * Si el backend pasa $menu plano, lo agrupamos en 2 secciones lógicas (visual):
       * - General: Dashboard si existe
       * - Administración: resto
       *
       * NOTA: No alteramos permisos aquí; se asume filtrado backend.
       */
      $flat = $menuItems;
      $general = [];
      $admin   = [];

      foreach ($flat as $it) {
          $route = (string)$it['route'];
          $icon  = (string)($it['icon'] ?? 'bi-circle');
          $label = (string)$it['label'];

          $entry = [
              'label' => $label,
              'icon'  => $icon,
              'route' => $route,
              'active'=> $isActive($route),
          ];

          if ($route === '/admin' || $route === 'admin' || $route === '/admin/') {
              $general[] = $entry;
          } else {
              $admin[] = $entry;
          }
      }

      $menuItems = [];
      if (!empty($general)) {
          $menuItems[] = ['section' => 'General', 'items' => $general];
      }
      if (!empty($admin)) {
          $menuItems[] = ['section' => 'Administración', 'items' => $admin];
      }
  }
?>
<div class="d-flex flex-column flex-lg-row admin-shell">

  <!-- SIDEBAR -->
  <aside class="sidebar border-end bg-white">
    <div class="p-3 border-bottom d-flex align-items-center justify-content-between">
      <div class="fw-semibold d-flex align-items-center gap-2">
        <i class="bi bi-shield-lock"></i>
        <span>Admin</span>
      </div>

      <button class="btn btn-sm btn-outline-secondary d-lg-none" type="button"
              data-bs-toggle="collapse" data-bs-target="#adminSidebarNav" aria-expanded="false">
        <i class="bi bi-list"></i>
      </button>
    </div>

    <div class="p-3 border-bottom">
      <div class="d-flex align-items-center gap-2">
        <div class="rounded-circle bg-primary text-white d-flex align-items-center justify-content-center"
             style="width:38px;height:38px;">
          <i class="bi bi-person"></i>
        </div>
        <div class="min-w-0">
          <div class="fw-semibold text-truncate"><?= htmlspecialchars($uName, ENT_QUOTES, 'UTF-8') ?></div>
          <?php if ($uEmail !== ''): ?>
            <div class="small text-muted text-truncate"><?= htmlspecialchars($uEmail, ENT_QUOTES, 'UTF-8') ?></div>
          <?php else: ?>
            <div class="small text-muted">Panel administrativo</div>
          <?php endif; ?>
        </div>
      </div>
    </div>

    <div class="collapse d-lg-block" id="adminSidebarNav">
      <nav class="p-3">

        <?php foreach ($menuItems as $group): ?>
          <?php
            $section = (string)($group['section'] ?? '');
            $items   = (array)($group['items'] ?? []);
          ?>
          <?php if ($section !== ''): ?>
            <div class="section-title"><?= htmlspecialchars($section, ENT_QUOTES, 'UTF-8') ?></div>
          <?php endif; ?>

          <?php foreach ($items as $item): ?>
            <?php
              $label  = (string)($item['label'] ?? '');
              $icon   = (string)($item['icon'] ?? 'bi-circle');
              $route  = (string)($item['route'] ?? '/admin');
              $active = (bool)($item['active'] ?? $isActive($route));
            ?>
            <a class="nav-link d-flex align-items-center gap-2 <?= $active ? 'active' : '' ?>"
               href="<?= $href($route) ?>">
              <i class="bi <?= htmlspecialchars($icon, ENT_QUOTES, 'UTF-8') ?>"></i>
              <span><?= htmlspecialchars($label, ENT_QUOTES, 'UTF-8') ?></span>
            </a>
          <?php endforeach; ?>

        <?php endforeach; ?>

        <hr class="my-3">

        <a class="nav-link d-flex align-items-center gap-2 text-danger"
           href="<?= $href('/logout') ?>">
          <i class="bi bi-box-arrow-right"></i>
          <span>Salir</span>
        </a>
      </nav>
    </div>
  </aside>

  <!-- CONTENT -->
  <main class="content-wrap flex-fill">
    <div class="p-3 border-bottom bg-white d-flex justify-content-between align-items-center">
      <div class="fw-semibold"><?= htmlspecialchars($pageTitle ?? 'Admin', ENT_QUOTES, 'UTF-8') ?></div>
      <div class="small text-secondary">Panel administrativo</div>
    </div>

    <div class="p-3">
      <?= $content ?? '' ?>
    </div>
  </main>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
